#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

import usrp_prims
import weakref
from usrp_fpga_regs import *

class db_base(object):
    """
    Abstract base class for all daughterboards.

    This defines the required operations and interfaces for all d'boards.
    """
    def __init__(self, usrp, which):
        """
        Initialize daughterboard interface.

        @param usrp: instance of usrp
        @param which: which daughterboard side: A = 0, B = 1
        @type which: int
        """

        if not (which in (0, 1)):
            raise ValueError, "Invalid value of which: %s" % (which,)

        self._u = weakref.proxy(usrp)

        self._which = which
        if hasattr(self._u, 'tx_freq'):   # is this a tx or rx daughterboard?
            self._tx = True
            self._slot = which * 2
        else:
            self._tx = False
            self._slot = which * 2 + 1

    def dbid(self):
        return self._u.daughterboard_id(self._which)

    def name(self):
        return usrp_prims.usrp_dbid_to_string(self.dbid())

    def side_and_name(self):
        return "AB"[self._which] + ': ' + self.name()

    # ------------------------------------------------------------------------
    # Automatic Transmit/Receive switching
    #
    # If automatic transmit/receive (ATR) switching is enabled in the
    # FR_ATR_CTL register, the presence or absence of data in the FPGA
    # transmit fifo selects between two sets of values for each of the 4
    # banks of daughterboard i/o pins.
    #
    # Each daughterboard slot has 3 16-bit registers associated with it:
    #   FR_ATR_MASK_*, FR_ATR_TXVAL_* and FR_ATR_RXVAL_*
    #
    # FR_ATR_MASK_{0,1,2,3}: 
    #
    #   These registers determine which of the daugherboard i/o pins are
    #   affected by ATR switching.  If a bit in the mask is set, the
    #   corresponding i/o bit is controlled by ATR, else it's output
    #   value comes from the normal i/o pin output register:
    #   FR_IO_{0,1,2,3}.
    #
    # FR_ATR_TXVAL_{0,1,2,3}:
    # FR_ATR_RXVAL_{0,1,2,3}:
    #
    #   If the Tx fifo contains data, then the bits from TXVAL that are
    #   selected by MASK are output.  Otherwise, the bits from RXVAL that
    #   are selected by MASK are output.

    def set_auto_tr(self,on):
        """
        Enable automatic Transmit/Receive switching (ATR).
        """
        if self._tx:
            if on:
                self._u._write_fpga_reg(FR_ATR_CTL, bmFR_ATR_CTL_EN)
            else:
                self._u._write_fpga_reg(FR_ATR_CTL, 0)
        else:
            raise AttributeError   # only allow set_auto_tr on Tx side

    def set_atr_mask(self, v):
        """
        Set Auto T/R mask.
        """
        return self._u._write_fpga_reg(FR_ATR_MASK_0 + 3 * self._slot, v)

    def set_atr_txval(self, v):
        """
        Set Auto T/R register value to be used when transmitting.
        """
        return self._u._write_fpga_reg(FR_ATR_TXVAL_0 + 3 * self._slot, v)

    def set_atr_rxval(self, v):
        """
        Set Auto T/R register value to be used when receiving.
        """
        return self._u._write_fpga_reg(FR_ATR_RXVAL_0 + 3 * self._slot, v)

    # derived classes should override the following methods

    def freq_range(self):
        """
        Return range of frequencies in Hz that can be tuned by this d'board.

        @returns (min_freq, max_freq, step_size)
        @rtype tuple
        """
        raise NotImplementedError

    def set_freq(self, target_freq):
        """
        Set the frequency.

        @param freq:  target RF frequency in Hz
        @type freq:   float

        @returns (ok, actual_baseband_freq) where:
           ok is True or False and indicates success or failure,
           actual_baseband_freq is the RF frequency that corresponds to DC in the IF.
        """
        raise NotImplementedError

    def gain_range(self):
        """
        Return range of gain that can be set by this d'board.

        @returns (min_gain, max_gain, step_size)
        Where gains are expressed in decibels (your mileage may vary)
        """
        raise NotImplementedError

    def set_gain(self, gain):
        """
        Set the gain.

        @param gain:  gain in decibels
        @returns True/False
        """
        raise NotImplementedError

    def is_quadrature(self):
        """
        Return True if this daughterboard does quadrature up or down conversion.
        That is, return True if this board requires both I & Q analog channels.

        This bit of info is useful when setting up the USRP Rx mux register.
        """
        raise NotImplementedError

    def i_and_q_swapped(self):
        """
        Return True if this is a quadrature device and ADC 0 is Q.
        """
        return False

    def set_enable(self, on):
        """
        For tx daughterboards, this controls the transmitter enable.
        """
        pass
    
