########################################################################
# housekeeping
########################################################################

use v6.d;

unit module ProcStats:ver<0.2.1>:auth<CPAN:lembark>;

########################################################################
# package variables & constants
########################################################################

# several of the posix fields are not avalable in Linux.
# fix for now is ignoring those -- though a beter fix is
# having separte modules for platforms that might support
# more of them.

constant FIELDS     =
<
    maxrss  ixrss   idrss   isrss
    minflt  majflt  nswap
    inblock oublock msgsnd
    msgrcv  nsignals nvcsw   nivcsw
>;
constant IGNORE     =
<
    ixrss idrss isrss
    nswap
    msgsnd msgrcv nsignals
>;

# wallclock and user time always increase.
# only include them if the other stats are
# non-zero.

constant COMPARE    =
<
    stime
    maxrss
    minflt majflt  
    inblock oublock
>;
constant HEADER     =
<
    wtime utime 
>;

constant WIDTH  = FIELDS>>.chars.max;
constant MICRO  = 1 / 1_000_000;

########################################################################
# subroutines
########################################################################

########################################################################
# in case someone wants to extract their own values use ":Extract".
# roll-your-own gormatting uses ":Diff".
# other than that :Print or :DEFAULT gets the diff-and-format.

sub extract-stats
(
    --> Hash
)
is export( :Extract )
{
    # grab wtime on the way in to avoid skew
    # computing differences.

    my $wtime       = now.Num;

    use nqp;
    nqp::getrusage( my int @raw );

    my ( $user_s, $user_us, $syst_s, $syst_us )
    = splice @raw, 0, 4;

    # times are reported as seconds + microseconds, converting
    # them to rationals here works better in Perl6.

    my $utime   = ( $user_s + $user_us * MICRO  ).round( MICRO );
    my $stime   = ( $syst_s + $syst_us * MICRO  ).round( MICRO );

    # faster to generat a hash and delete the items via slice
    # than zip only the ones that matter.

    my %rusage  = FIELDS Z=> @raw;
    %rusage{ IGNORE } :delete;

    % = 
    (
          |%rusage
        , :$wtime
        , :$utime
        , :$stime
        , sample => $++
    )
}

sub diff-stats
(
      %sample
    , Bool() :$final = False  
    , Bool() :$first = $final 
    , Bool() :$force = $final
    --> Hash
)
is export( :Extract )
{
    state %init;
    state %last;

    %init
    or do
    {
        %init = %last = %sample;

        return % = ( |HEADER, |COMPARE ) Z=> ( 0 xx HEADER+COMPARE )
    };

    # note that %prior here is a copy,
    # allowing updating %last
    # w/o getting zero for everything.

    my %prior
    = $first
    ?? %init
    !! %last
    ;

    # note '# Sample: ', %sample.perl;
    # note '# Prior:  ', %prior.perl;

    %last   = %sample;

    # i.e, if the values compared are different then return
    # those along with the header fields. otherwise return 
    # nothing.

    my @keys
    =  $force 
    ?? COMPARE
    !! COMPARE.grep( { %sample{$_} != %prior{$_} } )
    ;

    ? @keys
    or return %;

    # note '# Keys:   ', @keys.perl;

    % = ( |HEADER, |@keys ) Z=> %prior{ |HEADER, |@keys }
}

sub print-stats
(
        Bool()      :$final = False  
    ,   Stringy()   :$label = ''
    ,   *%args
    --> Bool
)
is export( :DEFAULT :Print )
{
    my %sample  = extract-stats;
    my %diff    = diff-stats %sample, :$final, |%args
    or return False;

    # note '# Diff:   ', %diff.perl;

    # if we get this far then there is something to print.
    # write-stat can live anywhere, write-diff is simpler
    # w/ access %diff so it lives here.

    sub write-stat ( Pair $p --> Nil )
    {
        note
        sprintf '%-*s : %s', WIDTH, $p.key, $p.value
    }

    sub write-diff( Pair $p --> Nil )
    {
        my $k   = $p.key;
        my $v   = ( %sample{ $k } - $p.value ).round( MICRO );

        write-stat ( $k => '+' ~ $v )
    }

    note '---';
    write-stat ( :$label                    ) if ? $label;
    write-stat ( Final  => True             ) if ? $final;
    write-stat ( output => $++              );
    write-stat ( sample => %sample<sample>  );

    once
    {
        # avoid leading '+' for the first pass to
        # show that these are baseline numbers.

        for |HEADER, |COMPARE -> $stat 
        {
            write-stat ( $stat => %sample{ $stat }  )
        }

        return True
    }

    for %diff.sort -> $stat
    {
        write-diff $stat
    }

    # caller knows sample was output.

    True;
}

########################################################################
# end
########################################################################
