proc refresh_grouplist {} {
    global GroupTable NumGroups GroupList GroupName2Index
    global VisibleGroups UnreadArticles ViewUnread

    set GroupList ""
    .metainfo.listbox delete 0 end 
    set UnreadArticles 0
    set VisibleGroups 0
    if {[info exists GroupName2Index]} {
	unset GroupName2Index
    }
    for {set i 0} {$i < $NumGroups} {incr i} {
	if {$GroupTable($i,unread) > 0} {
	    set line [format "%4.4s %s" $GroupTable($i,unread) \
		      $GroupTable($i,name)]
	    incr VisibleGroups
	    incr UnreadArticles $GroupTable($i,unread)
	} else {
	    if {$ViewUnread} {
		set line [format "%4.4s %s" $GroupTable($i,unread) \
			  $GroupTable($i,name)]
	    }
	}
	if {$ViewUnread || $GroupTable($i,unread) > 0} {
	    .metainfo.listbox insert end $line
	}
	lappend GroupList $i
	set GroupName2Index($GroupTable($i,name)) $i
    }
    write_status
}


proc refresh_articlelist {} {
    global ArticleTable NumArticles ArticleList CurrentUnread CurrentArticle

    .metainfo.listbox delete 0 end
    set CurrentUnread 0
    set ArticleList [lisort $ArticleList]
    foreach article $ArticleList {
	if {$ArticleTable($article,status) == "u"} {
	    incr CurrentUnread
	}
	.metainfo.listbox insert end [format_article_line $article]
    }
    select_index [lsearch $ArticleList $CurrentArticle]
    write_status
}

proc update_articlelist {number} {
    global ArticleTable ArticleList
    
    set index [lsearch $ArticleList $number]
    .metainfo.listbox delete $index
    .metainfo.listbox insert $index [format_article_line $number]
}

    

proc format_article_line {number} {
    global ArticleTable
    
    if {![info exists ArticleTable($number,subject)]} {
	# this is a cancelled article.
	set ArticleTable($number,subject)  "<<Cancelled>>"
	set ArticleTable($number,from) ""
	set ArticleTable($number,name) ""
	set ArticleTable($number,email) ""
	set ArticleTable($number,lines) ""
	set ArticleTable($number,status) "c"
    }
    set string "([string range $ArticleTable($number,name) 0 19])"
    if {$ArticleTable($number,status) != "u"} {
	set status $ArticleTable($number,status)
    } else {
	set status " "
    }
    return [format "%s  %d  %-40.40s %s" $status $number $ArticleTable($number,subject) $string]
}
				  


proc get_current_selection {n} {
    set i [.metainfo.listbox curselection]
    set count 0
    set ret {}
    foreach item $i {
	set add [.metainfo.listbox get $item]
	lappend ret $add
	incr count
	if {$count == $n} {
	    return $ret
	}
    }
}


proc clear_metainfo {} {	
    .metainfo.listbox delete 0 end
}

proc clear_text {} {
    .info.text tag delete Followup
    .info.text tag delete Highlight
    .info.text delete 0.0 end
}

proc set_text {text} {
    global Resources

    clear_text 
    set linenum 1
    set foundSubject 0
    .info.text insert end $text
    set line [.info.text get $linenum.0 "$linenum.0 lineend"]
    set empty_count 0
    while {$empty_count < 30} {
	set j [catch {set k [regexp "$Resources(followupExpr)" $line junk]}]
	if {!$j} { 
	    if {$k} {
		.info.text tag add Followup $linenum.0 $linenum.end
	    }
	}
	set j [catch {set k [regexp -indices "$Resources(highlightExpr)" "$line" ind]}]
	if {!$j} {
	    if {$k} {
		.info.text tag add Highlight $linenum.0 $linenum.end
	    }
	}
	if {!$foundSubject} {
	    if {[string first "Subject:" $line] != -1} {
		.info.text delete $linenum.0 "[expr $linenum+1].0"
		.info.text insert 1.0 "$line\n"
		.info.text tag add Subject 1.0 "1.0 lineend"
		set foundSubject 1
	    }
	}
	incr linenum
	set line [.info.text get $linenum.0 "$linenum.0 lineend"]
	if {![string compare $line ""]} {
	    incr empty_count
	} else {
	    set empty_count 0
	}
    }
    if {$Resources(followupFont) != ""} {
	.info.text tag configure Followup -font "$Resources(followupFont)"
    }
    if {$Resources(boldFont) != ""} {
	.info.text tag configure Subject -font "$Resources(boldFont)"
    }
    if {$Resources(highlightForeground) != ""} {
	.info.text tag configure Subject \
	    -foreground "$Resources(highlightForeground)"
	.info.text tag configure Highlight \
	    -foreground "$Resources(highlightForeground)"
    }
    if {$Resources(highlightBackground) != ""} {
	.info.text tag configure Subject \
	    -background "$Resources(highlightBackground)"
	.info.text tag configure Highlight \
	    -background "$Resources(highlightBackground)"
    }
    if {$Resources(highlightRelief) != ""} {
	.info.text tag configure Subject -relief $Resources(highlightRelief)
    }
    if {$Resources(highlightBorderWidth) != ""} {
	.info.text tag configure Subject \
	    -borderwidth $Resources(highlightBorderWidth)
    }
}

proc count_range {first last range} {
    set list [split $range ","]
    set num 0
    foreach item $list {
	if {[string first "-" $item] != -1} {
	    set l [split $item "-"]
	    set lower [lindex $l 0]
	    set upper [lindex $l 1]
	    if {$lower < $first} {set lower $first}
	    if {$upper > $last} {set upper $last}
	    set incnum [expr [expr $upper-$lower]+1]
	    if {$incnum > 0} {
		incr num $incnum
	    }
	} else {
	    incr num 
	}
    }
    return $num
}


proc group_index {name} {
    global GroupTable NumGroups

    for {set i 0} {$i < $NumGroups} {incr i} {
	if {$GroupTable($i,name) == $name} {
	    return $i
	}
    }
    return -1
}


proc group_name {index} {
    global ViewUnread GroupTable NumGroups 

    if {$ViewUnread} {
	return $GroupTable($index,name)
    } else {
	set count -1
	for {set i 0} {$count != $index} {incr i} {
	    if {$GroupTable($i,unread) > 0} {
		incr count
	    }
	}
	set s $GroupTable([expr $i-1],name)
	return $s
    }
}

proc construct_unread_list {} {
    global ArticleTable NumArticles ArticleList

    set num 0
    set ret ""
    foreach item $ArticleList {
	if {$ArticleTable($item,status) == "u"} {
	    incr num
	    lappend ret $item
	}
    }
    return $ret
}

proc compute_new_range {} {
    global ArticleTable NumArticles ArticleList 

    set ret ""
    set first [lindex $ArticleList [expr $NumArticles-1]]
    set last  [lindex $ArticleList 0]
    set list [construct_unread_list]
    if {$list == ""} {
	return ""
    }
    set last 0
    set i 1
    set upper -1
    while {$i < [llength $list]} {
	while {[lindex $list $i] == [expr [lindex $list [expr $i-1]]+1]} {
	    incr i
	}
	set lower [lindex $list $last]
	set upper [lindex $list [expr $i-1]]
	if {$lower == $upper} {
	    append ret "$lower,"
	} else {
	    append ret "$lower-$upper,"
	}
	set last $i
	incr i
    }
    if {[info exists upper]} {
	if {[lindex $list [expr [llength $list]-1]] != $upper} {
	    # we didn't get the last one, tack it on the end.
	    append ret "[lindex $list [expr [llength $list]-1]],"
	} 
    }
    return [string range $ret 0 [expr [string length $ret]-2]]
}

proc get_unread {group} {
    global GroupTable
    
    set index [group_index $group]
    set ret [count_range $GroupTable($index,first) \
	   $GroupTable($index,last) $GroupTable($index,irange)]
    return $ret
}



proc find_group {name} {
    global GroupTable NumGroups

    for {set i 0} {$i < $NumGroups} {incr i} {
	if {$name == $GroupTable($i,name)} {
	    return $i
	}
    }
    return -1
}

proc in_range {range value} {
    
    set list [split $range ","]
    foreach item $list {
	set i [string first "-" $item]
	if {$i != -1} {
	    set l [split $item "-"]
	    set lower [lindex $l 0]
	    set upper [lindex $l 1]
	    if {$value >= $lower && $value <= $upper} {
		return 1
	    }
	} else {
	    set check_val $item
	    if {$item == $value} {
		return 1
	    }
	}
    }
    return 0
}
	    
	
    
proc compute_inverse_range {range end} {
    set ret [InverseRange $range $end]
    return $ret
    if {$range == ""} {
	# no range, inverse is 1 to last
	return "1-$end"
    }
    set list [split $range ","]
    set last 1
    foreach item $list {
	set l [split $item "-"]
	if {[llength $l] == 2} {
	    set lower [lindex $l 0]
	    set upper [lindex $l 1]
	} else {
	    set lower $item
	    set upper $item
	}
	if {$last < [expr $lower-1]} {
	    append ret "$last-[expr $lower-1],"
	}
	if {$last == [expr $lower-1]} {
	    append ret "$last,"
	}
	set last [expr $upper+1]
    }
    if {$last <= $end} {
	if {$last == $end} {
	    append ret "$last,"
	} else {
	    append ret "$last-$end,"
	}
    }
    return [string range $ret 0 [expr [string length $ret]-2]]
}



proc adjust_range {first end range} {

    set i [scan $range "%d" lower]
    if {$i != 1} {return $range}
    if {$lower < $first} {
	set i [string length $lower]
	set range "$first[string range $range $i end]"
    }
    set i [string last "," $range]
    if {$i == -1} {
	return $range
    }
    set last [string range $range [expr $i+1] end]
    set l [split $last "-"]
    set lastnum [lindex $l [expr [llength $l]-1]]
    if {$lastnum > $end} {
	if {[llength $l] == 2} {
	    set range "[string range $range 0 $i][lindex $l 0]-$end"
	} else {
	    set range "[string range $range 0 $i]$end"
	}
    }
    return $range
}
	

proc group_listbox_index {name} {
    global GroupTable NumGroups

    set count 0
    for {set i 0} {$GroupTable($i,name) != $name} {incr i} {
	if {$GroupTable($i,unread) > 0} {
	    incr count
	}
	if {$i == $NumGroups} {
	    return -1
	}
    }
    return $count
}


proc select_index {index} {
    global Mode GroupTable NumGroups ViewUnread

    .metainfo.listbox select clear
    if {$Mode == "Groups"} {
	# Is the index they want out of range?
	if {$index < 0 || $index > $NumGroups} {
	    # can't select that
	    return
	}
	# do we have to take special care because ViewUnread is not on?
	if {!$ViewUnread} {
	    # get group_name to index and use that instead of what was passed to us
	    set index [group_listbox_index $GroupTable($index,name)]
	}
    }
    .metainfo.listbox select from $index
    .metainfo.listbox select to $index
    set_view metainfo $index
}

 
proc set_view {window index} {
    if {$window == "metainfo"} {
	sb_set_view .metainfo.vertical $index
    } else {
	sb_set_view .text.vertical $index
    }
}

 
proc sb_set_view {s index} {
 
    set list [$s get]
    set total [lindex $list 0]
    set units [lindex $list 1]
    set top [lindex $list 2]
    set bottom [lindex $list 3]
    
    if {$index < $top || $index > $bottom} {
        set pos [expr $index-$units/2]
        if {$pos < 0} {set pos 0}
        set pos2 [expr $pos+$units]
        $s set $total $units $pos $pos2
        set command [lindex [$s config -command] 4]
        [lindex $command 0] [lindex $command 1] $pos
    } else {
    }
}

 
proc find_next_unread {direction wrap} {
    global CurrentArticle
    global ArticleTable NumArticles ArticleList
 
    if {$direction == "backward"} {
	set increment -1
	set wrap_val -1
	set wrap_to [expr $NumArticles-1]
    } else {
	set increment 1
	set wrap_val [expr $NumArticles]
	set wrap_to 0
    }
    set stop_index [lsearch $ArticleList $CurrentArticle]
    set i [expr $stop_index+$increment]
    while {$i != $stop_index} { 
	if {$i == $wrap_val} {
	    if {$wrap} {
		# wrap around search
		set i $wrap_to
	    } else {
		set i $stop_index
	    }
	} else {
	    if {$ArticleTable([lindex $ArticleList $i],status) == "u"} {
		return [lindex $ArticleList $i]
	    }
	    incr i $increment
	}
    }
    return -1;
}


proc find_next_subject {direction} {
    global ArticleTable ArticleList NumArticles CurrentArticle
    global GroupTable GroupIndex
    global ControlC Resources

    if {$direction == "backward"} {
	write_scroll "Searching for previous article of same subject..."
	set increment -1
    } else {
	write_scroll "Searching for next article of same subject..."
	set increment 1
    }
    set i $CurrentArticle
    incr i $increment
    set original_length [llength $ArticleList] 
    incr original_length 5000
    set loaded [expr $GroupTable($GroupIndex,last)+1]
    set subject $ArticleTable($CurrentArticle,subject)
    set si [string last "Re: " $subject]
    set subject [string range $subject [expr $si+4] end]
    while {($i >= $GroupTable($GroupIndex,first)) && 
	($i <= $GroupTable($GroupIndex,last)) && 
	(!$ControlC)} {
	    write_message "Scanning... $i (Control-C aborts)"
	    update
	    if {![info exists ArticleTable($i,subject)]} {
		# we don't have the article loaded, but maybe it exists
		# lets load in twenty at a time to save round-trip time.
		# First we check to see if we need to load more in.
		if {$i < $loaded} {
		    if {$direction == "backward"} {
			set loaded [expr $i-$Resources(PREFETCH)]
		    } else {
			set loaded [expr $i+$Resources(PREFETCH)]
		    }
		    if {$loaded < 1} { 
			set loaded 1
		    }
		    if {$direction == "backward"} {
			load_article_list $GroupTable($GroupIndex,name) \
			    "$loaded-$i" "{subject} {from}"
		    } else {
			load_article_list $GroupTable($GroupIndex,name) \
			    "$i-$loaded" "{subject} {from}"
		    }
		}
		# if the ArticleTable variable doesn't exist now,
		# then we give up on it.
		if {![info exists ArticleTable($i,subject)]} {
		    incr i $increment
		    continue;
		} 
		# Here we update the display for the article list, adding in the new
		# article listings we just loaded... 
		for {set k $i} {$k != [expr $loaded+$increment]} {incr k $increment} {
		    if {[info exists ArticleTable($k,subject)]} {
			set ArticleTable($k,status) "r"
			insert_article $k
		    }
		}
		refresh_articlelist
		# set the view to our current_selection
		set_view metainfo [lindex [.metainfo.listbox curselection] 0]
	    }
	    if {[string first $subject $ArticleTable($i,subject)] != -1} {
		# we have a match...
		# cleanup.....
		if {[llength $ArticleList] > $original_length} {  
		    set ArticleList [lreplace $ArticleList \
				     $original_length end]
		}
		refresh_articlelist
		return $i
	    }
	    incr i $increment
	}
    if {$ControlC} {set ControlC 0}
    if {[llength $ArticleList] > $original_length} {  
	set ArticleList [lreplace $ArticleList $original_length end]
    }
    write_status
    return -1
}





proc find_next_group {} {
    global GroupTable NumGroups GroupIndex GroupList

    for {set i $GroupIndex} {$i < $NumGroups} {incr i} {
	if {$GroupTable($i,unread) > 0} {
	    return $i
	}
    }
    for {set i [expr $GroupIndex-1]} {$i >= 0} {incr i -1} {
	if {$GroupTable($i,unread) > 0} {
	    return $i 
	}
    }
    return -1
}


proc insert_article {number} {
    global ArticleTable ArticleList NumArticles
    global GroupTable GroupIndex CurrentArticle CurrentUnread

    set i 0
    if {[lsearch $ArticleList $number] == -1} {
	lappend ArticleList $number
	set ArticleList [lisort $ArticleList]
    } else {
	return [lsearch $ArticleList $number]
    }
    set i [lsearch $ArticleList $number]
    if {![info exists ArticleTable($number,subject)]} {
	load_article_list $GroupTable($GroupIndex,name) $number "{subject} {from}"
    }
    if {[info exists ArticleTable($number,status)]} {
	if {$ArticleTable($number,status) == "u"} {
	    set ArticleTable($number,status) "r"
	    incr CurrentUnread -1
	}
    }
    .metainfo.listbox insert [expr [expr [.metainfo.listbox size]-$i]+1] \
	[format_article_line $number]
    return $i
}    
	


proc group_find {requirement direction} {
    global GroupTable NumGroups GroupIndex Mode
    
    if {$Mode != "Groups"} {
	error "group_find called when not in Groups mode!"
	return -1
    }
    if {[.metainfo.listbox curselection] == ""} {
	return -1
    }
    set GroupIndex [lindex [.metainfo.listbox curselection] 0]
    if {$NumGroups == -1 || $GroupIndex == -1} {
	return -1
    }
    if {$direction == "next"} {
	set incval 1
    } else {
	set incval -1
    }
    if {$direction == "next"} {
	if {$GroupIndex == [expr $NumGroups-1]} {
	    # we're at the end... no more next groups.
	    return -1
	} 
    } else {
	if {$GroupIndex == 0} {
	    # we're at the beginning, no more prev groups.
	    return -1
	}
    }
    if {$requirement == "all"} {
	return $GroupTable([expr $GroupIndex+$incval],name)
    } else {
	set index [expr $GroupIndex]
	while {$index != -1 && $index != $NumGroups} {
	    if {$GroupTable($index,unread) > 0} {
		return $GroupTable($index,name)
	    }
	    incr index $incval
	}
	# couldn't find any more groups in that direction with unread
	# articles.
	return -1
    }
}

    

proc next_article {} {
    global GroupTable GroupIndex ArticleList CurrentArticle ArticleTable
    global ControlC

    set art [lsearch $ArticleList $CurrentArticle]
    if {$art == -1} {
	#  strange.... we are probably in trouble if this
	# ever happens....
	return -1
    }
    set number [expr $CurrentArticle+1]
    if {[lsearch $ArticleList $number] != -1} {
	set CurrentArticle $number 
	return $number
    }
    set ret ""
    set original_length [llength $ArticleList]
    while {($ret == "") && (!$ControlC)} {
	if {$number > $GroupTable($GroupIndex,last)} {
	    set ret -1
	    continue
	}
	write_message "Scanning... $number (Control-C aborts)"
	load_article_list $GroupTable($GroupIndex,name) $number "{subject} {from}"
	if {[info exists ArticleTable($number,number)]} {
	    set CurrentArticle $number
	    refresh_articlelist
	    return $number
	}
	incr number
    }
    if {$ControlC} {set ControlC 0}
    return -1
}


proc prev_article {} {
    global GroupTable GroupIndex ArticleList CurrentArticle ArticleTable
    global ControlC Resources

    set art [lsearch $ArticleList $CurrentArticle]
    if {$art == -1} {
	#  strange.... we are probably in trouble if this
	# ever happens....
	puts stdout "Error in previous article."
	return -1
    }
    set number [expr $CurrentArticle-1]
    if {[lsearch $ArticleList $number] != -1} {
	set CurrentArticle $number
	return $number
    }
    set original_length [llength $ArticleList]
    while {!$ControlC} {
	if {$number < $GroupTable($GroupIndex,first)} {
	    set ret -1
	    set ControlC 1
	    continue
	}
	write_message "Scanning... $number (Control-C aborts)"
	if {[info exists ArticleTable($number,number)]} {
	    set CurrentArticle $number
	    set ret $number
	    set ControlC 1
	    continue
	} else {
	    set load [expr $number-$Resources(PREFETCH)]
	    set range "$load-$number"
	    #load in the last XX articles and shove them in the listbox.  Also
	    # increment number by one so that we'll check this article again.
	    load_article_list $GroupTable($GroupIndex,name) $range "{subject} {from}"
	    set inserted 0
	    for {set k $number} {$k >= $load} {incr k -1} {
		if {[info exists ArticleTable($k,subject)]} {
		    set ArticleTable($k,status) "r"
		    insert_article $k
		    if {!$inserted} {set first_one $k} 
		    set inserted 1
		}
	    }
	    set_view metainfo [lindex [.metainfo.listbox curselection] 0]
	    if {$inserted} {
		# first refresh the article list
		refresh_articlelist
		return $first_one
	    }
	}
	incr number -1
    }
    if {$ControlC} {set ControlC 0}
    if {[info exists ret]} {
	refresh_articlelist
	return $ret
    } else {
	return -1
    }
}

proc do_post_article {group article file} {
    global Socket Resources

    if {$article != "original"} { 
	set answer [option_dialog .postquerry 800 "Followup Post Query" "Post response to article $article in $group?" Post Re-edit Cancel]
    } else {
	set answer [option_dialog .postquery 800 "Original Post Query" "Post article to $group?" Post Re-edit Cancel]
    }
    if {$answer == "Cancel"} {
	return 1
    }
    if {$answer == "Re-edit"} {
	return 0
    }
    interface_state wait
    write_scroll "Posting Article to $group....please wait..."
    if {![file exists $file]} {
	return 1
    }
    if {$Resources(POSTINGPROG)  != "nntp" && \
	$Resources(POSTINGPROG) != "NNTP"} {
	    set i [catch {set out [exec sh -c "$Resources(POSTINGPROG) $file"]}]
	    if {$i} {
		write_scroll "WARNING - Unable to post article... did you have more included text than new text???"
		interface_state normal
		return -1
	    }
    } else {
	puts $Socket "post"
	set file [open $file "r"]
	while {[gets $file line] != -1} {
	    puts $Socket $line
	}
	puts $Socket "."
	flush $Socket
	set code -1
	while {$code != 240} {
	    gets $Socket line
	    scan $line "%d" code
	    if {$code == 441} {
		# article not posted -- more included text than new text...
		write_brief "Article not posted.... more included text than new text..."
		interface_state normal
		return 1
	    }
	}
    }
    write_scroll "Article Posted..."
    interface_state normal
    return 1
}
    

proc mail_file {group article file} {
    global Options

    set answer [option_dialog .mailquerry 800 "Mail Reply Query" "Send mail response to article $article in group $group?" Send Re-edit Cancel]
    if {$answer == "Cancel"} {
	return 1
    }
    if {$answer == "Re-edit"} {
	return 0
    }
    set infile [open $file "r"]
    gets $infile line
    close $infile
    if {![regexp "To\: (.*)" $line junk email]} {
	option_dialog .blah 1200 "Error" "Bad mail format... message not sent" OK
	write_brief "Bad mail format... message not sent"
	return 1
    }
    exec sh -c "/usr/lib/sendmail -oo -oi -t < $file" &
    write_brief "Mail Message Sent..."
    return 1
}



proc check_for_posts {} {
    global PostingList Options Resources

    flush stdout
    if {$PostingList == {}} {
	flush stdout
	return
    }
    foreach entry $PostingList {
	set file "[lindex $entry 3].done"
	set type [lindex $entry 0]
	if {[file exists $file]} {
	    if {$type == "post"} {
		set i [lsearch $PostingList $entry]
		set PostingList [lreplace $PostingList $i $i]
		if {[do_post_article [lindex $entry 1] [lindex $entry 2] [lindex $entry 3]]} {
		    catch "exec rm [lindex $entry 3]"
		    catch "exec rm [lindex $entry 3].done"
		} else {
		    catch "exec rm [lindex $entry 3].done"
		    if {$Options(Gce)} {
			set i [catch {set out [exec /bin/sh -c "($Resources(gceEditor) [lindex $entry 3] ; /usr/bin/touch [lindex $entry 3].done) &" &]}]
		    } else {
			set i [catch {set out [exec /bin/sh -c "($Resources(EDITOR_COMMAND) [lindex $entry 3] ; /usr/bin/touch [lindex $entry 3].done) &" &]}]
		    }
		    if {$i} {
			write_scroll "ERROR -- $out"
		    }
		    lappend PostingList $entry
		}
	    }
	    if {$type == "mail"} {
		set i [lsearch $PostingList $entry]
		set PostingList [lreplace $PostingList $i $i]
		if {[mail_file [lindex $entry 1] [lindex $entry 2] [lindex $entry 3]]} {
		    exec rm [lindex $entry 3]
		    exec rm [lindex $entry 3].done
		} else {
		    exec rm [lindex $entry 3].done
		    exec /bin/sh -c "($Resources(EDITOR_COMMAND) [lindex $entry 3] ; /usr/bin/touch [lindex $entry 3].done) &" &
		    lappend PostingList $entry
		}
	    }
	}
    }
    after 3000 check_for_posts
}


proc print_group {index} {
    global GroupTable

    puts stdout "-($index)----------------"
    puts stdout "Name: $GroupTable($index,name)"
    puts stdout "First: $GroupTable($index,first)"
    puts stdout "Last: $GroupTable($index,last)"
    puts stdout "read: $GroupTable($index,range)"
    puts stdout "unread: $GroupTable($index,irange)"
    puts stdout "count: $GroupTable($index,unread)"
    puts stdout "status: $GroupTable($index,status)"
    puts stdout "---------------------------"
}


proc print_article {which} {
    global ArticleTable
    
    puts stdout "-($which)-------------------"
    puts stdout "Subject : $ArticleTable($which,subject)"
    puts stdout "From    : $ArticleTable($which,from)"
    puts stdout "Email   : $ArticleTable($which,email)"
    puts stdout "Name    : $ArticleTable($which,name)"
    puts stdout "Date    : $ArticleTable($which,date)"
    puts stdout "Lines   : $ArticleTable($which,lines)"
    puts stdout "Id      : $ArticleTable($which,message-id)"
    puts stdout "Refs    : $ArticleTable($which,references)"
    puts stdout "F-up 2  : $ArticleTable($which,followup-to)"
}    

#
# This procedure makes sure the indicated article has all of its
# information loaded.
#
proc ensure_article {art} {
    global ArticleTable GroupTable GroupIndex

    set fields [list subject from email name date lines message-id \
		references followup-to]
    set list ""
    foreach field $fields {
	if {![info exists ArticleTable($art,$field)]} {
	    lappend list $field
	}
    }
    if {$list != ""} {
	load_article_list $GroupTable($GroupIndex,name) $art $list
    }
}

proc append_newsgroup {name first last range} {
    global GroupTable NumGroups GroupIndex  GroupName2Index
    global Resources

    set i [group_index $name]
    if {$i == -1} {
	set GroupIndex $NumGroups
	incr NumGroups
    } else {
	set GroupIndex $i
    }
    set GroupTable($GroupIndex,name) $name
    set GroupTable($GroupIndex,first)  $first
    set GroupTable($GroupIndex,last)   $last
    set GroupTable($GroupIndex,range) [adjust_range $first \
				       $last $range]
    if {$GroupTable($GroupIndex,first) < 1} {
	set GroupTable($GroupIndex,first) 1
    }
    if {$GroupTable($GroupIndex,last) < 1} {
	set GroupTable($GroupIndex,last) 1
    }
    set GroupTable($GroupIndex,irange) \
	[adjust_range $first $last \
	 [compute_inverse_range $GroupTable($GroupIndex,range) \
	  $GroupTable($GroupIndex,last)]]
    set GroupTable($GroupIndex,status) "s"
    set GroupTable($GroupIndex,unread) \
	[get_unread $name]
    set GroupName2Index($name) $GroupIndex
} 
 


proc write_status {} {
    global Mode UnreadArticles VisibleGroups NumGroups CurrentUnread
    global GroupTable GroupIndex ArticleTable CurrentArticle 
    global NumActive SubscribeList

    if {$Mode == "Groups"} {
	write_message "$UnreadArticles unread articles in $VisibleGroups/$NumGroups subscribed groups."
    }
    if {$Mode == "Articles"} {
	write_message "$CurrentUnread unread articles in $GroupTable($GroupIndex,name) : Article $CurrentArticle ($GroupTable($GroupIndex,last))"
    }
    if {$Mode == "NewGroups"} {
	write_message "[.metainfo.listbox size] new newsgroups..."
    }
    if {$Mode == "GroupList"} {
	write_message "$NumActive Total newsgroups.... [llength $SubscribeList] subscribed."
    }
}



proc ag_command_frame {} {
    
    frame .agcom
    button .agcom.subscribe -text "Subscribe" -command "set_group_status subscribe"
    button .agcom.unsubscribe -text "Unsubscribe" \
	-command "set_group_status unsubscribe"
    button .agcom.toggle -text "Toggle Status" -command "set_group_status toggle"
    button .agcom.save -text "Save Changes" -command "ag_save"
    button .agcom.quit -text "Quit" -command "ag_save ; rescan 0 ; quit_mode"
    button .agcom.abort -text "Abort" -command "quit_mode"
    pack append .agcom \
	.agcom.toggle {left fillx expand} \
	.agcom.subscribe {left fillx expand} \
	.agcom.unsubscribe {left fillx expand} \
	.agcom.save {left fillx expand} \
	.agcom.quit {left fillx expand} \
	.agcom.abort {left fillx expand}
    generate_cursorlist
}


proc construct_gn2i {} {
    global GroupTable NumGroups GroupName2Index

    unset GroupName2Index
    for {set i 0} {$i < $NumGroups} {incr i} {
	set GroupName2Index($GroupTable($i,name)) $i
    }
}


proc delete_group {group} {
    global GroupTable NumGroups GroupName2Index

    set index $GroupName2Index($group)
    for {set i $index} {$i < [expr $NumGroups-1]} {incr i} {
	set j [expr $i+1]
	set GroupTable($i,name) $GroupTable($j,name)
	set GroupTable($i,range) $GroupTable($j,range)
	set GroupTable($i,irange) $GroupTable($j,irange)
	set GroupTable($i,first) $GroupTable($j,first)
	set GroupTable($i,last) $GroupTable($j,last)
	set GroupTable($i,unread) $GroupTable($j,unread)
	set GroupTable($i,status) $GroupTable($j,status)
    }
    unset GroupTable($i,name)
    unset GroupTable($i,range)
    unset GroupTable($i,irange)
    unset GroupTable($i,first)
    unset GroupTable($i,last)
    unset GroupTable($i,unread)
    unset GroupTable($i,status)
    incr NumGroups -1
    construct_gn2i
}


proc ag_save {} {
    global SubscribeList UnsubscribeList GroupTable NumGroups 
    global GroupName2Index InterfaceBusy Mode Resources

    if {$InterfaceBusy} {return}
    interface_state wait
    write_scroll "Saving changes...." 
    if {$Mode == "NewGroups"} {
	set list {}
	for {set i 0} {$i < [.metainfo.listbox size]} {incr i} {
	    set line [.metainfo.listbox get $i]
	    regexp {^([^ \t]+)[ \t]*(.*)$} $line junk status group
	    set status [string index $status 0]
	    if {![info exists GroupName2Index($group)]} {
		if {$status != "u"} {
		    set GroupTable($NumGroups,name) "$group"
		    set GroupTable($NumGroups,range) ""
		    set GroupTable($NumGroups,irange) ""
		    set GroupTable($NumGroups,first) 0
		    set GroupTable($NumGroups,last) 0
		    set GroupTable($NumGroups,unread) 0
		    set GroupTable($NumGroups,status) "$status"
		    set GroupName2Index($group) $NumGroups
		    incr NumGroups
		}
	    } else {
		set j $GroupName2Index($group)
		set GroupTable($j,status) $status
		if {$status == "u"} {
		    lappend list $group
		}
	    }
	}
	interface_state normal
	save_newsrc
	foreach group $list {
	    delete_group $group
	}
	# Now save the access time to our .tknewsrc
	set i [catch {set mytime [file mtime $Resources(NEWSRCFILE)]}]
	if {$i} {
	    set mytime [file mtime [tilde_sub ~]]
	}
	set myl ""
	lappend myl "lastNewgroupCheck $mytime"
	set_tknewsrc $myl
    } else {
	foreach group $SubscribeList {
	    if {![info exists GroupName2Index($group)]} {
		# construct a pseudo-GroupTable entry
		set GroupTable($NumGroups,name) "$group"
		set GroupTable($NumGroups,range) ""
		set GroupTable($NumGroups,irange) ""
		set GroupTable($NumGroups,first) 0
		set GroupTable($NumGroups,last) 0
		set GroupTable($NumGroups,unread) 0
		set GroupTable($NumGroups,status) "s"
		set GroupName2Index($group) $NumGroups
		incr NumGroups
	    }
	}
	foreach group $UnsubscribeList {
	    if {[info exists GroupName2Index($group)]} {
		# delete the group from the GroupTable
		delete_group $group
	    }
	}
	interface_state normal
	save_newsrc
    }
}



proc get_active_list {} {
    global ActiveTable GroupTable NumGroups

    set list ""
    for {set i 0} {$i < $NumGroups} {incr i} {
	set name $GroupTable($i,name)
	lappend list "subscribed      $name"
	set atemp($i,name) $ActiveTable($name,name)
	set atemp($i,first) $ActiveTable($name,first)
	set atemp($i,last) $ActiveTable($name,last)
	unset ActiveTable($name,name)
	unset ActiveTable($name,first)
	unset ActiveTable($name,last)
    }
    set alist [array name ActiveTable]
    foreach item $alist {
	if {[regexp "(.*),name" $item junk group] > 0} {
	    if {[string first "." $group] == -1} {
	    }
	    lappend list "unsubscribed    $group"
	}
    }
    for {set i 0} {$i < $NumGroups} {incr i} {
	set name $atemp($i,name)
	set ActiveTable($name,name) $name
	set ActiveTable($name,first) $atemp($i,first)
	set ActiveTable($name,last) $atemp($i,last)
    }
    return $list
}


proc get_access_time {file} {
    set i [catch "file mtime $file" output]
    if {$i} {
	return -1
    } 
    return $output
}


proc metainfo_adjust {action} {
    global InterfaceOrientation
    global NewGeom 
    
    set packing $InterfaceOrientation
    if {$action == "grow"} {
	set value 1
    } else {
	set value -1
    }
    if {$packing == "horizontal"} {
	set value [expr $value*5]
    }
    if {[info exists NewGeom]} {
	set geom $NewGeom
    } else {
	set geom [lindex [.metainfo.listbox configure -geometry] 4]
    }
    set l [split $geom "x"]
    set width [lindex $l 0]
    set l [split [lindex $l 1] "+-"]
    set height [lindex $l 0]
    if {$packing == "vertical"} {
	# increment the height by 'value'
	incr height $value
    } else {
	incr width $value
    }
    set newgeom $width
    append newgeom "x"
    append newgeom $height
    set NewGeom $newgeom
    after 200 "metainfo_doadjust $newgeom"
}
    
proc metainfo_doadjust {newgeom} {
    global NewGeom NewTextGeom 
    
    if {$NewGeom == $newgeom} {
	.metainfo.listbox configure -geometry $newgeom
    }
    update
}
    


proc group_getpath {name} {
    global Resources
    
    set ret $Resources(SPOOLDIR)
    set l [split $name "."]
    foreach dir $l {
	append ret "/$dir"
    }
    return $ret
}
	



proc load_active_groups {} {
    global Resources
    global ActiveTable NumActive SubscribeList 

    if {[info exists ActiveTable]} {
	set list [get_active_list]
    } else {
	if {[info exists Resources(NNTPSERVER)]} {
	    puts $Socket "list"
	    flush $Socket
	    set file $Socket
	} else {
	    set file [open "$Resources(NEWSACTIVEFILE)" "r"]
	}
	set list ""
	set NumActive 0
	set a [gets $file line]
	while {($a != -1) && ($line != ".")} {
	    set i [string first " " $line]
	    set line [string range $line 0 [expr $i]]
	    set group [string range $line 0 [expr $i-1]]
	    set j [lsearch $SubscribeList $group]
	    if {$j == -1} {
		lappend list "unsubscribed    $group"
	    } else {
		lappend list "subscribed      $group"
	    }
	    set a [gets $file line]
	    set line [string trimright $line]
	    incr NumActive
	}
	if {![info exists Resources(NNTPSERVER)]} {
	    close $file
	}
    }
    flush stdout
    set list [lsort $list]
    flush stdout
    foreach element $list {
	.metainfo.listbox insert end $element
    }
    return $list
}



proc new_tmp_file {desig} {
    return "/tmp/$desig-[randnum 90000]"
}



proc create_tknewsrc {} {
    global Version Resources LIBRARY_DIR
    
    if {$Resources(autoHelp)} {
	set i [catch {set file [open "$LIBRARY_DIR/help/welcome" "r"]}]
	if {!$i} {
	    free_dialog .nograb1 300 "WELCOME" [GetFile $file] Dismiss
	}
	catch "close $file"
    }
    set file [open $Resources(TKNEWSRC) "w"]
    set i [catch {set time [file mtime $Resources(NEWSRCFILE)]}]
    if {$i} {set time [file mtime [tilde_sub ~]]}
    incr time -604800
    puts $file "*lastNewgroupCheck: $time"
    close $file
}

proc line_has_default {file which val} {
    global Resources Version

    if {$val != "default"} {return 0}
    if {$which == "lastNewgroupCheck"} {
	set i [catch {set time [file mtime $Resources(NEWSRCFILE)]}]
	if {$i} {
	    set time [file mtime [tilde_sub ~]]
	}
	# set time back one week from last time we accessed our newsrc
	incr time -604800
	puts $file "*lastNewgroupCheck: $time"
	return 1
    }
    if {$which == "version"} {
	puts $file "*version: $Version"
	return 1
    }
    return 0
}



proc set_tknewsrc {list} {
    global Resources Version

    set file [open $Resources(TKNEWSRC) "r"]
    set tname [new_tmp_file tknewsrc]
    set tfile [open $tname "w"]
    while {[gets $file line] != -1} {
	set found -1
	set count 0
	foreach l $list {
	    set which [lindex $l 0]
	    if {[regexp $which $line] != 0} {
		set val [lindex $l 1]
		set found $count
		# line matched... 
		if {![line_has_default $tfile $which $val]} {
		    puts $tfile "*$which: $val"
		}
	    }
	    incr count
	}
	if {$found != -1} {
	    set list [lreplace $list $found $found]
	} else {
	    puts $tfile $line
	}
    }
    close $file
    foreach l $list {
	set which [lindex $l 0] 
	set val [lindex $l 1]
	if {![line_has_default $tfile $which $val]} {
	    puts $tfile "*$which: $val"
	}
    }
    close $tfile
    exec mv $tname $Resources(TKNEWSRC)
}
	
proc get_tknewsrc_line {which} {
    global Resources

    if {![file exists $Resources(TKNEWSRC)]} {
	create_tknewsrc
    }
    set file [open $Resources(TKNEWSRC) "r"]
    set line ""
    while {$line == ""} {
	set line [FileGetLine $file "*$which"]
	if {$line == ""} {
	    # the line doesn't exist, set it to an appropriate value.
	    close $file
	    set_tknewsrc {$which default}
	    return ""
	}
    }
    close $file
    return $line
}     


proc load_new_groups {} {
    global Resources

    if {![file exists $Resources(TKNEWSRC)]} {
	create_tknewsrc
    }
    set i 0
    while {!$i} {
	set line [get_tknewsrc_line "lastNewgroupCheck"]
	set i [regexp "lastNewgroupCheck:\[ \t\]+(\[0-9\]+)" $line junk time]
	if {!$i} {
	    # didn't get time, format of the line was bad.
	    # set the time as appropriate
	    set_tknewsrc {{lastNewgroupCheck default}}
	}
    }
    set list [get_new_groups $time]
    if {$list == {}} {
	write_scroll "No new newsgroups..."
    } else {
	foreach group $list {
	    .metainfo.listbox insert end "unsubscribed    $group"
	}
    }
    return $list
}



proc set_mode {mode} {
    global Mode 

    set Mode $mode
}

    

proc check_install {} {
    global Version

    set line [get_tknewsrc_line "version"]
    set i [regexp $Version $line junk version]
    if {!$i} {
	version_info 
	set_tknewsrc {version $Version}
    } 
}


proc version_info {} {
    global LIBRARY_DIR Version

    set i [catch {set file [open "$LIBRARY_DIR/VersionInfo.$Version"  "r"]}]
    if {$i} {
	set text "Information not available..."
    } else {
	set text ""
	while {[gets $file line] != -1} {
	    append text "$line\n"
	}
    }
    option_dialog .versioninfo 300 "Version Info" $text Dismiss
}

    
proc start_auto_rescan {} {
    global ElapsedTime TIME_GRAIN 
    global Resources

    set ElapsedTime 0
    after $Resources(rescanDelay) auto_rescan
    after $TIME_GRAIN update_elapsed_time
}

proc update_elapsed_time {} {
    global ElapsedTime TIME_GRAIN

    incr ElapsedTime [expr $TIME_GRAIN/1000]
    after $TIME_GRAIN update_elapsed_time
}

proc auto_rescan {} {
    global ElapsedTime LastActiveTime Mode
    global Resources
    
    if {[expr $ElapsedTime-$LastActiveTime] > 
	[expr $Resources(rescanDelay)/60000]} {
	    if {$Mode == "Groups"} {
		write_message "Auto-rescan in progress..."
		write_scroll "Auto rescan..."
		rescan 1
		write_scroll "Auto-rescan complete"
	    }
	    set LastActiveTime $ElapsedTime
	}
    after $Resources(rescanDelay) auto_rescan
}


proc disable_menu {m} {
    for {set i 0} {$i < 20} {incr i} {
	$m entryconfigure $i -state disabled
    }
}


proc set_menu_state {} {
    global Mode Options

    #
    # Also set the dragndrop icon
    #
    if {$Options(Gce)} {
	if {$Mode == "Articles"} {
	    .comm.icon.object configure -state normal
	} else {
	    .comm.icon.object configure -state disabled
	}
    }
    set m .menubar.action.m
    disable_menu $m
    set list {}
    if {$Mode == "Groups"} {
	set list {1 4 7 9 10 11 13 14 15 16 17 18}
    } 
    if {$Mode == "Articles"} {
	set list {0 1 2 3 4 5 6  8 9 10 11 12 13 18}
    }
    foreach index $list {
	$m entryconfigure $index -state normal
    }

    set m .menubar.mark.m
    disable_menu $m
    if {$Mode == "Articles"} {
	for {set i 0} {$i < 6} {incr i} {
	    $m entryconfigure $i -state normal
	}
    }
    
    set m .menubar.move.m 
    disable_menu $m
    set list {}
    if {$Mode == "Articles"} {
	set list {0 1 2 3 4 5}
    }
    foreach index $list {
	$m entryconfigure $index -state normal
    }
}
	
    

proc trim_range {range max} {
    set l [split $range ","]
    set count 0
    set index [expr [llength $l]-1]
    set nrange ""
    while {$index >= 0} {
	set r [lindex $l $index]
	incr count [count_range 0 999999 $r]
	set i [expr $count-$max]
	if {$i > 0} {
	    # we've gone over the maximum number of articles allowed
	    # We'll trim the range at the max value and return it.
	    if {[string first "-" $r] != -1} {
		set j [split $r "-"]
		set lower [lindex $j 0]
		set upper [lindex $j 1]
		incr lower $i
		set nr "$lower-$upper"
		set nrange "$nrange,$nr"
		return [string range $nrange 1 end]
	    } else {
		# this was just a one article range and thus
		# we just ignore it.
		return [string range $nrange 1 end]
	    }
	} else {
	    # the range hasn't gone over the max limit, so just 
	    # append it to our return value.
	    set nrange "$nrange,$r"
	}
	incr index -1
    }
    return [string range $nrange 1 end]
}
	    
	    

proc search_init {} {
    global Search Resources InterfaceBusy

    if {$InterfaceBusy} {return}
    set Search(string) [.cpanel.io.entry get]
    if {$Search(string) == ""} {
	option_dialog .si 1200 "Search Error" \
	    "No search string specified" OK
	focus .metainfo
	return
    }
    interface_state busy
    .info.text tag delete Search
    set list ""
    set Search(num_matches) 0
    set lines [lindex [.info.vertical get] 0]
    for {set i 1} {$i <=  $lines} {incr i} {
	set line [.info.text get $i.0 "$i.0 lineend"]
	set res [catch \
	       {set m [regexp -indices $Search(string) $line match]}]
	
	if {!$res} {
	    if {$m} {
		.info.text tag add Search \
		    $i.[lindex $match 0] $i.[expr [lindex $match 1]+1]
		lappend list $i
		incr Search(num_matches)
	    }
	}
    }
    set Search(list) $list
    set Search(state) 0
    .info.text tag configure Search \
	-foreground $Resources(highlightForeground) \
	-background $Resources(highlightBackground) 
    focus .metainfo
    interface_state normal
}

proc search_next {} {
    global Search

    set i $Search(state)
    if {$i == -1} {
	option_dialog .sn 1200 "Search Error" \
	    "No search currently in effect." OK
	return
    }
    if {!$Search(num_matches)} {
	return
    }
    set oldline [lindex $Search(list) $i]
    incr i
    set newline [lindex $Search(list) $i]
    while {$i < $Search(num_matches) && $newline == $oldline} {
	incr i
	set newline [lindex $Search(list) $i]
    }
    if {$i == $Search(num_matches)} {
	set i 0
	set newline [lindex $Search(list) 0]
    }
    set Search(state) $i
    .info.text yview [expr $newline-2]
    set info [.info.vertical get]
}


proc search_clear {} {
    global Search

    set Search(state) -1
    .info.text tag delete Search
}

proc change_text_font {name1 name2 op} {
    global Resources Mode
    
    puts stdout "Changing text font to $Resources(textFont)."
   .info.text configure -font $Resources(textFont)
}

proc toggle_keyboard {name1 name2 op} { 
    global Resources BindingList Mode

    if {$Resources(disableKeyboard)} {
	# override all key bindings.
	set list [array names BindingList]
	foreach action $list {
	    for {set u 0} {$u < [llength $BindingList($action)]} {incr u} {
		set l [lindex $BindingList($action) $u]
		set window [lindex $l 0]
		set binding [lindex $l 1]
		if {![regexp {[bB]utton} $binding junk]} {
		    bind $window $binding ""
		} else {
		}
	    }
	}
	catch {unset BindingList}
    } else {
	set_translations
    }
    destroy_menus
    load_menus
    if {[info exists Mode]} {
	configure_for_mode
    }
}


#
# This procedure sets the state of the command panel buttons, disabling
# those which are not currently applicable.
#
proc set_cpanel_state {} {
    global Mode ButtonList

    set alllist [list enter_group quit_mode goto_article \
		 goto_current_selection do_the_right_thing view_action \
		     select_action article_action catch_up mark_article \
		     save_newsrc post_article mail_article save_article \
		     rescan add_newsgroup all_groups unsubscribe \
		     save_tknewsrc list_old]
    set grouplist [list enter_group goto_current_selection \
		   do_the_right_thing select_action catch_up \
		       save_newsrc "post_article original" \
		       "mail_article compose" \
		       rescan add_newsgroup all_groups unsubscribe \
		       save_tknewsrc list_old]
    set articlelist [list goto_current_selection do_the_right_thing \
		     view_action select_action article_action \
			 catch_up mark_article save_newsrc \
			 post_article mail_article save_article \
			 unsubscribe]
    for {set i 0} {$i < $ButtonList(NumButtons)} {incr i} {
	set b .cpanel.buttons.button$i
	case $Mode in {
	    {Groups} {
		foreach string $grouplist {
		    if {[string first $string $ButtonList(action,$i)] != -1} {
			$b configure -state normal
		    } else {
			$b configure -state disabled
		    }
		}
	    }
	    {ArticleList} {
		foreach string $articlelist {
		    if {[string first $string $ButtonList(action,$i)] != -1} {
			$b configure -state normal
		    } else {
			$b configure -state disabled
		    }
		}
	    }		    
	}
    }
}
		    
#
# This procedure configures the menus and the command panel (if it 
# exists) based on the mode.
#
proc configure_for_mode {} {
    set_menu_state
}


proc tilde_sub {path} {
    
    set i [string first "/" $path]
    if {$i == -1} {
	return $path
    }
    if {[string index $path 0] != "~"} {
	return $path
    }
    set expand [string range $path 0 [expr $i-1]]
    set err [catch {set result [glob $expand]}]
    if {$err} {
	# can't expand... oh well
	return $path
    }
    return "$result[string range $path $i end]"
}

