#
# utils.tk
#	Utilities for tk2.3
#	These are convenience procedures that ease construction of
#	buttons, listboxes, etc.  They define all the colors of the
#	widgets based on a set of complementary colors that can
#	be defined externally.  (See also colors.tk.)
#
#	Buttons
#		buttonFrame
#		simpleButton
#		packedButton
#		framedButton
#		packedCheckButton
#		packedRadioButton
#	Menu
#		basicMenu
#		packedMenuButton
#		menuAndButton
#		framedMenuButton
#		menuAndFButton
#	Scrollbar
#		basicScrollbar
#	Listbox
#		labeledListbox
#		unixCommandListbox
#	Entry
#		labeledEntry
#		commandEntry
#		labeledEntryWithDefault
#	Feedback
#		feedbackSetup
#		feedback
#	Toplevel
#		notifier
#	Message
#		unixCommandMessageButton
#		unixCommandMessage
#

#
# to_tx - insert characters into the tx command stream.  This is used to
# feed commands to the csh running in the tx that started this program.
#
proc to_tx {str} {
	puts stdout "\33insert [list $str]"
	puts stdout "\33insert \\n"
}

#
# showArgs - 
proc showArgs { args } {
    puts stdout "NumArgs: [llength $args]"
    set i 1
    foreach arg $args {
	puts stdout " Arg$i: $arg"
	incr i
    }
}

#
# selfName - determine the name of a nested widget
#	parent is either "." or ".foo.bar"
#	name is ".zork"
#
proc selfName { parent name } {
    if {[string compare $parent "."] == 0} {
	set self $name
    } else {
	set self $parent$name
    }
    return $self
}

proc utilsInit { } {
    utilSetFont fixed
    #
    # Default colors.
    # See also colors.tk for a better setColorCube
    #
    global backgroundColor paleBackground foregroundColor 
    global passiveColor activeColor
    set backgroundColor		#cb02dd
    set paleBackground		#ffceff
    set foregroundColor		black
    set passiveColor		#eeadf3
    set activeColor			#f154ff

}

proc utilSetFont { font } {
    global buttonFont labelFont menuFont entryFont
    # Default font for buttons, labels, menus
    set buttonFont		$font
    set labelFont		$font
    set menuFont		$font
    set entryFont		$font    
}
#
# buttonFrame creates a frame that is designed to hold a row of buttons
#
proc buttonFrame { parent {name .buttons} {border 2} } {
    global backgroundColor
    if ![info exists backgroundColor] {
	utilsInit
    }
    set self [selfName $parent $name]
    set color [format #%02x%02x%02x 240 128 0]
    frame $self -borderwidth $border -background $backgroundColor \
	-relief raised
    pack append $parent $self {top fillx}
    return $self
}
#
# packedButton adds a button to a row of buttons
#
proc packedButton { parent name label command {position left} {color default} } {
    global foregroundColor activeColor passiveColor
    global buttonFont
    global FontWidgets	;# Remember widgets with a font

    if ![info exists buttonFont] {
	utilsInit
    }
    set savedColor [getColorCube]
    if {[string compare $color "default"] != 0} {
	setColorCube $color
    }
    set self [selfName $parent $name]
    button $self -text $label -command $command \
	-font $buttonFont \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-activeforeground $passiveColor
    lappend FontWidgets $self
    pack append $parent $self $position
    setColorCube $savedColor
    return $self
}
#
# framedButton - like packedButton but with more space around it
#
proc framedButton {parent name label command {position left} {color default} } {
    global foregroundColor activeColor passiveColor backgroundColor
    global buttonFont FontWidgets

    set padwidth 4		;# Framing width.  Could be a parameter

    set self [selfName $parent $name]

    if ![info exists buttonFont] {
	utilsInit	;# Define colors and fonts for utils.tk procs
    }

    # Keep frame in the default background color
    frame $self -borderwidth $padwidth -background $backgroundColor

    # "Color cubes" are sets of colors that go together.
    # setColorCube sets the current set of colors.
    # Here we can change the color scheme of the button
    set savedColor [getColorCube]	
    if {[string compare $color "default"] != 0} {
	setColorCube $color
    }

    button $self.b -text $label -command $command \
	-font $buttonFont \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-activeforeground $passiveColor
    lappend FontWidgets $self.b		;# Supports mxedit font changes
    pack append $parent $self $position
    pack append $self $self.b { fill }
    setColorCube $savedColor
    return $self.b
}

#
# simpleButton makes some simplifying assumptions - similar to packedButton
#
proc simpleButton { label command {position left} {color default} } {
    global foregroundColor activeColor passiveColor
    global FontWidgets	;# Remember widgets with a font
    global buttonFont

    if ![info exists foregroundColor] {
	utilsInit
    }
    set savedColor [getColorCube]
    if {[string compare $color "default"] != 0} {
	setColorCube $color
    }
    set self [selfName $parent $name]
    button $self -text $label -font $buttonFont -command $command \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor
    lappend FontWidgets $self
    pack append $parent $self $position
    setColorCube $savedColor
    return $self
}
#
# packedCheckButton
#
proc packedCheckButton { parent name label command { variable selectedButton } {position left} } {
    global passiveColor foregroundColor activeColor
    global buttonFont FontWidgets

    if ![info exists foregroundColor] {
	utilsInit
    }
    set self [selfName $parent $name]
    checkbutton $self -text $label -font $buttonFont -command $command \
	-variable $variable \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-selector $activeColor
    lappend FontWidgets $self
    pack append $parent $self $position
    return $self

}

#
# packedRadioButton
#
proc packedRadioButton { parent name label command { variable selectedButton } {position left} } {
    global passiveColor foregroundColor activeColor
    global buttonFont FontWidgets

    if ![info exists foregroundColor] {
	utilsInit
    }
    set self [selfName $parent $name]
    radiobutton $self -text $label -font $buttonFont -command $command \
	-variable $variable \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-selector $activeColor
    lappend FontWidgets $self
    pack append $parent $self $position
    return $self

}

#
# Basic Menu
#
proc basicMenu { name } {
    global foregroundColor
    global activeColor
    global backgroundColor
    global paleBackground
    global passiveColor

    global menuFont FontWidgets

    if ![info exists menuFont] {
	utilsInit
    }
    set self [menu $name -font $menuFont \
	-selector $activeColor \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activeforeground $paleBackground \
	-activebackground $activeColor]
    lappend FontWidgets $self

    return $self
}
#
# packedMenuButton adds a menubutton to a row of buttons
#
proc packedMenuButton { parent name label menu {position left} {color default} } {
    global foregroundColor activeColor passiveColor paleBackground
    global menuFont FontWidgets

    if ![info exists menuFont] {
	utilsInit
    }
    set savedColor [getColorCube]
    if {[string compare $color "default"] != 0} {
	setColorCube $color
    }
    set self [selfName $parent $name]
    menubutton $self -text $label -menu $menu \
	-relief raised \
	-font $menuFont \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-activeforeground $paleBackground
    lappend FontWidgets $self
    pack append $parent $self $position
    setColorCube $savedColor
    return $self
}
# menuAndButton

proc menuAndButton { menubar name label {where {left}} } {
    set menuPathName $menubar${name}.menu
    packedMenuButton $menubar ${name} $label $menuPathName $where
    set menu [basicMenu $menuPathName]
    return $menu
}

# framedMenuButton

proc framedMenuButton {parent name label menu {position left} {color default} } {
    global foregroundColor activeColor passiveColor backgroundColor
    global buttonFont FontWidgets

    set padwidth 4		;# Framing width.  Could be a parameter

    set self [selfName $parent $name]

    if ![info exists buttonFont] {
	utilsInit	;# Define colors and fonts for utils.tk procs
    }

    # Keep frame in the default background color
    frame $self -borderwidth $padwidth -background $backgroundColor

    # "Color cubes" are sets of colors that go together.
    # setColorCube sets the current set of colors.
    # Here we can change the color scheme of the button
    set savedColor [getColorCube]	
    if {[string compare $color "default"] != 0} {
	setColorCube $color
    }

    menubutton $self.b -text $label -menu $menu \
	-relief raised \
	-font $buttonFont \
	-background $passiveColor \
	-foreground $foregroundColor \
	-activebackground $activeColor \
	-activeforeground $passiveColor
    lappend FontWidgets $self.b		;# Supports mxedit font changes
    pack append $parent $self $position
    pack append $self $self.b { fill }
    setColorCube $savedColor
    return $self.b
}

# menuAndFButton (framed button)
#
proc menuAndFButton { menubar name label {where {left}} } {
    set menuPathName $menubar${name}.b.menu
    framedMenuButton $menubar ${name} $label $menuPathName $where
    set menu [basicMenu $menuPathName]
    return $menu
}

#
# basicScrollbar
#
proc basicScrollbar { parent command
		      {where {left filly frame w}}
		      {name .scroll} } {
    global passiveColor activeColor paleBackground backgroundColor
    if ![info exists backgroundColor] {
	utilsInit
    }
    set self [scrollbar $parent$name -command "$command" \
	-relief raised \
	-background $backgroundColor \
	-foreground $passiveColor \
	-activeforeground $activeColor]
    pack append $parent $self $where
    return $self
}
#
# basicListbox creates a listbox and a scrollbar
#
proc basicListbox { parent name {geometry 10x5} {position left} } {
    global passiveColor activeColor paleBackground
    global labelFont FontWidgets
    if ![info exists labelFont] {
	utilsInit
    }
    set self [selfName $parent $name]
    frame   $self  -background $passiveColor
    lappend FontWidgets $self.list
    scrollbar $self.scroll -command "$self.list yview" 	\
	-background $paleBackground -foreground $passiveColor 	\
	-activeforeground $activeColor
    listbox $self.list -geometry $geometry -yscroll "$self.scroll set" 	\
	-background $paleBackground -selectbackground $activeColor \
	-font $labelFont -relief raised
    pack append $parent $self "$position"
    pack append $self $self.scroll {right filly} $self.list {left expand fill}
    return $self
}
#
# labeledListbox creates a listbox that has a label above it
#
proc labeledListbox { parent name
		    {text "Label"} {geometry 10x5} {position left} } {
    global passiveColor activeColor paleBackground
    global labelFont FontWidgets
    if ![info exists labelFont] {
	utilsInit
    }
    set self [selfName $parent $name]
    frame   $self  -background $passiveColor
    label   $self.label -text $text -font $labelFont -background $passiveColor
    lappend FontWidgets $self.label $self.list
    scrollbar $self.scroll -command "$self.list yview" 	-background $paleBackground -foreground $passiveColor 	-activeforeground $activeColor
    listbox $self.list -geometry $geometry -yscroll "$self.scroll set" 	-background $paleBackground -selectbackground $activeColor -font $labelFont
    pack append $parent $self "$position"
    pack append $self $self.label {top} $self.scroll {right filly} $self.list {left expand fill}
    return $self
}
#
# labeledListbox2 creates a listbox that has a label above it and 2 scrollbars
#
proc labeledListbox2 { parent name
		    {text "Label"} {geometry 10x5} {position left} } {
    global passiveColor activeColor paleBackground
    global labelFont FontWidgets
    if ![info exists labelFont] {
	utilsInit
    }
    set self [selfName $parent $name]
    frame   $self  -background $passiveColor
    label   $self.label -text $text -font $labelFont -background $passiveColor
    lappend FontWidgets $self.label $self.list
    scrollbar $self.yscroll -command "$self.list yview" -orient vertical \
	-background $paleBackground -foreground $passiveColor 	\
	-activeforeground $activeColor
    scrollbar $self.xscroll -command "$self.list xview" -orient horizontal \
	-background $paleBackground -foreground $passiveColor \
	-activeforeground $activeColor
    listbox $self.list -geometry $geometry -yscroll "$self.yscroll set" \
	-xscroll "$self.xscroll set" -font $labelFont \
	-background $paleBackground -selectbackground $activeColor
    pack append $parent $self $position
    pack append $self $self.label {top} $self.yscroll {right filly} $self.xscroll {bottom fillx} $self.list {left expand fill}
    return $self
}
#
# labeledEntry creates an entry that has a label to its left
#
proc labeledEntry { parent name {label "Entry:"} {width 20} {where {left} }} {
    global foregroundColor backgroundColor paleBackground
    global passiveColor activeColor
    global labelFont entryFont FontWidgets
    if ![info exists backgroundColor] {
	utilsInit
    }
    set self [selfName $parent $name]
    # Geometry and Packing
    frame $self -borderwidth 2 -background $backgroundColor -relief raised
    label $self.label -text $label -background $paleBackground \
	-relief flat -font $labelFont -borderwidth 0
    entry $self.entry -width $width  -font $entryFont \
		-relief flat -borderwidth 0 \
		-background $paleBackground \
		-foreground $foregroundColor \
		-selectforeground $passiveColor \
		-selectbackground $activeColor
    lappend FontWidgets $self.label $self.entry
    pack append $parent $self $where
    pack append $self $self.label {left} \
			$self.entry {right fillx expand}

    $self.entry icursor 0

    return $self
}

# commandEntry --
# An entry widget for entering commands
proc commandEntry { parent { width 20 } { where {bottom fillx expand} } } {
    set self [labeledEntry $parent .command "Command:" $width $where]
    bind $self.entry <Return> "eval \[$self.entry get\]"
    return $self
}

#
# Entry with default value remembered in /tmp/file
#
proc defaultGeneric { parent name default } {
    if [file  exists /tmp/$parent/$name] {
	return [exec cat /tmp/$parent/$name]
    } else {
	if {! [file isdirectory /tmp/$parent]} {
	    exec mkdir /tmp/$parent
	}
    }
    exec echo $default > /tmp/$parent/$name
    return [exec cat /tmp/$parent/$name]

}
proc labeledEntryWithDefault { parent name label width default {where {bottom} } } {
    set widget [labeledEntry $parent $name $label $width $where]
    proc default$name { } "return \[defaultGeneric $parent $name \{$default\}\]"
    proc get$name { } "return \[lindex \[$widget.entry get\] 0\]"
    $widget.entry insert 0 [default$name]
    bind $widget.entry <Return> "
	set fileID \[open /tmp/$parent/$name w\]
	puts \$fileID \[get$name\]
	close \$fileID
#	puts stdout \$parent: Remembering $name \[get$name\]\"	
    "
    return $widget
}


#
# feedback
# Create a frame to hold messages, and define a procedure to display them.
# The feedback procedure will be named
# feedback$parent (e.g., feedback.foo)
#

proc feedbackSetup { parent name {width 58} {border 6} } {
    global backgroundColor paleBackground
    global entryFont FontWidgets
    global _feedbackWidget
    if ![info exists backgroundColor] {
	utilsInit
    }
    set self [selfName $parent $name]

    frame $self -borderwidth 2 -background $backgroundColor -relief raised

    entry $self.entry -width $width -background $paleBackground -font $entryFont
    lappend FontWidgets $self.entry
    pack append $self $self.entry {left fillx expand}
    pack append $parent $self {left fillx expand}

    # Define a per-call procedure to allow for multiple feedback widgets
    proc feedback$parent { text } "
	    $self.entry delete 0 end ;
	    $self.entry insert 0 \$text ;
	    "

    # Save the name of the feedback entry for simple clients
    set _feedbackWidget $self.entry

    return $self
}
proc feedback { text } {
    global _feedbackWidget
    if ![info exists _feedbackWidget] {
	puts stderr $text
    } else {
	$_feedbackWidget delete 0 end
	$_feedbackWidget insert 0 $text
	update idletasks
    }
}

#
# notifier
#
proc notifier {name title text {font _default_font_ } } {
    global paleBackground $name backgroundColor
    global entryFont FontWidgets

    if {$font == "_default_font_"} {
	set font $entryFont
    }

    if {[info exists $name] && [expr {[string compare [set $name] 1] == 0}] } { 
	destroy $name
	set $name 0
	return ""
    } else {

	toplevel $name
	set $name 1
	if ![info exists backgroundColor] {
	    utilsInit
	}

	wm title $name $title
    
	buttonFrame $name
    
	packedButton $name.buttons .quit "Dismiss" "destroy $name ; global $name ; set $name 0" left
    
	message $name.msg -aspect 300 -text $text -background $paleBackground -font $entryFont
	lappend FontWidgets $name.msg
	pack append $name $name.msg {top expand}
	return $name
    }
}

#
# unixCommandMessageButton -
#   A button that runs a UNIX command and puts it output in a message widget
#
proc unixCommandMessageButton { parent name label title args} {
    set self [selfName $parent $name]
    set cmd "unixCommandMessage $name \"$title\" "
    foreach a $args {
	set cmd [concat $cmd $a]
    }
    packedButton $parent $name $label $cmd
    return $self
}
#
# unixCommandMessage -
#  Exec a UNIX command and put the output in a message widget
#
proc unixCommandMessage {name title args} {
    toplevel $name

    wm title $name $title

    frame $name.buttons -borderwidth 10 -background \
	    [format "#%02x%02x%02x" 128 128 200]
    pack append $name $name.buttons {top fillx}

    packedButton $name.buttons .quit "Quit" "destroy $name" left

    message $name.msg -aspect 300 -font fixed -text [eval exec $args]
    pack append $name $name.msg {top expand}
    return $name
}
#
# unixCommandListbox -
#  Exec a UNIX command and put the output in a labeledListbox
#
proc unixCommandListbox {name title label args} {
    toplevel $name

    wm title $name $title

    buttonFrame $name

    packedButton $name.buttons .quit "Quit" "destroy $name" left

    labeledListbox $name .dir $label 20x15 left
    foreach i [eval exec $args] {
	$name.dir.list insert end $i
    }
    return $name
}

#####################################################################
# These are additions to the entry widget bindings that rightfully
# belong in tk.tcl, but I don't want folks to have to modify that.
# These add mxedit-like bindings to entry widgets.

# The procedure below is invoked to delete the character to the right
# of the cursor in an entry widget.

proc tk_entryDelRight w {
    set x [$w index insert]
    if {$x != -1} {$w delete $x}
}

# proc to move the cursor in an entry back one character

proc tk_entryBack1char w {
    set x [$w index insert]
    $w icursor [incr x -1]
}

# proc to move the cursor in an entry forward one character

proc tk_entryForw1char w {
    set x [$w index insert]
    $w icursor [incr x +1]
}

# proc to move the cursor in an entry to the end of the line

proc tk_entryEndOfLine w {
    $w icursor end
}

# The procedure below is invoked to backspace over one character
# in an entry widget.  The name of the widget is passed as argument.

proc tk_entryBackspace w {
    set x [expr {[$w index insert] - 1}]
    if {$x != -1} {$w delete $x}
}

# The procedure below is invoked to backspace over one word in an
# entry widget.  The name of the widget is passed as argument.

proc tk_entryBackword w {
    set string [$w get]
    set curs [expr [$w index insert]-1]
    if {$curs < 0} return
    for {set x $curs} {$x > 0} {incr x -1} {
	if {([string first [string index $string $x] " \t"] < 0)
		&& ([string first [string index $string [expr $x-1]] " \t"]
		>= 0)} {
#	    puts stdout "x is $x, string is \"$string\""
	    break
	}
    }
    $w delete $x $curs
}


#
# traceprint
#
proc traceprint { name op oldValue newValue } {
    puts stdout [concat $name " " $op " " $oldValue " " $newValue "\n"]
    return $newValue
}
