###############################################################################
#			TCL With Objects - TWO v1
# 			-------------------------
# 	This is the TCL implementation of TCL With Objects (TWO),
# version 1.  See README for documentation.  See comments in the code
# below for descriptions of global variables and each TCL function if you
# need to modify this code.  Please send comments to aginter@cuug.ab.ca.
# I do follow comp.lang.tcl, but not diligently.
# 
#		     COPYRIGHT & WARRANTEE DISCLAIMER
#		     --------------------------------
# 	I hereby relinquish all copyrights to this software and place it
# into the public domain. 
# 	IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY
# PARTY FOR ERRORS CONTAINED HEREIN OR DIRECT, INDIRECT, SPECIAL,
# INCIDENTAL, OR CONSEQUENTIAL DAMAGES IN CONNECTION WITH THE FURNISHING,
# PERFORMANCE, OR USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
# DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE. 
# 	THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT. 
# THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND
# DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
# UPDATES, ENHANCEMENTS, OR MODIFICATIONS. 
# 
# Andrew Ginter			aginter@cuug.ab.ca
# 
# CONTENTS
#	class	- define a class
#	method	- define a method
#	new	- create an object
#	delete	- destroy an object
#	gensym	- generate short symbol name
#	lassign	- assign list elements to variables
###############################################################################

# GLOBAL VARIABLES
#
# Each class has an entry in the _class array:
#
# _class(className) = 
#    {id 			# Short string identifying the class
#     {instanceVar1 ... varN}	# instance vars defined for class
#     baseClass			# name of the base class, if any
#     {method1 ...N}}		# names of all methods defined for the class
#
# The _methods array is a cache.  It associates method names like "foo"
# with their full name like "className::foo".  The classId may be the
# ID of a derived class, not of the class defining "foo".
#
# _methods(classId,methodName) = [list realClass::methodName]
#
# The _ob array stores instance variables.  The array is indexed by
# a short object ID and an integer variable index.  The index is that
# of the instance variables list in the _class entry for the object.
#
# _ob(id,N) = instanceVarValue
#
# HACK: using the _ob to hold all instance variables would seem to
# prohibit array instance variables.  However:
# 	"upvar a(b) c ; set c(3) d"
# seems to work.  ??  Until I learn that it really doesn't work, I'll
# leave it like this.  If it turns out it really doesn't work, I'll
# have to move instance variables from the _ob array to simple global
# variables like _ob.id,N, or even ${this}.id,N.  But that would
# clutter the global name-space something fierce.  Or maybe the whole
# thing should be re-written in C with all the upvar's going to an
# invisible stack frame like they really ought to.
#
# The _object array associates objects with their ID and their class ID.
# The delete command uses it to 
#
# _object(name) = {objId classId}
#

#---------------------------------------------------------------------------
# Usage: gensym ?last?
#
# Returns a sequence of short, unique ids.  If "last" is specified, the one
# in the sequence following "last" is returned.
#---------------------------------------------------------------------------
if {! [info exists _id]} {set _id 0}
proc gensym {{last {}}} {
    if {$last == {}} {
	global _id
    } else {
	set _id $last
    }
    scan ${_id} %x id
    incr id
    set _id [format %x ${id}]
}

#---------------------------------------------------------------------------
# Usage: class name baseClass {var1 ... varN}
#
# Defines the class "name" with instance variables var1 -> varN.  Updates
# the global _class array.  Creates a procedure whose name is the class.
# The class procedure creates objects which are instances of the class.
#---------------------------------------------------------------------------
proc class {name baseClass varList} {
    global _class

    #
    # Make sure the base class exists
    #
    if {($baseClass != {}) && (! [info exists _class($baseClass)])} {
	error "The specified base class does not exist: $baseClass"
    }

    #
    # Build a list of instance variables.  Each line of the list may contain
    # one or more variable names, interspersed with comments.
    #
    set realVarList {}
    while {$varList != {}} {
	regexp "(\[^\n]*)\n*(.*)" $varList dummy firstLine varList
	for {set firstChar [string index [lindex $firstLine 0] 0]} \
	    {($firstChar != {}) && ($firstChar != {#})} \
	    {set firstLine [lrange $firstLine 1 end]
	     set firstChar [string index [lindex $firstLine 0] 0]} {
	    lappend realVarList [lindex $firstLine 0]
	}
    }
    set classId [gensym]
    set _class($name) [list $classId $realVarList $baseClass {}]

    #
    # Build the class proc
    #
    proc $name {objName args} "uplevel \"new [list $name] \[list \$objName] \$args\""

    #
    # Create the default constructor and destructor & return
    #
    method $name::~$name {} {}
    method $name::$name {} {}
    return $name
}

#---------------------------------------------------------------------------
# Usage: new className objName arg1 ... N
#
# Creates an instance of the given class with the name objName.  Updates the
# global _object array and creates a procedure whose name is the object.
# The procedure has a calling sequence of:
#
#	objName method ?arg1 ... N?
#
# and converts such calls into calls of methods, like:
#
#	className::methodName objName objectId ?arg1 ... N?
#---------------------------------------------------------------------------
proc new {className objName args} {
    global _class _object

    #
    # Complain if the object already exists
    #
    if {[info exists _object($objName)]} {
	error "The object '$objName' already exists"
    }

    #
    # Create a unique ID for the object
    #
    set id [gensym]
    set _object($objName) [list $id $className]

    #
    # Create the method dispatcher for the object.  If it can't find the method
    # in the cache, update the cache by searching the tree the hard way.
    #
    lassign $_class($className) classId
    proc $objName {mn args} "
	global _methods
	if !\[info exists _methods($classId,\$mn)] {
	    set _methods($classId,\$mn) \[_findMethod $objName \$mn $className]
	}
	uplevel \"\$_methods($classId,\$mn) $objName $id \$args\"
    "

    #
    # Call the constructor.
    #
    uplevel "$className::$className $objName $id $args"
    return $objName
}

#---------------------------------------------------------------------------
# Usage: _findMethod methodName className
#
# Returns the fully-qualified method name corresponding to methodName when
# invoked on a member of className.  Updates the method cache with this info
# as well.
#---------------------------------------------------------------------------
proc _findMethod {this methodName className} {
    global _class
    set classId [lindex $_class($className) 0]

    #
    # If the method name already has a class qualifier, search the ancestors
    # for a class with that name, containing the indicated method.
    #
    if {[regexp "(.*)::(.*)" $methodName dummy methodClass methodName]} {
	for {set curClass $className} \
	    {$curClass != {}} \
	    {set curClass [lindex $_class($curClass) 2]} {
	    if {$curClass == $methodClass} {
		if {[lsearch -exact [lindex $_class($curClass) 3] $methodName] >= 0} {
		    return [set _methods($classId,$methodClass::$methodName) \
				[list $methodClass::$methodName]]
		}
		error "The class '$methodClass' has no method '$methodName'"
	    }
	}
	error "The object '$this' is not an instance of '$methodClass'"
    }

    #
    # The method has no class qualifier.  Search the ancestors for a class
    # containing a method with the indicated name.
    #
    for {set curClass $className} \
	{$curClass != {}} \
	{set curClass [lindex $_class($curClass) 2]} {
	if {[lsearch -exact [lindex $_class($curClass) 3] $methodName] > 0} {
	    return [set _methods($classId,$methodName) \
			[list $curClass::$methodName]]
	}
    }
    error "The method '$methodName' is not defined for the object '$this'"
}

#---------------------------------------------------------------------------
# Usage: delete objName
#
# Destroy the object.  Invoke the destructors in derived->base class order
# and turf all the instance variables as you go.  If an error occurs during
# destructor evaluation, propogate the error condition up the stack.
#---------------------------------------------------------------------------
proc delete objName {
    global _id _class _object _ob

    if {[catch {lassign $_object($objName) id className}]} {
	error "There is no object '$objName'"
    }

    #
    # Invoke the destructors, bottom to top.
    #
    set numVars 0
    for {set curClass $className} {$curClass != {}} {set curClass $base} {
	lassign $_class($curClass) classId vars base
	set status [catch {$objName $curClass::~$curClass}]
	if {$status} {
	    return $errorMsg -code $status -errorinfo $errorInfo \
		   -errorcode $errorCode
	}
	incr numVars [llength $vars]
    }

    #
    # Unset all the instance variables
    #
    lassign $_class($className) classId
    for {set i 0} "\$i < $numVars" {incr i} {
	catch {unset _ob($id,$i)}
    }

    #
    # Unset the object identifier
    #
    unset _object($objName)
    return
}

#---------------------------------------------------------------------------
# Usage: method className::methodName {arg1 ... N} body
#
# Defines a method methodName for the class className.  The method definition
# is converted into a proc that looks like:
#
#	proc className::methodName {this, _id, arg1 ... N} {prologue ; body}
#
# The method prologue looks like:
#
#	upvar #0 _ob(classId,objId) var1 ...
#
# That is: it connects local variable names to elements of the global _ob
# array.  Constructors & destructors are magic'ed as well, to do implicit
# base class constructor/destructor calling.
#
# Updates the global _class array to reflect the existence of the new method.
# Flushes the global _methods array in case some object was already using
# an identically named method from a base class and now should start using
# the new method.
#---------------------------------------------------------------------------
proc method {name argList body} {
    global _class _methods

    #
    # Get the class info
    #
    if {! [regexp "(.*)::(.*)" $name dummy className methodName]} {
	error "Unrecognized argument -- should be: method className::methodName argList body"
    }
    if {[catch {lassign $_class($className) classId classVars baseClass methods}]} {
	error "No such class: $className"
    }

    #
    # Define constructors.  If there is a base class, and the first
    # function call in the constructor is not an explicit call to the
    # base class constructor, don't emit the usual invisible call to
    # the base class constructor.  The base class constructor is always
    # called first so constructors are invoked in base->derived class
    # order.
    #
    if {$className == $methodName} {
	set needsBaseConstructor 0
	if {$baseClass != {}} {
	    set needsBaseConstructor 1
	    for {set curBody $body} \
		{$curBody != {}} \
		{regexp "(\[^\n]*)\n*(.*)" $curBody dummy firstLine curBody} {
		if {[string index [lindex $curBody 0] 0] != {#}} {
		    if {([lindex $curBody 0] == {$this}) &&
			([lindex $curBody 1] == "$baseClass::$baseClass")} {
			    set needsBaseConstructor 0
		    }
		    break
		}
	    }
	}
	if {$needsBaseConstructor} {
	    if {$body == {}} {
		set body "\$this [list $baseClass::$baseClass]"
	    } else {
		set body "\$this [list $baseClass::$baseClass] ; $body"
	    }
	}

    #
    # Define normal method procs & destructors
    #
    } else {
	if {[string index $methodName 0] == {~}} {
	    if {[llength $argList]} {
		error "No parameters may be passed to destructors: $name"
	    }
	}
    }

    #
    # Get the list of variables to be upvar'ed in the method prologue.
    # Flag the method args so none of them get clobbered by the 'upvar'
    # in the prologue code.
    #
    foreach var $argList {
	set vars($var) 1
    }
    set fullVarList {}
    for {set curClass $className} {$curClass != {}} \
	{set curClass [lindex $_class($curClass) 2]} {
	foreach var [lindex $_class($curClass) 1] {
	    if {[info exists vars($var)]} {
		set realName $curClass::$var
	    } else {
		set realName $var
	    }
	    set vars($realName) 1
	    set fullVarList "[list $realName] $fullVarList"
	}
    }

    #
    # Build the method prolog
    #
    set prologue {}
    set varId 0
    foreach var $fullVarList {
	append prologue { _ob($_id,} "[set varId [gensym $varId]]) " [list $var]
    }
    if {$prologue != {}} {
	set prologue "upvar #0$prologue"
    }

    #
    # Prepend the prologue to the body
    #
    if {($prologue != {}) && ($body != {})} {
	set body "$prologue ; $body"
    }
    proc $name "this _id $argList" $body

    #
    # Update the class definition.  Don't goof things up if
    # this is a redefinition of an existing method.
    #
    if {[lsearch -exact $methods $methodName] < 0} {
	lappend methods $methodName
	set _class($className) [list $classId $classVars $baseClass $methods]
    }

    #
    # Invalidate all method cache entries with the same $methodName as
    # the method just defined.
    #
    foreach entry [array names _methods *::$methodName] {
	catch {unset _methods($entry)}
    }
    return $name
}

#---------------------------------------------------------------------------
# Usage: lassign list var1 ... varN
#
# This is the TclX lassign done the hard way, for you poor suckers without
# TclX installed.  It assigns the i'th element of 'list' to 'vari' and 
# returns the unassigned portion of the list.
#---------------------------------------------------------------------------
if {[info commands lassign] == {}} {
    proc lassign {list args} {
	for {set i 0} "\$i < [llength $args]" {incr i} {
	    set [lindex $args $i] [lindex $list $i]
	}
	lrange $list $i end
    }
}
