/*-
    pickUtils.c - utilities for use in picking
*/

/*
    terms for this file

    box - a rectangle described by two corners e.g.
	box =   xA yA xB yB

    area - a rectangle described by four boundary lines  e.g.
	area =  leftBound rightBound bottomBound topBound

    rect - a rectangle described by a point and a width and height e.g.
	rect =  x y w h
*/
#include <pickUtils.h>


typedef struct {
    int left;
    int right;
    int bottom;
    int top;
} area;

/*-

    left of area 	|= bit1
    above area   	|= bit4
    right of area	|= bit2
    below area		|= bit3

    ------------------
    1001 | 1000 | 1010
    ------------------
    0001 | 0000 | 0010
    ------------------
    0101 | 0100 | 0110
    ------------------

*/

/* 
    Return the area code attributes for point x y with respsect to area 'a'.
    The attributes are encoded as bits within an integer.

    The codes are shown above.
*/
static int 
areaCode(x, y, a)
    int x,y;
    area *a;
{
    code r;
    /* set left/right bits */
    if (x<a->left)
	r = 01;
    else if (x>a->right)
	r = 02;
    else
	r = 00;

    /* set top/bottom bits */
    if (y<a->bottom)
	r |= 04;
    else if (y>a->top)
	r |= 010;

    /* */
    return r;
}

int
linePartOfArea(x1,y1,x2,y2,a)
    int x1,y1,x2,y2;
    area *a;
{
    int t;
#   define swap(a,b) t=a,a=b,b=t

    int c1 = areaCode(x1,y1,a);
    int c2 = areaCode(x2,y2,a);

    /*
     * If the area code for point point1 and point2 are 0
     * the line lies completely within the aperature area
     * so we return true. 
     *
     * if not we have to check the line further...
     */

    while (c1 || c2) {
	/*
	 * line is not completely inside the area so
	 * At least one of the endpoints is outside the area
	 */
	if (c1&c2) {
	    /*
	     * A most trivial rejection! 
	     * both points are to the left|top|right|bottom of
	     * the area. 
	     *
	     * Return that the line is outside of the area
	     */
	    return 0;
	}
	/*
	 * Bummer. 
	 */
	if (c1==0) {
	    /* 
	     * c1 is in the area, c2 is not
	     * so switch the points we do this because we always
	     * reel in c1 towards c2
	     */
	    swap(c1,c2);
	    swap(x1,x2);
	    swap(y1,y2);
	}
	/*
	 * The remaining code clips out the portion of the line outside
	 * of the area.
	 */
	if (c1&01) {
	    /*
	     * point is left of area, reel it in (along the line segment) 
	     * from the left to right putting it on the left edge of the area.
	     */
	    x1 = a->left;
	    y1 = y1 + (y2-y1) * (a->left-x1) / (x2-x1);
	} else if (c1&02) {
	    /*
	     * point is right of area, reel it in (along the line segment) 
	     * from right to left putting it on the right edge of the area.
	     */
	    x1 = a->right;
	    y1 = y1 + (y2-y1) * (a->right-x1) / (x2-x1);
	} else if (c1&04) {
	    /*
	     * point is below area, reel it in (along the line segment) 
	     * from below upwards putting it on the bottom edge of the area.
	     */
	     y1 = a->bottom;
	     x1 = x1 + (x2-x1) * (a->bottom-y1) / (y2-y1);
	} else if (c1&010) {
	    /*
	     * point is above area, reel it in (along the line segment) 
	     * from above downwards putting it on the top edge of the area.
	     */
	     y1 = a->top;
	     x1 = x1 + (x2-x1) * (a->top-y1) / (y2-y1);
	}
	/*
	 * Now make another try. Before we do though
	 * Since x1,y1 has moved we need to recompute its area code 'c1'
	 */
	c1 = areaCode(x1,y1,a);
    }
    /*
     * The segment is now completely within the area.
     */
    return 1;
#   undef swap
}
