/*
 * TmWidget.c --
 *	This module contains the main set of functions
 *	common to all widget types. ie it implements the
 *	Tm Core widget stuff.
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tm.h"
#include "tmFuncs.h"
#include <Xm/List.h>

/*
 *--------------------------------------------------------------
 *
 * Tm_ParentWidgetFromPath --
 *
 *	Given a Tm widget pathname, finds the parent Xt widget.
 *
 * Results:
 *
 *	returns the Xt parent
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Widget Tm_ParentWidgetFromPath (interp, pathName)
    Tcl_Interp *interp;
    char *pathName;
{
    char *p;
    int numChars;
    Tm_Widget *info;
    Tcl_HashEntry *hPtr;

    /*
     * Strip the parent's name out of pathName (it's everything up
     * to the last dot).  There are two tricky parts: (a) must
     * copy the parent's name somewhere else to avoid modifying
     * the pathName string (for large names, space for the copy
     * will have to be malloc'ed);  (b) must special-case the
     * situation where the parent is ".".
     */

    p = strrchr(pathName, '.');
    if (p == NULL) {
        Tcl_AppendResult(interp, "bad window path name \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }

    numChars = p-pathName;

    p = (char *) XtMalloc((unsigned) (numChars+1));
    if (numChars == 0) {
	*p = '.';
	p[1] = '\0';
    } else {
	strncpy(p, pathName, numChars);
	p[numChars] = '\0';
    }

    hPtr = Tcl_FindHashEntry(&WidgetTable, p);
    if (hPtr == NULL) {
        Tcl_AppendResult(interp, "no such widget \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }
    info = (Tm_Widget *) Tcl_GetHashValue(hPtr);

    XtFree(p);
    return (info->widget);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_WidgetInfoFromPath --
 *
 *	looks up the hash table to find the info about the widget
 *
 * Results:
 *
 *	returns the widget info record.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Tm_Widget * 
Tm_WidgetInfoFromPath (interp, pathName)
    Tcl_Interp *interp;
    char *pathName;
{
#define FIXED_SPACE 5
    Tm_Widget *info;
    Tcl_HashEntry *hPtr;


    hPtr = Tcl_FindHashEntry(&WidgetTable, pathName);
    if (hPtr == NULL) {
        Tcl_AppendResult(interp, "no such widget \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }
    info = (Tm_Widget *) Tcl_GetHashValue(hPtr);

    return (info);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_WidgetCallbackHandler --
 *
 *	nearly all callbacks are vectored through here.
 *	It calls the appropriate callback with right
 *	Tcl command
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_WidgetCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    Tcl_Interp *interp;
    char *command;
    char *msg;

    interp = c_data->widget_info->interp;
#   ifdef DEBUG
    fprintf(stderr, "%s\n", (char *) c_data->callback_func);
#   endif
    command = Tm_ExpandPercents(c_data->widget_info, call_data, 
		(char *) c_data->callback_func);
#   ifdef DEBUG
    fprintf(stderr, "%% expanded command: %s\n", command);
#   endif

    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
	msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
	if (msg == NULL) {
	    msg = interp->result;
	}
	XtAppWarningMsg(XtWidgetToApplicationContext(w),
		"TclError", "TclError", "TclError", msg, NULL, 0);
    }
}


/*
 *--------------------------------------------------------------
 *
 * Tm_DestroyWidgetCallbackHandler --
 *
 *	nearly all callbacks are vectored through here.
 *	It calls the appropriate callback with right
 *	Tcl command
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_DestroyWidgetHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    Tcl_Interp *interp;
    char *path;

    interp = c_data->widget_info->interp;
    path = c_data->widget_info->pathName;

    Tcl_DeleteCommand(interp, path);
    XtFree(path);
    ckfree(c_data->widget_info);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_TextVerifyCallbackHandler --
 *
 *	special case callback handler for Text Verify callbacks.
 *	It calls the appropriate callback with right
 *	Tcl command, then sets fields as needed by Text
 *	(or will do)
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_TextVerifyCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    XmTextVerifyCallbackStruct *verify_data = 
			(XmTextVerifyCallbackStruct *) call_data;
    Tcl_Interp *interp;
    char *path;
    char *msg;
    char *command;
    int doit;
    XmTextPosition startPos, endPos;
    char *ptr;
    int length;
    char buf_doit[128];
    char buf_startPos[128];
    char buf_endPos[128];
    char buf_length[128];
    char buf[128];
    char *buf_ptr;

    interp = c_data->widget_info->interp;
    path = c_data->widget_info->pathName;

/* in here we have to set tcl vbls to the values of the callback fields
   and afterwards get their values and set them in the callback data
*/
    Tcl_SetVar(interp, TM_TEXT_DOIT, "true", TCL_GLOBAL_ONLY);

    sprintf(buf_startPos, "%ld", verify_data->startPos);
    Tcl_SetVar(interp, TM_TEXT_STARTPOS, buf_startPos, TCL_GLOBAL_ONLY);

    sprintf(buf_endPos, "%ld", verify_data->endPos);
    Tcl_SetVar(interp, TM_TEXT_ENDPOS, buf_endPos, TCL_GLOBAL_ONLY);

    length = verify_data->text->length;
    buf_ptr = XtMalloc(length + 1);
    strncpy(buf_ptr, verify_data->text->ptr, length);
    buf_ptr[length] = '\0';
    Tcl_SetVar(interp, TM_TEXT_PTR, buf_ptr, TCL_GLOBAL_ONLY);

    sprintf(buf_length, "%d", length);
    Tcl_SetVar(interp, TM_TEXT_LENGTH, buf_length, TCL_GLOBAL_ONLY);

    command = Tm_ExpandPercents(c_data->widget_info, call_data, 
		(char *) c_data->callback_func);
    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	return;
    }

    /* now set results back into callback struct for Text */
    msg = Tcl_GetVar(interp, TM_TEXT_DOIT, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_doit) != 0) {
        if (Tcl_GetBoolean(interp, msg, &doit) == TCL_ERROR) {
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
            return;
        }
        verify_data->doit = doit;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_STARTPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_startPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        startPos = strtol(msg, NULL, 0);
        verify_data->startPos = startPos;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_ENDPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_endPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        endPos = strtol(msg, NULL, 0);
        verify_data->endPos = endPos;
    }
    msg = Tcl_GetVar(interp, TM_TEXT_PTR, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_ptr) != 0) {
	XtFree(verify_data->text->ptr);
	XtFree(buf_ptr);
        verify_data->text->ptr = XtNewString(msg);
    }
    msg = Tcl_GetVar(interp, TM_TEXT_LENGTH, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_length) != 0) {
        if (Tcl_GetInt(interp, msg, &length) == TCL_ERROR) {
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
            return;
        }
        verify_data->text->length = length;
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertValueFromString(w, resources, num_resources,
		resource, orig_value, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n;
    XrmValue from, converted;
    XtArgVal tmpval;

#   ifdef DEBUG
    fprintf(stderr, "converting from string \"%s\" of type %s\n", orig_value, resource);
#   endif
    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    /* special case - same type */
    if (strcmp(resources->resource_type, XtRString) == 0) {
	*new_value =  orig_value;
	return TRUE;
    }
    from.size = strlen(orig_value) + 1;
    from.addr = orig_value;
    converted.size = resources->resource_size;
    converted.addr = XtMalloc(converted.size);
    if (!XtConvertAndStore(w, XtRString, &from,
		resources->resource_type, &converted)) {
        fprintf(stderr, "No converter %s to %s\n",
		XtRString, resources->resource_type);
	return FALSE;
    }
    *new_value = 0;
    _XtCopyToArg(converted.addr, new_value, converted.size);
/*
    bzero(new_value, sizeof(XtArgVal));
    bcopy(converted.addr, new_value, converted.size);
    *new_value = *(XtArgVal *) (converted.addr);
*/
    
    return TRUE;
}
		

/*
 *--------------------------------------------------------------
 *
 * Tm_MaybeSetStringTableSize --
 *
 *	Hack for Motif XmStringTables.
 *	Motif *sometimes* NULL terminates its XmStringTables
 *	most times doesnt.
 *	XmStringGet...() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	_XmStringIsXmString() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	i.e. there is no internal way of finding the size of one
 *	of these things. So we have to check if we have one and
 *	then look up the resource that tells us the size :-(
 *
 * Results:
 *
 * Side effects:
 *
 *	Sets global vbl Tm_XmStringTableSize
 *--------------------------------------------------------------
 */

static void
Tm_MaybeSetStringTableSize(w, resource)
    Widget w;
    String resource;
{
    extern int Tm_XmStringTableSize;
    WidgetClass class;
    Arg arg;
    char *size_resource;

    class = XtClass(w);
    if (class == xmListWidgetClass) {
	if (strcmp (resource, XmNitems) == 0) {
	    size_resource = XmNitemCount;
	} else
	if (strcmp (resource, XmNselectedItems) == 0) {
	    size_resource = XmNselectedItemCount;
	} else
	return;
    } else
    return;

    /* Sigh, we have a StringTable, find its size */
    XtSetArg(arg, size_resource, &Tm_XmStringTableSize);
    XtGetValues(w, &arg, 1);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertValueToString(w, resources, num_resources,
		resource, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char **new_value;
{
    int n;
    XrmValue from, converted;
    Arg args[1];

    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    Tm_MaybeSetStringTableSize(w, resource);

    from.size = resources->resource_size;
    from.addr = (XtArgVal *) XtMalloc(from.size);
    XtSetArg(args[0], resource, from.addr);
    XtGetValues(w, args, 1);
    
    /* special case - same type */
    if (strcmp(resources->resource_type, XtRString) == 0) {
	*new_value =  *(char **) from.addr;
	return TRUE;
    }
    converted.addr = NULL;
    if (!XtConvertAndStore(w, resources->resource_type, &from,
			XtRString, &converted)) {
	fprintf(stderr, "No converter %s to %s\n",
			resources->resource_type, XtRString);
	XtFree(from.addr);
	return FALSE;
    }
    bzero(new_value, sizeof(XtArgVal));
    bcopy(converted.addr, new_value, converted.size);
    
    XtFree(from.addr);
    return TRUE;
}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueFromString(w, sec_resources, num_sec_resources,
		resource, orig_value, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueFromString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    orig_value, new_value)) {
	    return True;
	}
    }
    return False;
}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueToString(w, sec_resources, num_sec_resources,
		resource, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char **new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueToString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    new_value)) {
	    return True;
	}
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_FreeSecondaryResources --
 *
 *	Motif gives us a copy of these, so we have to tidy up
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static void
Tm_FreeSecondaryResources(sec_resources, num_sec_resources)
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
{
    int n;

    for (n = 0; n < num_sec_resources; n++) {
	XtFree((char *) sec_resources[n]->resources);
	XtFree((char *) sec_resources[n]);
    }

    if (num_sec_resources > 0) {
        XtFree((char *) sec_resources);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tm_SetValues --
 *
 *	set resource values on a widget
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_SetValues(pathName, interp, w, parent, class, argv, argc, args, num_args)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    Widget parent;
    WidgetClass class;
    char **argv;
    int argc;
    Arg args[];
    int *num_args;
{
    XtResourceList resources, constraint_resources;
    XmSecondaryResourceData *sec_resources;
    int num_resources;
    int num_constraint_resources;
    int num_sec_resources;
    int num_values = 0;
    XrmValue from, converted;
    XtArgVal new_value;
    char *resource;
    char *value;
    Tm_ClientData *client_data;

    if (argc > TM_MAXARGS*2) {
	return;
    }

    XtGetResourceList(class, &resources, &num_resources);
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    /* skip this one silently */
	    argc -= 2; argv += 2;
	    continue;
	}
	resource = argv[0]+1;
        value = argv[1];

        if (Tm_ConvertValueFromString(w, resources, num_resources,
		resource, value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else
        if (Tm_ConvertValueFromString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else 
        if (Tm_ConvertSubValueFromString(w, sec_resources, 
	            num_sec_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else {
	    fprintf(stderr, "Conversion from String to %s failed\n",
				resource);
	}
	argc -= 2;
	argv += 2;
    }

    *num_args = num_values;

/*
    Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
*/
    /* memory leak: should XtFree all new_values stored in args[] */
}

/*
 *--------------------------------------------------------------
 *
 * Tm_GetValues --
 *
 *	get the resource values out of a widget
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_GetValues(pathName, interp, w, class, argv, argc)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
    char **argv;
    int argc;
{
    XtResourceList resources, constraint_resources;
    XmSecondaryResourceData *sec_resources;
    int num_resources;
    int num_constraint_resources;
    int num_sec_resources;
    int num_values = 0;
    XrmValue from, converted;
    char *new_value;
#   define MAXARGS 100
    char *resource;
    Tm_ClientData *client_data;
    Widget parent;

    if (argc > MAXARGS*2) {
	return;
    }

    parent = XtParent(w);
    XtGetResourceList(class, &resources, &num_resources);

    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
  
    if (XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    /* skip this one silently */
	    fprintf(stderr, "Skipping argument %s\n", argv[0]);
	    argc -= 2; argv += 2;
	    continue;
	}
	resource = argv[0]+1;

        if (Tm_ConvertValueToString(w, resources, num_resources,
		resource, &new_value)) {
	    /* store new_value in variable in argv[1] now */
#	    ifdef DEBUG
	    fprintf(stderr, "Got value: %s\n", (char *) new_value);
#	    endif
/*
	    Tcl_SetVar(interp, argv[1], (char *) new_value, TCL_GLOBAL_ONLY);
*/
	    Tcl_SetVar(interp, argv[1], (char *) new_value, 0);
	} else
        if (Tm_ConvertValueToString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    Tcl_SetVar(interp, argv[1], (char *) new_value, TCL_GLOBAL_ONLY);
	} else 
        if (Tm_ConvertSubValueToString(w, sec_resources, 
	            num_sec_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    Tcl_SetVar(interp, argv[1], (char *) new_value, TCL_GLOBAL_ONLY);
	} else {
	    fprintf(stderr, "Conversion from %s to String failed\n",
					resource);
	}
	argc -= 2;
	argv += 2;
    }
    Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
}
