########################################################################
# housekeeping
########################################################################

use v6.d;
unit module ProcStats:ver<0.0.3>:auth<CPAN:lembark>;

########################################################################
# subroutines
########################################################################

sub dump-rusage
(
    Bool()      :$final = False  ,
    Bool()      :$force = False  ,
    Stringy()   :$label = $final ?? 'Final' !! ''
)
is export( :DEFAULT )
{
    use nqp;

    constant KEYZ =
    <
        utime   stime    maxrss  ixrss
        idrss   isrss    minflt  majflt
        nswap   inblock  oublock msgsnd
        msgrcv  nsignals nvcsw   nivcsw
    >;  

    state %last is default(-1);

    nqp::getrusage( my int @sample );

    my %sample      = KEYZ Z=> @sample;
    state %first    = %sample;

    # report stats from initial pass on final output.

    %last       = %first if $final;

    # catch: stime varies each cycle.
    # fix: is buffer it and check the rest.

    my $stime   = %sample<stime> :delete;

    for
        %sample.grep( { $force || .value != %last{.key} } ).sort
        -> $diff
    {
        sub align-keys (Pair $p) 
        {
            note
            sprintf( '%-*s : %s', once {KEYZ».chars.max}, $p.key, $p.value );
            return
        }

        FIRST
        {
            # output the header once, first in
            # each cycle through the for-loop.
            # avoids writing a header without
            # any real data.
            #
            # this is where stime is output if
            # anything changed.

            note '---';
            align-keys ( sample => $++  );
            align-keys ( :$label        ) if $label;
            align-keys ( :$stime        );
        }

        align-keys $diff;
    }

    %last = %sample;
}

=finish

=begin pod

=head1 NAME

ProcStats - dump rusage process statistics with optional label.

=head1 SYNOPSIS

    # print rusage output to stderr.
    # "sample" is incremented with each output.
    #
    # first pass ("sample 0") dumps all stats.
    # successive ones only list changes.
    #
    # system time (stime) invariably increases due to 
    # rusage call and is only output if another field
    # causes output.
    #
    # format is YAML hash with one document per sample.


    dump-rusage( label => "$*PROGRAM" );

    ---
    sample   : 0
    label    : rand-dictonary
    stime    : 724655
    idrss    : 129224
    inblock  : 0
    isrss    : 0
    ixrss    : 39741
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 32466
    nvcsw    : 0
    oublock  : 0
    utime    : 0

    # force output even if there are no changes.

    dump-rusage( label => 'Initial users', :force );
    ---
    sample   : 1
    label    : Initial users
    stime    : 748975
    idrss    : 129620
    nswap    : 32639

    # write output only if values other than stime change.

    dump-rusage;

    ---
    sample   : 2
    stime    : 769168
    idrss    : 129864
    nswap    : 32732

    # add label "Final", print differences with 
    # sample #0 values (vs. previous one), default
    # label to 'Final'.

    dump-rusage( :final, :force );

    ---
    sample   : 92
    label    : Final
    stime    : 126712
    idrss    : 136936
    inblock  : 0
    isrss    : 0
    ixrss    : 70682
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 34532
    nvcsw    : 0
    oublock  : 0
    utime    : 1

=head1 DESCRIPTION

=head2 Notes


=head1 SEE ALSO

=item getrusage(3) getrusage(3p)

getrusage is POSIX.

    struct rusage 
    {
       struct timeval ru_utime; /* user CPU time used */
       struct timeval ru_stime; /* system CPU time used */
       long   ru_maxrss;        /* maximum resident set size */
       long   ru_ixrss;         /* integral shared memory size */
       long   ru_idrss;         /* integral unshared data size */
       long   ru_isrss;         /* integral unshared stack size */
       long   ru_minflt;        /* page reclaims (soft page faults) */
       long   ru_majflt;        /* page faults (hard page faults) */
       long   ru_nswap;         /* swaps */
       long   ru_inblock;       /* block input operations */
       long   ru_oublock;       /* block output operations */
       long   ru_msgsnd;        /* IPC messages sent */
       long   ru_msgrcv;        /* IPC messages received */
       long   ru_nsignals;      /* signals received */
       long   ru_nvcsw;         /* voluntary context switches */
       long   ru_nivcsw;        /* involuntary context switches */
    };

