signature SORT_ARG =
  sig
     type entry
     val gt : entry * entry -> bool
  end

signature RADIX_SORT_ARG =
  sig
    type entry
    val maxBit : int
    val fetchBit : int -> entry -> bool
  end

signature SORT =
  sig
     type entry
     val sort : entry list -> entry list
  end

signature EQUIV_ARG =
   sig
     type entry
     val gt : entry * entry -> bool
     val eq : entry * entry -> bool
   end

signature EQUIV =
  sig
     type entry

     (* equivalences: take a list of entries and divides them into
        equivalence classes numbered 0 to n-1.

        It returns a triple consisting of:

	  * the number of equivalence classes
          * a list which maps each original entry to an equivalence
            class.  The nth entry in this list gives the equivalence
            class for the nth entry in the original entry list.
          * a list which maps equivalence classes to some representative
            element.  The nth entry in this list is an element from the
            nth equivalence class
      *)

     val equivalences : entry list -> (int * int list * entry list)
  end

(* An O(n^2) insertion sort routine *)

functor InsertSortFun(A : SORT_ARG) : SORT =
  struct

       type entry = A.entry
       val sort = 
          let fun insert (a,nil) = [a]
                | insert (a, l as h :: t) = 
                        if A.gt(h,a) then a::l else h :: insert (a,t)
	  in fn l => fold insert l nil
          end
   end

(* An O(n lg n) merge sort routine *)

functor MergeSortFun(A : SORT_ARG) : SORT =
  struct

      type entry = A.entry

      (* sort: an O(n lg n) merge sort routine.  We use insert sorting
         to create lists of 5 elements in length, and then repeatedly
         merge these lists in passes until only one list is left.

         After the nth pass, each lists has approximating 5 * 2 ^ n elements.
         Each pass takes O(n) time, and there are at most lg n passes.
        *)

      fun sort nil = nil
        | sort l =
             let fun insert (a,nil) = [a]
                   | insert (a, l as h :: t) = 
                          if A.gt(h,a) then a::l else h :: insert (a,t)

                 fun insertSort l = fold insert l nil

                 fun make5List (a :: b :: c :: d :: e :: rest) =
                      insertSort (a :: b :: c :: d :: e :: nil) ::
                         make5List rest
                   | make5List l = [insertSort l]

		 (* merge: merge two lists *)

                 fun merge (l as a::at,r as b::bt) =
                       if A.gt(a,b)
                       then b :: merge(l,bt)
                       else a :: merge(at,r)
                   | merge (l,nil) = l
                   | merge (nil,r) = r

                 (* scan: merge pairs of lists on a list of lists.
                    Reduces the number of lists by about 1/2 *)

                 fun scan (a :: b :: rest) = merge(a,b) :: scan rest
                   | scan l = l

	         (* loop: calls scan on a list of lists until only
                    one list is left.  It terminates only if the list of
                    lists is nonempty.  (The pattern match for sort
                    ensures this.) *)

                 fun loop (a :: nil) = a
                   | loop l = loop (scan l)

              in loop (make5List l)
              end
   end

functor QuickSortFun(A : SORT_ARG) : SORT =
  struct
     type entry = A.entry

     structure InsertSort = InsertSortFun(val gt = A.gt
				    type entry = A.entry)

     val sort = 
        let val medianOfThree = fn (a,b,c) =>
              let val (s1,s2) = if A.gt(a,b) then (a,b) else (b,a)
              in if A.gt(c,s1) then s1
                 else if A.gt(c,s2) then c
	         else s2
              end

	   exception findThreeFails

           val findThree =
              fn (l as h :: _, size) =>
 	         let fun findEnd (a :: nil) = a
                       | findEnd (a :: r) = findEnd r
		       | findEnd nil = raise findThreeFails
                     fun loop (b :: r, 0) =
			  (h,b,findEnd r)
  		       | loop (b :: r, size) = loop(r,size-1)
	               | loop (nil,_) = raise findThreeFails
                in loop(l,size div  2)
 		end
               | _ => raise findThreeFails

          val partition = fn (key, l) => 
             let fun loop (h :: t, left, right,leftSize) =
		   if A.gt(h,key)
		      then loop(t,left, h :: right,leftSize)
		      else loop(t,h :: left, right, leftSize+1)
                   | loop (nil,left,right,leftSize) = (left,right,leftSize)
             in loop(l,nil,nil,0)
             end

          val magicSize = 9

         fun qsort(l,size) =
	    if size <= magicSize then InsertSort.sort l
	    else let val partitionElement = medianOfThree(findThree(l,size))
		     val (left,right,leftSize) = partition(partitionElement,l)
                    in qsort(left,leftSize) @ qsort(right,size-leftSize)
                    end
    in fn l => qsort(l,length l)
    end
end

(* an O(n) integer sorting routine *)
	    
functor RadixSortFun(A : RADIX_SORT_ARG) : SORT =
  struct

    type entry = A.entry

    val sort = fn l =>
       let val partition = fn n =>
	     let val getBit = A.fetchBit n
	         fun partition (nil,left,right) = (left,right)
                   | partition (h :: t,left,right) =
		       if getBit h
			 then partition(t,left,h :: right)
			 else partition(t,h :: left, right)
	     in fn l => partition(l,nil,nil)
  	     end
	   fun radixsort(position, nil) = nil
	     | radixsort(position, l as _ :: nil) = l
 	     | radixsort (position, l) =
		if position >= 0 then
		    let val (left,right) = partition position l
		    in radixsort(position-1,left) @ radixsort(position-1,right)
		    end
	        else l
      in radixsort(A.maxBit, l)
      end
  end

(* an O(n lg n) routine for placing items in equivalence classes *)

functor EquivFun(A : EQUIV_ARG) : EQUIV =
   struct

      (* Our algorithm for finding equivalence class is simple.  The basic
         idea (omitting details) is to sort the entries and place all
         duplicates entries in the same equivalence class.

         Let the original entry list be E.  We map E to a list of a pairs
         consisting of the entry and its position in E, where the positions
         are numbered 0 to n-1.  Call this list of pairs EP.

         We then sort EP on the original entries.  The second elements in the
         pairs now specify a permutation that will return us to EP.

         We then scan the sorted list to create a list of representative
         entries (call this R), a list of integers which permutes the sorted
         list back to the original list (call this P), and a list of integers
          which gives the equivalence class for the nth entry in the sorted 
         list (call this SE)

         We then return the length of R, R, and the list that results from
         permuting SE by P.
     *)	

       type entry = A.entry
             
       val gt = fn ((a,_),(b,_)) => A.gt(a,b)

       structure Sort = MergeSortFun(type entry = A.entry * int
     				     val gt = gt)
       val assignIndex =
          fn l =>
             let fun loop (index,nil) = nil
                   | loop (index,h :: t) = (h,index) :: loop(index+1,t)
             in loop (0,l)
             end 
    
       val createEquivalences =
          let fun loop ((e,_) :: t, prev, class, R , SE) =
                  if A.eq(e,prev)
		    then loop(t,e,class,R, class :: SE)
 		    else loop(t,e,class+1,e :: R, (class + 1) :: SE)
                | loop (nil,_,_,R,SE) = (rev R, rev SE)
          in fn nil => (nil,nil)
               | (e,_) :: t => loop(t, e, 0, [e],[0])
          end

       val inversePermute = fn permutation =>
              fn nil => nil
               | l as h :: _ =>
                   let val result = Array.array(length l,h)
                       fun loop (elem :: r, dest :: s) =
			     (Array.update(result,dest,elem); loop(r,s))
                         | loop _ = ()
                       fun listofarray i =
			  if i < Array.length result then 
				(Array.sub(result,i)) :: listofarray (i+1)
                          else nil
                    in loop (l,permutation); listofarray 0
		    end

       val makePermutation = map (fn (_,b) => b)

       val equivalences = fn l =>
	   let val EP = assignIndex l
	       val sorted = Sort.sort EP
               val P = makePermutation sorted
               val (R, SE) = createEquivalences sorted
            in (length R, inversePermute P SE, R)
            end
end
(*
structure Int = 
  struct
    type entry = int
    val gt = Integer.>
    val eq = (op =) : int * int -> bool
  end

structure IntSort = SortFun(Int)
structure IntEquiv = EquivFun(Int)

fun makeList i = if i >=0 then i :: (makeList (i-1)) else nil
*) 


structure Int = 
  struct
    type entry = int
    val gt = Integer.>
    val eq = (op =) : int * int -> bool
  end

structure QuickSort = QuickSortFun(Int)

structure RadixInt =
   struct
     type entry =  int
     val maxBit = 30
     val fetchBit = fn n =>
	    if n = 30 then 
		fn x => x > 0
	    else 
	        let val mask = Bits.lshift(1,n)
	        in fn x => Bits.andb(mask,x) <> 0
		end
   end

structure RadixSort = RadixSortFun(RadixInt)

structure Test =
struct
    fun makeList i = if i >=0 then i :: (makeList (i-1)) else nil
    val test = makeList 50
    val say = outputc std_out
    val saynum = fn (n : int) => say (makestring n ^ " ")
    val _ = map saynum (QuickSort.sort test)
    val _ = say "\n"
    val _ = map saynum (RadixSort.sort test)
    val _ = say "\n"
end   
