(* Copyright (c) 1991 by Carnegie Mellon University *)

(* machine.sig
 *
 * Signature for a structure which generates code for a C code machine. 
 *
 * Author: David Tarditi
 *         Carnegie Mellon University
 *         tarditi@cs.cmu.edu
 *
 * General
 *   A structure matching this signature should provide abstract types for
 *   constructing instructions for a C code machine.  The C code machine is
 *   implemented by translating these instructions directly to C.
 *  
 *   This signature is derived from cmachine.sig.
 *  
 *   A program consists of function definitions and declarations.  A function
 *   definition is a sequence of labelled statements.  Only the first labelled
 *   statement may be jumped to from other functions.  The other labels
 *   can only be jumped to from within a function. A statement is a
 *   side-effecting machine instruction.   Statements are constructed from
 *   effective addresses (EA), integers, and other statements.  The other
 *   statements represent continuation statements.
 *  
 *   A declaration declares static constants located in memory.  These
 *   constants can be strings or floating point numbers.  We could add record
 *   constants if the CPS language used by SML/NJ had them.
 *  
 *   Our effectives addresses consist of immediate integers, registers,
 *   function names (labels), string locations, and real locations.
 *  
 *   The scope of the first labelled statement in a function can have global
 *   or local scoping.  It should have global scoping if the function is an
 *   entry point for the program.
 *  
 *   The structure should provide functions to print programs, print C
 *   declarations for the beginning of files, and association
 *   lists for entry points.
 *  
 *   The structure should provide flags for controlling optimizations and
 *   enabling debugging
 *
 * Operational Semantics
 *    The functions presented below can be regarded as constructors for
 * an algebraic datatype.  We present an operational semantics which
 * any C program produced from an object of this algebra should match.
 *
 *    The goal of this semantics is to serve as a reference guide for
 * implementors and maintainers, so we leave certain portions incomplete
 * in the interest of readability.
 *
 * Our semantic objects consist of:
 *
 *       programs   A program is a list of functions and a list of
 *                  constant definitions
 *       registers
 *       locations, integers, floating point numbers, strings, records
 *       access paths
 *       a register store 
 *       a conventional store
 *       statements
 *       effective addresses (EA)
 *       labels 
 *
 *   The types of semantic objects are:
 *       type register
 *       type loc         locations
 *       type integer
 *       type fp          floating point values
 *       type records
 *       type string
 *       type EA = register + loc
 *       type regstore : register -> loc + integer
 *       type convstore : loc -> loc + integer + fp + string + records + label
 *
 *  Our semantics function M will have the type:
 *
 *        program -> label -> regstore -> convstore -> (regstore * convstore)
 *
 *  We will not define M, but we will define the semantics of statements.
 *
 *  In the interest of explaining the semantics of statements succinctly,
 *  we assume side-effecting functions defined on the various stores
 *  as follows:
 *
 *         A <-B : The register store maps register A to B, and leaves
 *         the mapping for all other registers the same.  A must be a
 *         register.  B must be an integer or location
 *
 *         contents(A): Fetch the value to which register A is mapped.  A must
 *         be a register
 *
 *         deref(A): Fetch the value to which location A is mapped by
 *         the conventional store.  A must be a location
 * 
 *         A := B:  Change the conventional store to map location A to
 *         B.  A must be a location.  B must be an integer, label, location,
 *         floating point, string, or record.
 *           
 *     move (a,b,s) = b <- contents(a)
 *
 *     jmp [ea1,ea2,..,ean] =
 *	     jump to ea1 with arg 1 in ea2, arg2 in ea 3, etc.
 *
 *     record(l,d,s) =
 *		    
 *             Builds a record from the list of EA * accesspath,
 *            assigns record address to d, continues with s.  An access path
 *            is a sequence of operations which either dereference field i,
 *            or return the address of field i
 *
 *	    let fun g (ea,OFFp j) = returns address of field j:
 *	                         contents(ea)+j (using pointer arithmetic)
 *	          | g (ea,SELp(j,p)) =  returns result of using access path p
 *	                          on contents of field j:
 *				     g(contents(contents(ea)+j),p)
 *
 *	     in f (d <- record-construct (map g l)
 *	     end
 *
 *      select(i,r,d,s) =
 *	   Return the contents of the ith field of r:
 *             d <- deref(contents(r)+i)
 *
 *     offset(i,r,d,s) =
 *         Return the address of the ith field of r:
 *              d <- contents(r)+i   
 *
 *     fetchIndexB(f,d,i) :
 *         Fetches the ith byte from the data object pointed to by f:
 *              d <- deref-ith-character(contents(f)+i)   
 *
 *     storeIndexB(f,d,i):
 *         Sets the ith byte in the data object:
 *              f + i := contents(d)
 *
 *     fetchIndex(r,d,i):
 *         Fetches the ith field of r:
 *           d <- deref(contents(r)+contents(i))
 *
 *     storeIndex(d,r,i): 
 *         Store d into the ith field of r:
 *           (contents(r)+contents(i)) := contents(d)
 *
 *   Bitwise operators.  Assume that bits are listed from msb to lsb.
 *   The destination EA must be a register:
 *
 *         ashl(a,b,c): c <- shift of contents(b) left by contents(a). Fill
 *                  the vacated least significant bits with zeros.
 *         ashr(a,b,c): c <- Shift of contents(b) right by contents(a).  Fill
 *                  vacated most signifcant bits with the sign bit of
 *                  contents(a) 
 *         orb(a,b,c): c <- contents(a) \/ contents(b)
 *         andb(a,b,c): c <- contents(a) /\ contents(b)
 *         xorb(a,b,c): c <- eor(contents(a),contents(b))
 *         notb(a,b) : b <- bitwise-not(b)
 *
 *   Arithmetic operators.  The destination must always be a register.
 *   For integer operators (which end in i), the operands must
 *   be integers.  For floating point operators (which end in f), the
 *   operand must be floating point locations.
 *
 *         addi(a,b,c): c <- contents(a)+contents(b).
 *         subi(a,b,c): c <- contents(b) - contents(a)
 *         divti(a,b,c): c<- contents(b)/ contents(a)
 *	   subti(a,b,c): c<- contents(b) - contents(a)
 *	   mulf (a,b,c): c<- location of the newly allocated space that holds
 *	                  deref(contents(a)) * deref(contents(a))
 *         divf (a,b,c): c <- pointer to newly allocated space that holds
 *                        deref(contents(a)) / deref(contents(b))
 *         addf (a,b,c): c <- pointer to newly allocated space that holds
 *                        deref(contents(a)) + deref(contents(b))
 *         subf (a,b,c): c <- location of the newly allocated space that holds
 *                         deref(contents(a)) - deref(contents(b))
 *
 *   Conditional statements:
 *
 *     If the condition is true when the operator is applied to the
 *     two operands, do the first statement.  if it is false, do the
 *     second statement:
 *         gbranch(cond,a,b,s1,s2) =
 *	       if contents(a) is not a loc pointing to floating point number
 *	       for contents(b) is not a loc point to floating point number
 *		     then error
 *		     else if cond(contents(a),contents(b))
 *                          then do stmt1
 *                          else do stmt2
 *         ibranch(cond,a,b,stmt1,stmt2) =
 *	         if contents(a) is not an integer
 *		 or contents(b) is not an integer
 *		     then error
 *		     else cond(contents a,contents b)
 *                          then do stmt1
 *                          else do stmt2
 *         rangeChk(a,b,stmt1,stmt2) 
 *	         if contents(a) is not an integer
 *		 or contents(b) is not an integer
 *		     then error
 *		     else contents(a) >= 0 and contents(a) < contents(b)
 *                          then do stmt1
 *                          else do stmt2
 *         bbs(n,a,stmt1,stmt2)
 *                if bit n of a is set, do stmt2, else do stmt1
 *)

signature ANSICMACHINE =
 sig
   structure CPS : CPS

   (* dump statistics info to standard output *)
 
   val statistics : unit -> unit

   (* Addressing modes:
	 There are five of them:
		immediate integer
	        register
	        functions (labels)
	        string location
	        floating point location
   *)
    
    type EA

    val immedEA : int -> EA
    val functionEA : unit -> EA
    val namedFunctionEA : string -> EA
    val stringEA : unit -> EA
    val realEA : unit -> EA

    (* Registers.  All registers must hold pointers or properly tagged
       integers, unless otherwise indicated *)

    (* Dedicated registers *)

    val exnPtr : EA    (* current exception handler *)
    val storePtr : EA  (* current store-list of modified words *)
    val dataPtr : EA   (* current top of heap pointer *)
    val limitPtr : EA  (* current heap limit pointer *)
    val varPtr : EA    (* per-thread state *)

    (* This register can hold only untagged ints *)
    
    val arithTemp : EA 

    (* These registers are not dedicated and must be disjoint *)

    val standardClosure : EA
    val standardArg : EA
    val standardCont : EA
    val miscRegs : EA list

    (* These are a list of registers which can be cached *)
     
    val localRegs : EA list

    (* All the registers *)

    val allRegs : EA list

    (* Abstract type for representing C programs *)

    type decl       (* declarations define strings and reals only *)
    type function
    type stmt       (* side-effecting statement *)
    type prog       (* a module *)

    (* Functions to create statements. *)

    val move : EA * EA * stmt -> stmt
    val jmp : EA list -> stmt
    val record : (EA * CPS.accesspath) list * EA * stmt -> stmt
    val select : int * EA * EA * stmt -> stmt
    val offset : int * EA * EA * stmt -> stmt
    val fetchIndexB : EA * EA * EA * stmt -> stmt
    val storeIndexB : EA * EA * EA * stmt -> stmt
    val fetchIndex : EA * EA * EA * stmt -> stmt
    val storeIndex : EA * EA * EA * stmt -> stmt
    val mkCase : EA * (int * stmt) list -> stmt

    val ashl : EA * EA * EA * stmt -> stmt
    val ashr : EA * EA * EA * stmt -> stmt
    val orb : EA * EA * EA * stmt -> stmt
    val andb : EA * EA * EA * stmt -> stmt
    val xorb : EA * EA * EA * stmt -> stmt
    val notb : EA * EA * stmt -> stmt

    (* arithmetic operators without overflow checking *)

    val addi : EA * EA * EA * stmt -> stmt
    val subi : EA * EA * EA * stmt -> stmt

    (* arithmetic operators with overflow checking *)

    val divti : EA * EA * EA * stmt -> stmt   
    val addti : EA * EA * EA * stmt -> stmt   
    val subti : EA * EA * EA * stmt -> stmt   
    val multi : EA * EA * EA * stmt -> stmt

    (* floating pointer operators *)

    val mulf : EA * EA * EA * stmt -> stmt
    val divf : EA * EA * EA * stmt -> stmt
    val addf : EA * EA * EA * stmt -> stmt
    val subf : EA * EA * EA * stmt -> stmt

    datatype condition = NEQ | EQL | LEQ | GEQ | LT | GT | ULT

    val bbs: int * EA * stmt * stmt -> stmt
    val ibranch : condition * EA * EA * stmt * stmt -> stmt
    val gbranch : condition * EA * EA * stmt * stmt -> stmt

    val rangeChk : EA * EA * stmt * stmt -> stmt
    val comment : string * stmt -> stmt

    (* declarations, functions, and programs *)

    val mkString : string * EA -> decl
    val mkReal : string * EA -> decl

    (* mkFunction: args is the list of EA's holding args 1 to n .  Name
       is the name of the function.   Body is the body of the function.
       Global is whether the function should have global or static
       scoping.   Alloc is the number of words allocated by the function.
       regMask is the register mask for the function *)

    val mkFunction : {args : EA list, name: EA,
		      body : stmt, alloc : int,
		      global: bool, regmask : int} list -> function
    val mkProg : decl list * function list -> prog
end
