(* Copyright (c) 1991 by Carnegie Mellon University *)

(* machine.sml
 *
 *  Print C machine instructions as C code
 *
 * Author: David Tarditi
 *         Carnegie Mellon University
 *         tarditi@cs.cmu.edu
 *
 * This program provides a set of functions to print instructions for a C
 * code machine as actual C code.
 *
 *)
            
functor CPrintFun(structure C : C
	          structure Control : CONTROL
	          structure Machine : ANSICMACHINE
	          sharing type Machine.prog = C.prog
	          sharing type Machine.EA = C.EA
	          sharing type Machine.decl = C.decl
	          sharing C.Bignum = Control.Bignum) : CPRINT =
   struct
      open Array   (* yuk! *)
      infix 9 sub

      structure C = C
      open Machine C Control  (* the sequence here is important *)

      val debug = ref false

      (* dist: create a list of integer between start and finish *)

      fun dist start finish =
	 if start <= finish then start :: dist (start+1) finish
	 else nil

      (* sublist: make a new list of elements satisfying pred *)

      fun sublist (pred : 'a->bool) =
	   let fun f nil = nil
		 | f (h::t) = if pred h then h :: f t else f t
	    in f
	    end

     (* prList: printa  list of values, separated by sep, preceded
        by first and followed by last *)

     fun prList (p:'a->string,sep,first,last) : 'a list -> string =
	 fn nil => ""
	  | h::t => first ^ p h ^ fold (fn (a,r) => sep^ p a ^r) t last

      val numRegs = List.length allRegs
      val R dataRegister = dataPtr

      val unTagInteger : EA -> int option =
	  fn (IMMED i) => SOME (Bits.rshift(i,1))
           | _ => NONE

      (* printing effective addresses *)

       val intPtrType = intType^"*"
       val intCast = "("^intType^")"
       val unsignedIntCast = "(unsigned "^intType^")"
       val intPtrCast = "("^intPtrType^")"
       val charCast = "(unsigned char)"
       val charPtrCast =  "(unsigned char *)"
       val floatCast = "*(" ^ floatType ^ "*)"
       val floatPtrCast = "(" ^ floatType ^ "*)"

      (* printing integers *)

      (* if i<0 then we chop off the tilde and replace it with a minus
         sign.  We don't use makestring(~i) because this will overflow
         when i=MININT *)

      val cInt = fn i => 
	    let val s = makestring i
            in if i>=0 then s else " -"^substring(s,1,String.length s-1)
            end

      val cbigint  =
	  let open Bignum
	      val i0 = inttobignum 0
	  in fn i => if i<i0
			 then " -"^makestring(~i)
		         else makestring i
          end

      local
         val regnames = array(numRegs,"")
         val setglobals = fn a =>
	     app (fn (R i)=>update(a,i,"R"^makestring i)) allRegs
         val setarray = fn a =>
	     app (fn (R i)=>update(a,i,"r["^makestring i^"]")) allRegs
	 val localnames = array(numRegs,"")
         val _ = (setglobals localnames;
		  app (fn (R i)=>update(localnames,i,"r"^makestring i)) localRegs)
      in
         val setRegNames = fn () =>
	     (if (!regArray) then setarray regnames else setglobals regnames)
         val labName : label -> string = fn i => "L"^makestring i
         val eaName : EA -> string = 
	   fn R i => regnames sub i
	    | LOCAL i => localnames sub i
	    | IMMED i => cInt i
	    | BIGNUM i => cbigint i
            | N i => labName i
	    | S i => labName i ^ ".s"
	    | REAL i => "&" ^ (labName i) ^ ".n"
            | EXTERN_REC n => "("^n^"+1)"
	    | NAMED n => n

         val regLocalName : EA -> string =
	   let val dataRegisterName = "*"^ (localnames sub dataRegister)
           in fn R i => 
	        if i=dataRegister then dataRegisterName
		else localnames sub i
            | a => eaName a
	   end
       end
 
      (* printing declarations *)

      (* string are printed as an array of integers, to avoid hard limits
         on token lengths in some C pre-processors.  Line breaks also
         have been inserted to avoid line length limits in C compilers *)

      val MLtoCString : string -> string =
	  let val linelength = 15
              fun stringmap (nil,_) = ["0}"]
 	        | stringmap (a::b,n) =
			 Integer.makestring(ord a) :: "," ::
		                (if n=linelength then "\n" :: stringmap(b,0)
				 else stringmap(b,n+1))
          in fn s => implode("{" :: stringmap (explode s,0))
	  end

      val MLtoCReal = fn s =>
	  implode (map (fn "~" => "-" | c => c) (explode s))

      val stringSizes : int list = 
	  let fun pow2 0 = 1
	        | pow2 i = 2 * pow2 (i-1)
	  in map pow2 (dist 0 16)
          end

      val stringStruct : string * int -> string =
	  fn (name,size) =>
	    "struct " ^ name ^ " {" ^ intType ^ " tag;" ^
	    "char s[" ^ (makestring size) ^ "];}"

      val printStructDefs : (string -> unit) -> unit =
	    fn say =>
	       let val mkStringStruct : int -> unit =
		    fn s => say (stringStruct("litString" ^ (makestring s),s) ^
			          ";\n")
	       in app mkStringStruct stringSizes;
	          say ("struct litFloat {" ^ intType ^ " tag;" ^
		       floatType ^ " n;};\n")
	       end

      val stringDefinition : (string * EA) -> string =
        fn (string,S label) =>
	   let open System.Tags
	       fun find (size :: rest) =
		   if String.length string < size then
		       "static struct litString" ^ (makestring size)
		   else find rest
		  | find nil =
		      ("static " ^ stringStruct (labName label,
						 String.length string+1))
                val tag = make_desc(String.length string, tag_embedded_string)
	    in find stringSizes ^
	        " "^labName label^"={"^(makestring tag)^ "," ^
		 MLtoCString string ^ "};\n"
            end
       | _ => ErrorMsg.impossible "c/machine.sml: 415"

      val realTag = makestring
	              (System.Tags.make_desc(floatSize,System.Tags.tag_string))

      val printDecl : (string -> unit) -> decl -> unit =
         fn say =>
             fn (STRING_DECL decl) => say (stringDefinition decl)
	      | (REAL_DECL (string,REAL label)) =>
		     say ("static struct litFloat "^labName label^"={" ^
			  realTag ^ "," ^ MLtoCReal string ^ "};\n")
              | _ => ErrorMsg.impossible "c/machine.sml: 425"

      (* print statements and expressions *)

      val conditionName : condition -> string = 
	   fn NEQ => "!="
	    | EQL => "=="
	    | LEQ => "<="
	    | GEQ => ">="
	    | LT =>  "<"
	    | GT =>  ">"
            | ULT => "<"

      val int2OpName : int2Op -> string =
	    fn ASHL => "<<"
	     | ASHR => ">>"
	     | ORB => "|"
	     | ANDB => "&"
	     | XORB => "^"
	     | MUL => "*"
	     | ADD => "+"
             | SUB => "-"
             | DIV => "/"
	     | PTRADD => "+"

      val int1OpName : int1Op -> string =
	   fn NOTB => "~"

      val floatOpName : floatOp -> string =
	    fn FMUL => "*"
             | FADD => "+"
             | FSUB => "-"
             | FDIV => "/"

      val castToInt = fn LOCAL 5 => intCast
	                | R _ => ""
			| LOCAL _ => ""
		        | IMMED _ => ""
		        | _ => intCast

      val castToIntPtr =fn (a as (LOCAL 5)) => eaName a
                          | a => "("^intPtrCast^eaName a^")"

	  
      val expToCode : exp -> string =
	let fun f e =
	  case e
	  of CAND (e1,e2) => "(" :: f e1  @ (" && " :: f e2 @ (")" :: nil))
	   | FCOND (c,e1,e2) =>
		 "(" :: floatCast :: eaName e1 :: conditionName c ::
		  floatCast :: eaName e2 :: ")" :: nil
           | ICOND (ULT,e1,e2) =>
		 "(" :: unsignedIntCast :: implode(f e1) :: conditionName ULT
		 :: unsignedIntCast :: implode(f e2) :: ")" :: nil

(* a hack to get the cast to work for the heap limit check when the heap
   limit pointer is cached in a local variable *)

           | ICOND (c,VAL e1,e2) =>
		 "(" :: castToInt e1 :: eaName e1 ::
		   (conditionName c :: f e2 @ [")"])
           | ICOND (c,e1,e2) =>
		 "(" :: f e1 @ (conditionName c :: f e2 @ [")"])
	   | IB (oper,e1,e2) =>
		 let val oper1 = f e1
		     val oper2 = f e2
		 in case oper
		    of PTRADD => "(" :: intCast :: "(" :: intPtrCast ::
			         oper1 @ ("+" :: oper2 @ ["))"])
		     | _ => "(" :: oper1 @ (int2OpName oper :: oper2 @ [")"])
	         end
	   | IU (oper,e) => "(" :: int1OpName oper :: f e @ [")"]
           | VAL e => [eaName e]
	in implode o f
        end

      val taggedIntOp : EA*EA*string * (string list -> string list) ->
	                   string list -> string list =
	  fn (ea1,ea2,cast,transform) =>
	    fn cont =>
	    let val EA1_name = eaName ea1
	    in case unTagInteger ea2
		    of SOME 0 => cast :: EA1_name :: cont
		     | SOME i => cast :: EA1_name ::
			            (transform [cInt i] @ cont)
	             | NONE => cast :: EA1_name :: (transform ("(" ::
			       eaName ea2 :: ">>1)" :: cont))
	    end

      local
        val intPtrCast = fn (LOCAL 5) => ""
		          | _ => intPtrCast
        val localDataPtr = LOCAL 5
      in
        val int2Op : int2Op * EA * EA * EA * 'a -> string list -> string list =
          fn (operator,ea1,ea2,dest,_) =>
	    fn cont =>
	     let val oper1 = eaName ea1
		 val oper2 = eaName ea2
	     in eaName dest :: "=" ::
	        (case operator
		 of PTRADD =>
	             if dest=localDataPtr 
			  then intPtrCast ea1 :: oper1 :: "+" :: oper2
			       :: ";\n" :: cont
			  else intCast :: "(" :: intPtrCast ea1 :: oper1
			       :: "+" :: oper2 :: ")" :: ";\n" :: cont
	         | _ => oper1 :: int2OpName operator :: oper2 :: ";\n" :: cont)
	    end
      end

      val int1Op : int1Op * EA * EA * 'a -> string list -> string list =
       fn (operator,ea1,dest,_) => fn cont =>
	    eaName dest :: "= " :: int1OpName operator :: eaName ea1 ::
	    ";\n" :: cont

      val floatOp : floatOp * EA * EA * EA * 'a -> string list -> string list =
	 let val floatArg = "(*" ^ floatPtrCast
	     val contents = fn (REAL r) => labName r ^ ".n"
			     | a => floatArg ^ eaName a ^ ")"
	 in fn (operator,dataPtr,ea1,ea2,_) => fn cont =>
	        floatCast :: "(" :: castToIntPtr dataPtr :: "+1)= " ::
		contents ea1 :: floatOpName operator :: contents ea2 ::
		";\n" :: cont
         end

      (* stmtToCode: convert a statement to a string *)

      val stmtToCode : stmt -> string = fn stmt =>
	let val rec f = fn stmt =>
	     case stmt
	     of (ASSIGN (ea,dest as (LOCAL 5),cont)) =>
	         eaName dest :: "=" :: castToIntPtr ea :: ";\n" :: f cont
	      | (ASSIGN (ea,dest,cont)) =>
	         eaName dest :: "=" :: castToInt ea ::
		  eaName ea :: ";\n" :: f cont
  	      | (JMP (ea :: _)) =>
	          "return(" :: castToInt ea :: eaName ea :: ");\n" :: nil
              | (ALLOC (elems,dataPtr,cont)) =>
		   let open CPS
		       fun d (SELp(j,p)) = 1 + d p
		         | d (OFFp 0) = 0
			 | d (OFFp _) = 1

		       fun prpath (path,SELp(j,p)) =
			   prpath (("*(" :: path) @ ["+",cInt j,")"],p)
			 | prpath (path,OFFp 0) = path 
			 | prpath (path,OFFp j) = intCast :: "(" :: path @ ["+",cInt j,")"]
		      val dataPtrName = castToIntPtr dataPtr

		      fun ast(0,rest) = rest
			| ast(i,rest) = "*"::ast(i-1,rest)

		      val dest = fn i =>
			  if !autoIncrement then
			      "*" ^ dataPtrName ^ "++"
			  else dataPtrName^"["^makestring i^"]"

	              fun scan(i,(from,OFFp 0) :: rest) =
			       dest i ::
			       "= " :: castToInt from :: eaName from ::
			       ";\n" :: scan(i+1,rest)
			| scan(i,(from,path) :: rest) =
			        dest i ::
			        "= " ::
			       prpath("(" :: intType :: ast (d path,[")",
			              eaName from]),path) @
			              (";\n" ::  scan(i+1,rest))
			| scan(_,nil) = nil
		   in scan(0,elems) @ f cont
		   end
	      | (CALL ((NAMED name,_,_),args,r,cont)) =>
		  (case r
		   of NONE => nil
		    | SOME ea => eaName ea :: "=" :: nil) @ (name::
                    prList (eaName,",","(",")") args ::
		    ";\n"::f cont)
	      | (CASE (v,cases)) =>
		    "switch (" :: eaName v :: ">>1)" ::
                    prList (fn (label,stmt) =>
			      "case "  ^ (makestring label) ^ ":" ^ implode (f stmt),
			    "","{\n","};\n") cases :: nil
	      | (COND (exp,stmt1,stmt2)) =>
		    "if(" :: expToCode exp ::
		     "){" :: (f stmt1) @ "}else{" :: f stmt2 @ ["};"]
	      | (FETCH (ea1,ea2,dest,cont)) =>
		      eaName dest :: "= *(" ::
		      taggedIntOp(ea1,ea2,intPtrCast,
                                         fn s => "+" :: s) (");\n" :: f cont)
	      | (FETCHB (ea1,ea2,dest,cont)) =>
		    eaName  dest :: "= *(" ::
		      taggedIntOp(ea1,ea2,charPtrCast,
                                         fn s => "+" :: s) (");\n" :: f cont)
	      | (GOTO l) => "goto " :: labName l :: ";\n" :: nil
              | (LABEL (l,stmt)) => labName l :: ": " :: f stmt
	      | (SET (value,dest,offset,cont)) =>
		    "*(" :: taggedIntOp(dest,offset,intPtrCast,
		                             fn s => "+" :: s) (")=" ::
			 castToInt value :: eaName value :: ";\n" :: f cont)
	      | (SETB (value,dest,offset,cont)) =>
		    "*(" :: taggedIntOp(dest,offset,charPtrCast,
		                             fn s => "+" :: s) (")=" ::
			 charCast :: eaName value :: ";\n" :: f cont)
              | (INT2OP (args as (_,_,_,_,cont))) =>
		        int2Op args (f cont)
	      | (INT1OP (args as (_,_,_,cont))) =>
		        int1Op args (f cont)
              | (FLOAT (args as (_,_,_,_,cont))) => floatOp args (f cont)
	      | (COMMENT (s,cont))  => "/*" :: s :: "*/\n" :: f cont
	      | (SEQ l) => fold (fn (a,rest)=> f a @ rest) l nil
	      | _ => ErrorMsg.impossible "c/machine.sml:504"
	in implode(f stmt)
	end

        val printFunction : (string -> unit ) -> function -> unit =
	 let val regType = "register " ^ intType ^ " "
             val basereg = regType ^ "*r;"
	 in fn say => fn FUNC (name,locals,stmt,isGlobal) =>
	    (say ((if isGlobal then "" else "static ") ^ intType
		  ^ " " ^ eaName name ^ "(r)\n"^basereg^"\n{" ^
		  ( (* if (!regOpt) 
		     then *) prList(regLocalName,",",regType,";\n") locals
		     (* else "" *)) ^
		  stmtToCode stmt ^ "}"))
	 end

        (* print declarations of functions *)

        val printFunctionDecls : (string -> unit) -> function list -> unit =
	   fn say => fn l =>
	       let val isGlobal = fn (FUNC a) => #4 a
		   fun printFuncs scope l =
		      say (prList(fn FUNC a => (eaName (#1 a))^"()",",\n",
			       scope^" "^intType^" ",";\n") l)
	       in printFuncs "extern" (sublist isGlobal l);
		  printFuncs "static" (sublist (not o isGlobal) l)
               end

        (* print a program *)

	val printProg : (string -> unit) * prog -> unit =
         fn (say,PROG(decls,funcs)) =>
	    (setRegNames();
	     app (printDecl say) decls;
	     printFunctionDecls say funcs;
	     app (printFunction say) funcs)
 
        (* declare the following external functions:
	            extern int invoke_gc(),mult(),inlined_gc()
           and the following external values:
		    overflow ref
		    pseudo-registers *)

       val startFile = fn say =>
          (say ("extern " ^ intType ^ " invoke_gc()," ^ mult ^
	       "(),inlined_gc()," ^ overflow ^ "[]" ^ 
	       (if !instrum then ",_s2c_jc,_s2c_gc,_s2c_ic,_s2c_wc,_s2c_rc,_s2c_hc"
		else "") ^
		  (if (!regArray)
		     then ";\n"
                     else prList (fn i=> (if i=dataPtr orelse i=limitPtr
				          then "*"
					  else "")^eaName i,
				    ",",",",";\n") allRegs));
          printStructDefs say)
 
        type entryPoint = {name : string, entry : string}

        val timestamp : unit -> string =
		let open System.Unsafe.CInterface
		     val timeofday : unit -> (int*int) =
			  (c_function "timeofday") handle CFunNotFound _ =>
			                           (fn () => (0,0))
		in fn () => 
		     case (timeofday ())
		     of (a,b) => (makestring a)^"#"^(makestring b)
		end 


        (* printDlist: print entry point assocation list:
	        A sequence of array definitions initialized to the following
		values:

	          {tag, name (a string address), entry point function,
		   pointer to next entry in the list}
           Also print a time stamp (for use in export)

         *)

        val printDlist : string * (string -> unit) * entryPoint list -> unit =
          fn (name,say,entryPoints : entryPoint list) =>
	    let
	     (* create a string declaration for each entry point name.
	        This creates a list of declarations and EA's *)

		val declsAndEA =
		      map (fn a=>
			      let val l = stringEA()
			      in (mkString (#name a,l),l)
			      end) entryPoints

		val decls = map #1 declsAndEA
		val stringEAs = map #2 declsAndEA

	     (* now create labels for each structure corresponding to
	        a list element *)

		val labels =
		  case entryPoints
		  of nil => nil
		   | h :: t => namedFunctionEA name ::
	                         map (fn _ => functionEA()) t

               val tag = makestring (System.Tags.make_desc(3, System.Tags.tag_record))
           fun pr ((({entry,...}:entryPoint)::er,label::lr,string::sr),first) =
 		  pr ((er,lr,sr),false) ^
		(if first then "" else "static ") ^ intType ^ " " ^
	       eaName label ^
	       "[4]={" ^ tag ^ "," ^ intCast ^ eaName string ^ "," ^ intCast ^
		entry ^ ",\n" ^
	         (case lr
		  of nil => "1"
		   | nextLabel :: _ => intCast^"("^eaName nextLabel ^"+1)") ^
		 "};\n"
	       | pr ((nil,nil,nil),_) = ""
               | pr _ = ErrorMsg.impossible "c/machine.sml: 787"
            in say (prList (fn a => #entry a^"()",
		    ",\n","extern "^intType^" ",";\n") entryPoints);
	       app (printDecl say) decls;
               say (pr((entryPoints,labels,stringEAs),true));
	       say ("char *sml2c_timestamp = \""^timestamp()^"\";\n")
	    end
end
