(* Copyright (c) 1991 by Carnegie Mellon University *)

(* c.sml
 *
 *  A functor to create the representation for instructions for a C code
 *  machine.
 *
 *)
   
functor CFun(structure Bignum : CBIGNUM
	     structure CPS : CPS) : C =
   struct
      structure CPS = CPS
      structure Bignum = Bignum

      type register = int
      type label = int

      datatype EA = R of register        (* register *)
	          | IMMED of int         (* immediate integer *)
                  | N of label           (* function name *)
                  | S of label           (* string location *)
                  | REAL of label        (* real location *)
                  | LOCAL of register    (* local register *)
                  | EXTERN_REC of string (* external record *)
	          | NAMED of string      (* external function *)
                  | BIGNUM of Bignum.bignum  (* arbitrary precision integer *)
		    
      datatype int2Op = ASHL | ASHR | ORB | ANDB | XORB
		      | MUL | ADD | SUB | DIV | PTRADD
      datatype int1Op = NOTB
      datatype floatOp = FMUL | FADD | FSUB | FDIV
      datatype condition = NEQ | EQL | LEQ | GEQ | LT | GT | ULT (* unsigned *)
      datatype decl = STRING_DECL of string * EA
		    | REAL_DECL of string * EA
      datatype exp = VAL of EA
	           | IB of int2Op * exp * exp
		   | IU of int1Op * exp
		   | ICOND of condition * exp * exp
		   | FCOND of condition * EA * EA
		   | CAND of exp * exp

      datatype stmt = ASSIGN of EA * EA * stmt
	            | ALLOC of (EA * CPS.accesspath) list * EA * stmt
		    | CALL of (EA * EA list * EA list) * EA list *
		               EA option * stmt
		    | CASE of EA * (int * stmt) list 
		    | COND of exp * stmt * stmt
		    | COMMENT of string * stmt
		    | FETCH of EA * EA * EA * stmt
		    | FETCHB of EA * EA * EA * stmt
		    | FLOAT of floatOp * EA * EA * EA * stmt
		    | GOTO of label
		    | INT2OP of int2Op * EA * EA * EA * stmt
		    | INT1OP of int1Op * EA * EA * stmt
		    | JMP of EA list
		    | LABEL of label * stmt
		    | SEQ of stmt list
		    | SET of EA * EA * EA * stmt 
		    | SETB of EA * EA * EA * stmt

      (* FUNC(name,local vars, body, external or static) *)

      datatype function = FUNC of EA * EA list * stmt * bool
      datatype prog = PROG of decl list * function list
end
