(*  Title: 	goals
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Goal stack package.

An "autotactic" feature (like Nuprl's) was tried but proved useless.
To apply a special tactic each time, define your own version of "by":
  fun myby tac = tac THEN my_autotac
At least this way you will remember which autotactic you are using!
*)

signature GOALS =
sig
  structure Tactical: TACTICAL
  local open Tactical Tactical.Thm
  in
    type gstack
    val back: unit -> unit
    val by: tactic -> unit
    val byev: tactic list -> unit
    val chop: unit -> unit
    val choplev: int -> unit
    val compat_goal: thm list -> int -> thm list
    val curr_prems: thm list ref
    val filter_goal: (term*term->bool) -> thm list -> int -> thm list
    val getgoal: int -> term
    val getstate: unit -> gstack
    val goal: theory -> string -> thm list
    val goals_limit: int ref
    val pr: unit -> unit
    val print_goals: int -> thm -> unit
    val print_thm: thm -> unit
    val prlev: int -> unit
    val prth: thm -> unit
    val prths: thm list -> unit
    val prove_goal: theory -> string -> (thm list -> tactic list) -> thm
    val result: unit -> thm  
    val uresult: unit -> thm  
    val setstate: gstack -> unit
    val topthm: unit -> thm
    val undo: unit -> unit
  end
end;


functor GoalsFun (structure Logic: LOGIC and Tactic: TACTIC
		  and Pretty: PRETTY) : GOALS = 
struct
structure Tactical = Tactic.Tactical
local open Tactic Tactic.Tactical Tactic.Tactical.Thm
in

(*Each level of goal stack includes a proof state and alternative states,
  the output of the tactic applied to the preceeding level.  *)
type gstack = (thm * thm Sequence.seq) list;


(*** References ***)

(*Max number of goals to print -- set by user*)
val goals_limit = ref 10;

(*Current assumption list -- set by "goal".
  Useful if you forgot to save the value returned by "goal". *)
val curr_prems = ref([] : thm list);

(*Current result maker -- set by "goal", used by "result".  *)
val curr_mkresult = ref((fn _=> error"No current state\n") : bool*thm->thm);

(*List of previous goal stacks, for the undo operation.  Set by setstate. 
  A list of lists!*)
val undo_list = ref([] : gstack list);


(*** Printing of theorems ***)

(*Print a meta-theorem.  To be called by other printing operations. *)
fun print_thm th : unit =
  let val {sign, hyps, prop,...} = rep_thm th
  in  Pretty.bg 0;  Sign.print_term sign prop;  
      if null hyps then ()
      else  (Pretty.brk(2,0);
	     Pretty.list ("[ ", " ]", Sign.print_term sign) hyps);
      Pretty.en()
  end;


(*Print a meta-theorem.  Top-level command. *)
fun prth th = (Pretty.init();  print_thm th;  Pretty.nl());

(*Print a list of theorems, separated by blank lines*)
val prths = print_list_ln prth;


(*Print thm A1,...,An/B in "goal style" -- premises as numbered subgoals*)
fun print_goals maxgoals th : unit =
  let val {sign, hyps, prop,...} = rep_thm th;
      fun printgoals (_, []) = ()
        | printgoals (n, A::As) =
             (Pretty.st (" "  ^ string_of_int n  ^  ". ");  
              Sign.print_term sign A;  Pretty.nl();
              printgoals (n+1,As));
      fun printpair (t,u) =
	(Sign.print_term sign t;  Pretty.st" =";  Pretty.brk(1,0);
	 Sign.print_term sign u);
      fun printff [] = ()
        | printff tpairs = 
	    (prs"\nFlex-flex pairs:\n";  
	     Pretty.list("", "", printpair) tpairs;  Pretty.nl());
      val (tpairs,As,B) = Logic.strip_horn(prop);
      val ngoals = length As
  in Pretty.init();  Sign.print_term sign B;  Pretty.nl();
     if ngoals=0  then prs"No subgoals!\n"
       else if ngoals>maxgoals 
       then (printgoals (1, front(maxgoals,As));
	     prs("A total of " ^ string_of_int ngoals ^ " subgoals...\n"))
       else printgoals (1, As);
     printff tpairs
  end;


(*** Setting up goal-directed proof ***)


(*Common treatment of "goal" and "prove_goal":
  Return assumptions, initial proof state, and function to make result. *)
fun prepare_proof thy agoal =
  let val chorn = Sign.type_assign (Sign.read_cterm(sign_of thy)(agoal,Aprop));
      val {sign, t=horn,...} = Sign.rep_cterm chorn;
      val (_,As,B) = Logic.strip_horn(horn);
      val cAs = map (Sign.cterm_of sign) As;
      fun result_error msg = error
	("result: " ^ msg ^ "\nGoal was " ^ agoal ^ "\n");
      (*discharges assumptions from state in the order they appear in goal;
	checks (if requested) that resulting theorem is equivalent to goal. *)
      fun mkresult (check,state) =
        let val ngoals = length (prems_of state);
            val th = implies_intr_list cAs state;
            val {hyps,prop,...} = rep_thm th
        in  if ngoals>0 then result_error 
		(string_of_int ngoals ^ " unsolved goals!")
            else  case hyps of
      	        [] => if (not check) orelse (prop aconv Sign.term_of chorn)
		      then  standard th 
		      else  result_error "proved a different theorem"
	      | _::_ => result_error "additional hypotheses"
        end;
      val prems = map (forall_elim_vars 0  o  assume) cAs
      and st0 = trivial (Sign.cterm_of sign B)
  in  (prems, st0, mkresult)  end;


(*Prove theorem using the tactics in sequence; check it has the specified form.
  Augments signature with all type assignments of goal.
  Syntax is similar to "goal" command for easy keyboard use.*)
fun prove_goal thy agoal tacsf =
  let val (prems, st0, mkresult) = prepare_proof thy agoal;
      val tac = EVERY (tacsf prems);
      val state = (case Sequence.pull (tapply(tac,st0)) of Some(st,_) => st
		 | _ => error ("prove_goal: tactic failed\n" ^ agoal ^ "\n"))
  in  mkresult (true,state)  end
  handle ERROR => error (*from type_assign, etc via prepare_proof*)
	    ("The above error occurred for " ^ agoal ^ "\n")
       | _ => error ("prove_goal: exception was raised!\n " ^ agoal ^ "\n");


(*** Commands etc ***)


(*Return the current goal stack, if any, from undo_list*)
fun getstate() : gstack = case !undo_list of 
      []   => error"No current state\n"
    | x::_ => x;

(*Pops the given goal stack*)
fun pop [] = error"Stack is empty\n"
  | pop (pair::pairs) = (pair,pairs);


(*Print a level of the goal stack.  Ignore Poly/ML spurious I/O exception*)
fun print_top ((th,_), pairs) = 
   (prs("Level " ^ string_of_int(length pairs) ^ "\n"); 
    print_goals (!goals_limit) th)
  handle Io _ => prs"\nio_failure! -- Poly/ML bug!!\n";   


(*Printing can raise exceptions, so the assignment occurs last*)
fun setstate newgoals = 
  (print_top (pop newgoals);  undo_list := newgoals :: !undo_list);

(*Given a proof state transformation, return a command that updates
    the goal stack*)
fun make_command com = setstate (com (pop (getstate())));

(*Apply a function on proof states to the current goal stack*)
fun apply_fun f = f (pop(getstate()));

(*Return the top theorem, representing the proof state*)
fun topthm () = apply_fun  (fn ((th,_), _) => th);

(*Return the final result.  *)
fun result () = !curr_mkresult (true, topthm());

(*Return the result UNCHECKED that it equals the goal -- for synthesis,
  answer extraction, or other instantiation of Vars *)
fun uresult () = !curr_mkresult (false, topthm());


(*Returning some subgoal in the proof state*)
fun getgoal_fun i ((th,_), _) : term =
      (case  nth_tail (i-1, prems_of th)  of
	    [] => error"getgoal: Goal number out of range\n"
	  | Q::_ => Q);

(*Get subgoal i from goal stack*)
fun getgoal i = apply_fun (getgoal_fun i);


(*Which thms could apply to goal i? (to debug tactics involving filter_thms) *)
fun filter_goal could ths i = filter_thms could (999, getgoal i, ths);

(*Which thms are compatible with goal i? *)
fun compat_goal thms i =
  compat_thms(itlist_right insert_thm (thms, Stringtree.null),
                      getgoal i);


fun chop_level n (pair,pairs) = nth_tail (length pairs - n, pair::pairs);

(*Print the given level of the proof*)
fun prlev n = apply_fun (print_top o pop o (chop_level n));
fun pr () = apply_fun print_top;


(*Read main goal.  Set global variables curr_prems, curr_mkresult. *)
fun goal thy agoal = 
  let val (prems, st0, mkresult) = prepare_proof thy agoal
  in  undo_list := [];
      setstate [ (st0, Sequence.null) ];  
      curr_prems := prems;
      curr_mkresult := mkresult;
      prems
  end 
  handle ERROR => error (*from type_assign, etc via prepare_proof*)
	    ("The above error occurred for " ^ agoal ^ "\n");



(*Proof step "by" the given tactic -- apply tactic to the proof state*)
fun by_com tac ((th,ths), pairs) : gstack =
      (case  Sequence.pull(tapply(tac, th))  of
	   None      => error"by: tactic returned no results\n"
	 | Some pair => (pair::(th,ths)::pairs));

fun by tac = make_command (by_com tac);

(* byev[tac1,...,tacn] applies tac1 THEN ... THEN tacn.
   Good for debugging proofs involving prove_goal.*)
val byev = by o EVERY;


(*Backtracking means find an alternative result from a tactic.
  If none at this level, try earlier levels*)
fun backtrack [] = error"backtrack: no alternatives\n"
  | backtrack ((_,thstr) :: pairs) =
      (case Sequence.pull thstr of
	   None      => backtrack pairs
	 | Some pair => pair::pairs);

fun back() = setstate (backtrack (getstate()));

(*Chop back to previous level of the proof*)
fun choplev n = make_command (chop_level n);

(*Chopping back the goal stack*)
fun chop () = make_command (fn (_,pairs) => pairs);

(*Restore the previous proof state;  discard current state. *)
fun undo() = case !undo_list of
      [] => error"No proof state\n"
    | [_] => error"Already at initial state\n"
    | _::newundo =>  (undo_list := newundo;  pr()) ;

end;
end;
