(*  Title: 	LK/set/resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Resolution for Classical Set Theory (with thanks to Philippe de Groote) 

Most of these analyse the formula  a:B  on the structure of B.
 *)

signature SET_RESOLVE = 
sig
  structure Set_Rule: SET_RULE
  local open Set_Rule.Pure
  in
  val Collect_left: thm
  val Collect_right: thm
  val Diff_left: thm
  val Diff_right: thm
  val duplicate_tac: int -> tactic
  val eqext_left_thin: thm
  val eqmem_left_thin: thm
  val equal_left_s: thm
  val equal_right: thm
  val ext_pack: thm list * thm list
  val Inter_left_nonempty: thm
  val Inter_left_thin: thm
  val Inter_right: thm
  val Int_left: thm
  val Int_right: thm
  val null_right: thm
  val Pow_left: thm
  val Pow_right: thm
  val refl_left: thm
  val Replace_left: thm
  val Replace_right_comb: thm
  val Replace_right_thin: thm
  val safe_setpc_tac: int -> tactic
  val setcons_left: thm
  val setcons_right1: thm
  val setcons_right2: thm
  val setcons_right: thm
  val setpc_step_tac: int -> tactic
  val setpc_tac: int -> tactic
  val set_pack: thm list * thm list
  val set_step_tac: int -> tactic
  val set_tac: int -> tactic
  val subset_left_thin: thm
  val subset_right: thm
  val thin_tac: int -> tactic
  val Union_left: thm
  val Union_null_right: thm
  val Union_right_thin: thm
  val Union_setcons_right: thm
  val Un_left: thm
  val Un_right: thm
end;
end;


functor Set_ResolveFun (structure Set_Rule: SET_RULE
			and LK_Resolve: LK_RESOLVE
    sharing LK_Resolve.LK_Rule.Pure = Set_Rule.Pure) : SET_RESOLVE = 
struct
structure Set_Rule = Set_Rule;
local  open  Set_Rule.Pure  LK_Resolve  LK_Resolve.LK_Rule
             Set_Rule  Set_Rule.Set_Syntax
in 


(*Duplicate some formula on the left or right.
  Both duplicate_left and duplicate_right fail if no formula exists. *)
fun duplicate_tac i = resolve_tac [duplicate_left,duplicate_right] i;

fun thin_tac i = resolve_tac [thin_left,thin_right] i;


(* Thinning of a:0 on the right*)
val null_right = prove_goal Set_Rule.thy
    "$H |- $E, $F ==> $H |- $E, a:0, $F"
 (fn prems=>
  [ (REPEAT (resolve_tac (thin_right::prems) 1)) ]);


(*Collect : subset comprehension*)

val Collect_right = prove_goal Set_Rule.thy
    "[| $H |- $E, a:A, $F;  $H |- $E, P(a), $F |] ==> \
\    $H |- $E, a : Collect(A,P), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Collect_def]),
    (REPEAT (resolve_tac (conj_right::prems) 1)) ]);


val Collect_left = prove_goal Set_Rule.thy
    "$H, a: A, P(a), $G |- $E ==> $H, a: Collect(A,P), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Collect_def]),
    (REPEAT (resolve_tac (conj_left::prems) 1)) ]);


(*Setcons: addition of a member *)

val setcons_right = prove_goal Set_Rule.thy
    "$H |- $E, a=b, a:B, $F ==> $H |- $E, a : (b::B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [setcons_def]),
    (REPEAT (resolve_tac (disj_right::prems) 1)) ]);

(*These two are for PROLOG-like enumeration of elements of a finite set
	in backtracking.*)

val setcons_right1 = prove_goal Set_Rule.thy
    "$H |- $E, a : (a::B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [setcons_def]),
    (REPEAT (resolve_tac (prems@[disj_right,refl]) 1)) ]);

val setcons_right2 = prove_goal Set_Rule.thy
    "$H |- $E, a:B, $F ==> $H |- $E, a : (b::B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [setcons_def]),
    (REPEAT (resolve_tac (prems@[disj_right,thin_right]) 1)) ]);

val setcons_left = prove_goal Set_Rule.thy
    "[| $H, a=b, $G |- $E;  $H, a:B, $G |- $E |] ==> $H, a:(b::B), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [setcons_def]),
    (REPEAT (resolve_tac (prems@[disj_left]) 1)) ]);


(*Subset relation*)

val subset_right = prove_goal Set_Rule.thy
    "(!x.$H, x:A |- $E, x:B, $F) ==> $H |- $E, A <= B, $F"
 (fn prems=>
  [ (rewrite_goals_tac [subset_def]),
    (REPEAT (resolve_tac (prems@[all_right,imp_right]) 1)) ]);


(* c is new variable. *)
val subset_left_thin = prove_goal Set_Rule.thy
    "[| $H, $G |- $E, c:A;  $H, c:B, $G |- $E |] ==> $H, A <= B, $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [subset_def]),
    (REPEAT (resolve_tac (prems @ [all_left_thin,imp_left]) 1)) ]);


(*Powerset rules in terms of subsets (not elements). 
  Further reasoning takes place using the many different subset rules. *)

val Pow_right = prove_goal Set_Rule.thy
    "$H |- $E, A<=B, $F ==> $H |- $E, A : Pow(B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Pow_def]),
    (REPEAT (resolve_tac prems 1))  ]);


val Pow_left = prove_goal Set_Rule.thy
    "$H, A<=B, $G |- $E ==> $H, A : Pow(B), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Pow_def]),
    (REPEAT (resolve_tac prems 1))  ]);


(*Equality by extensionality, in terms of subsets. *)

val equal_right = prove_goal Set_Rule.thy
    "[| $H |- $E, A<=B, $F;  $H |- $E, B<=A, $F |] ==> $H |- $E, A=B, $F"
 (fn prems=>
  [ (rewrite_goals_tac [ext_def]),
    (REPEAT (resolve_tac (prems @ [conj_right]) 1)) ]);

val equal_left_s = prove_goal Set_Rule.thy
    "$H, A<=B, B<=A, $G |- $E ==> $H, A=B, $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [ext_def]),
    (REPEAT (resolve_tac (prems @ [conj_left]) 1)) ]);



(*Equality by extensionality, in terms of membership; c is new variable.  *)

val eqext_left_thin = prove_goal Set_Rule.thy
  "[| $H, $G |- $E, c:A, c:B;  $H, c:B, c:A, $G |- $E |] ==> $H, A=B, $G |- $E"
 (fn prems=>
  [ (reresolve_tac (prems @ [basic,equal_left_s,subset_left_thin]) 1) ]);


(* Equality-left by equality of members;  c is new var.
    If  a=b   then   ALL x. a:x <-> b:x  *)
val eqmem_left_thin = prove_goal Set_Rule.thy
  "[| $H, $G |- $E, a:c, b:c;  $H, $G, b:c, a:c |- $E |] ==> $H, a=b, $G |- $E"
 (fn prems=>
  [ (resolve_tac [cut] 1),
    (resolve_tac [cut] 2),
    (DEPTH_SOLVE_1 (resolve_tac (prems @ [thin_left]) 3)),
    (resolve_tac [cut] 1),
    (resolve_tac [cut] 1),
    (DEPTH_SOLVE_1 (resolve_tac (prems @ [thin_left,thin_right]) 1)),
    (REPEAT (resolve_tac [basic,sym,equal_members] 1)) ]);


(*Thinning of a=a on the left.  
  If one of the a's is a Var, causes replacement throughout goal*)
val refl_left = prove_goal Set_Rule.thy
    "$H, $G |- $E ==> $H, a=a, $G |- $E"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [thin_left]) 1)) ]);


(*Replace-right.  Simple version, no new variables *)
val Replace_right_comb = prove_goal Set_Rule.thy
    "$H |- $E, a:B, $F ==> $H |- $E, f(a) : Replace(f,B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Replace_def]),
    (REPEAT (resolve_tac (prems @ [refl, exists_right_thin, conj_right]) 1)) ]);


(*Replace-right.   "a" is new variable. *)
val Replace_right_thin = prove_goal Set_Rule.thy
    "[| $H |- $E, a:B, $F;  $H |- $E, c=f(a), $F |] ==> \
\    $H |- $E, c : Replace(f,B), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Replace_def]),
    (REPEAT (resolve_tac (prems @ [exists_right_thin,conj_right]) 1)) ]);


val Replace_left = prove_goal Set_Rule.thy
    "(!x.$H, x:B, c=f(x), $G |- $E) ==> $H, c: Replace(f,B), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Replace_def]),
    (REPEAT (resolve_tac (prems @ [exists_left,conj_left]) 1)) ]);


(*Union-right.   B is new variable.*)
val Union_right_thin = prove_goal Set_Rule.thy
  "[| $H |- $E, A:B, $F;  $H |- $E, B:C, $F |] ==> $H |- $E, A : Union(C), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Union_def]),
    (REPEAT (resolve_tac (prems @ [exists_right_thin,conj_right]) 1)) ]);


val Union_left = prove_goal Set_Rule.thy
    "(!x.$H, A:x, x:C, $G |- $E) ==> $H, A:Union(C), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Union_def]),
    (REPEAT (resolve_tac (prems @ [exists_left,conj_left]) 1)) ]);


(*Inter-right.  The second premise handles the case C=0.*)
val Inter_right  = prove_goal Set_Rule.thy
    "(!x.$H, x:C |- $E, A:x, $F) ==> $H, C<=0 |- $E, $F ==> \
\    $H |- $E, A: Inter(C), $F"
 (fn prems=>
  [ (rewrite_goals_tac [Inter_def]),
    (reresolve_tac (prems@[Collect_right,all_right,imp_right]) 1),
    (resolve_tac [cut] 1),
    (resolve_tac [thin_right] 2  THEN  resolve_tac prems 2),
    (reresolve_tac [basic,subset_right,Union_right_thin] 1),
    (resolve_tac [thin_right] 1  THEN  resolve_tac prems 1) ]);



(*Inter-left.   B is new variable.
  Does not add the assumption that C is nonempty -- this typically causes
    gross additional computation.  Use the rule Inter_left_nonempty instead. *)
val Inter_left_thin = prove_goal Set_Rule.thy
    "[| $H, A:B, $G |- $E;  $H, $G |- $E, B:C |] ==> $H, A: Inter(C), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Inter_def]),
    (resolve_tac [Collect_left] 1),
    (resolve_tac [thin_left] 1),
    (reresolve_tac (prems@[all_left_thin,imp_left]) 1) ]);


(*A way of exploiting the equality Inter(0)=0.  For interactive use only.
  Use Inter_left to add the assumption x:A.  (Currently unused??) *)
val Inter_left_nonempty = prove_goal Set_Rule.thy
    "(!x.$H, x:C, $G |- $E) ==> $H, A : Inter(C), $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Inter_def]),
    (resolve_tac [Collect_left] 1),
    (resolve_tac [Union_left] 1),
    (resolve_tac [thin_left] 1  THEN  resolve_tac [thin_left] 1  THEN 
     resolve_tac prems 1) ]);


(*A special case of Union_right with no new variables. *)
val Union_setcons_right = prove_goal Set_Rule.thy
    "$H |- $E, $F, A:b, A : Union(C) ==> $H |- $E, A : Union(b::C), $F"
 (fn prems=>
  [ (resolve_tac [cut] 1),
    (resolve_tac [cut] 1),
    (resolve_tac [thin_right] 1  THEN  resolve_tac prems 1),
    (REPEAT (resolve_tac [basic,refl,Union_left,Union_right_thin,setcons_right]
      1)) ]);


(*Deletion of  a: Union(0) on the right.
  Avoids use of Union_right, which would cause a needless case split.*)
val Union_null_right = prove_goal Set_Rule.thy
    "$H |- $E, $F ==> $H |- $E, a: Union(0), $F"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [thin_right]) 1)) ]);


(*Binary Union*)

val Un_right = prove_goal Set_Rule.thy
    "$H |- $E, $F, c:A, c:B ==> $H |- $E, c : A Un B, $F"
 (fn prems=>
  [ (rewrite_goals_tac [Un_def]),
    (reresolve_tac (prems@[Union_null_right, Union_setcons_right]) 1) ]);


val Un_left = prove_goal Set_Rule.thy
    "[| $H, c:A, $G |- $E;  $H, c:B, $G |- $E |] ==> $H, c: A Un B, $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Un_def]),
    (reresolve_tac [basic, null_left, Union_left, setcons_left, 
		       eqext_left_thin] 1), 
    (REPEAT(resolve_tac [thin_left] 1  THEN  resolve_tac [thin_left] 1  THEN 
   	    resolve_tac prems 1)) ]);


(*A special case of Inter_left with no new variables.
  Not currently used.  Not obviously useful!
val Inter_setcons_left = prove_goal Set_Rule.thy
    "[| $H, $G, C<=0, A:b |- $E;  $H, $G, A:b, A : Inter(C) |- $E |] ==> $H, A : Inter(b::C), $G |- $E"    ****)


(*Binary Intersection*)

val Int_right = prove_goal Set_Rule.thy
    "[| $H |- $E, c:A, $F;  $H |- $E, c:B, $F |] ==> $H |- $E, c : A Int B, $F"
 (fn prems=>
  [ (rewrite_goals_tac [Int_def]),
    (REPEAT (resolve_tac (prems@[Collect_right]) 1)) ]);


val Int_left = prove_goal Set_Rule.thy
    "$H, c:A, c:B, $G |- $E ==> $H, c : A Int B, $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Int_def]),
    (REPEAT (resolve_tac (prems@[Collect_left]) 1)) ]);



(*Set Difference*)

val Diff_right = prove_goal Set_Rule.thy
    "[| $H |- $E, c:A, $F;  $H, c:B |- $E, $F |] ==> $H |- $E, c:A-B, $F"
 (fn prems=>
  [ (rewrite_goals_tac [Diff_def]),
    (REPEAT (resolve_tac (prems@[Collect_right,not_right]) 1)) ]);


val Diff_left = prove_goal Set_Rule.thy
    "$H, c:A, $G |- $E, c:B ==> $H, c : A-B, $G |- $E"
 (fn prems=>
  [ (rewrite_goals_tac [Diff_def]),
    (REPEAT (resolve_tac (prems@[Collect_left,not_left]) 1)) ]);



(*Set Theory including Union and Intersection*)

val set_pack =
   ([null_left, Collect_right, Collect_left,
     setcons_right, setcons_left, subset_right, Pow_right, Pow_left,
     Replace_right_comb, Replace_left,
     Union_left, Inter_right, 
     Un_right, Un_left, Int_right, Int_left, Diff_right, Diff_left],
    [null_right, subset_left_thin, Replace_right_thin,
     Union_right_thin, Inter_left_thin]);


(*Extensionality*)

val ext_pack = ([equal_right], [eqext_left_thin]);



(*Tactic for Set Theory rules including union and intersection.*)
val set_tac = repeat_goal_tac [triv_pack, set_pack];


(*Tactic for Set Theory and Predicate Calculus rules. *)
val setpc_tac = repeat_goal_tac [triv_pack, set_pack, LK_pack];


val safe_setpc_tac = safe_goal_tac [triv_pack, set_pack, LK_pack];


(*Single-step tactics*)

val set_step_tac = step_tac [triv_pack, set_pack];

val setpc_step_tac = step_tac  [triv_pack, set_pack, LK_pack];

end;
end;
