(*  Title: 	HOL/wf
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Well-founded relations: induction and recursion
	 (Compare Suppes p. 197)

Should H be an object-function rather than a meta-function?? But then its
type would have to be a product, not a function!!

31/10/89: Adapted to use Pi and Sigma types in construction of wfrec - mc.
*)


goals_limit := 2;

val wf_const_decs = 
  [ (["wf_rel"],  [Atype,Aterm]--->Aform),
    (["restrict"],  [Atype,Aterm,Aterm]--->Atype),
    (["funof"],  [Atype,Aterm-->Atype,Aterm,Aterm,Aterm]--->Aterm),
    (["recfun"], [Atype,Aterm-->Atype,Aterm,[Aterm,Aterm]--->Aterm,Aterm,Aterm]
			---> Aform),
    (["rec_union","rec_limit"], 
	[Atype,Aterm-->Atype,Aterm,[Aterm,Aterm]--->Aterm,Aterm]
			---> Aterm),
    (["wfrec"], [Atype,Aterm-->Atype,Aterm,[Aterm,Aterm]--->Aterm,Aterm] 
			---> Aterm) ];


val wf_thy = extend_theory tarski_thy  "wf" 
    ([], wf_const_decs)

  [ ("wf_rel_def", (*R is a well-founded relation over A*)
     "wf_rel(A,R) == ALL S:A->bool. \
\	(ALL x:A. (ALL y:A. <y,x> <: R --> y <: S) --> x<:S) \
\	--> (ALL x:A. x<:S)"),
 
    ("restrict_def", (*restriction of type A below x in R*)
     "restrict(A,R,x) == {y: A. <y,x> <: R}"),

    ("funof_def",   (*convert S to function from A to B below x in R*)
     "funof(A,B,R,S,x) == lam y:restrict(A,R,x). PICK z:B(y). <y,z> <: S"),


    ("recfun_def",  (*S is graph of function from A to B below x*)
     "recfun(A,B,R,H,S,a) == ALL x:A. \
\	    (<x,a> <: idtranclose(A,R)  -->  (EX z:B(x). <x,z> <: S))  \
\         & (ALL z:B(x). <x,z> <: S  -->  <x,a> <: idtranclose(A,R)  \
\	         	      &  (z = H(x, funof(A,B,R,S,x)) : B(x)) )"),

    ("rec_union_def",  (*union of all recfuns below a*)
     "rec_union(A,B,R,H,a) == (lam z:Sigma(A,B). term( \
\         EX S:Sigma(A,B)->bool. \
\         (EX b:A. <b,a>  <: R & recfun(A,B,R,H,S,b)) & z <: S))"),

    ("rec_limit_def",  (*union of all recfuns below a*)
     "rec_limit(A,B,R,H,a) == (lam z:Sigma(A,B). term( \
\          z <: rec_union(A,B,R,H,a) | \
\          (z = <a,H(a,funof(A,B,R,rec_union(A,B,R,H,a),a))> : Sigma(A,B))))"),

    ("wfrec_def", 
     "wfrec(A,B,R,H,a) == PICK z:B(a). EX s: Sigma(A,B)->bool. \
\			      recfun(A,B,R,H,s,a) & <a,z> <: s") ];

local val ax = get_axiom wf_thy
in  val wf_rel_def = ax"wf_rel_def"
    and restrict_def = ax"restrict_def"
    and funof_def = ax"funof_def"
    and recfun_def = ax"recfun_def"
    and rec_union_def = ax"rec_union_def"
    and rec_limit_def = ax"rec_limit_def"
    and wfrec_def = ax"wfrec_def"
end;



val wf_rel_ind = prove_goal wf_thy
    "wf_rel(A,R) ==> (!x.[| x:A;  ALL y:A. <y,x> <: R --> P(y) |] ==> P(x)) ==> a: A ==> P(a)"
 (fn asms=>
  [ (res_inst_tac [ ("a","lam x:A. term(P(x))",Aterm) ]  all_elim 1), 
    (resolve_tac (map (rewrite_rule [wf_rel_def]) asms) 1),
    (REPEAT (Class.onestep_tac 1)),
    (resolve_tac asms 1),
    (Class.typechk_tac asms),
    (REPEAT (Class.step_tac [] 1  THEN  Class.typechk_tac asms)) ]);


(*Well-founded induction tactic*)
fun wf_ind_tac sa asms i =
    res_inst_tac [ ("a",sa,Aterm) ] wf_rel_ind i  THEN
    ares_tac asms i  THEN
    Class.typechk_tac asms;


val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool |] ==> wf_rel(A,tranclose(A,R))";
by (rewrite_goals_tac [wf_rel_def]);
by (REPEAT (Pc.onestep_tac 1));
by (resolve_tac [all_elim] 1  THEN  assume_tac 1  THEN  chain_tac 1);
by (wf_ind_tac "kb" asms 1);
by (REPEAT (Pc.onestep_tac 1  ORELSE  eresolve_tac [tranclose_elim] 1));
by (Pc.typechk_tac asms);
by (REPEAT (eresolve_tac [all_elim] 1  THEN  chain_tac 1));
by (REPEAT (Pc.step_tac asms 1));
val tranclose_wf = result();


(*A well-founded relation is anti-reflexive! *)
val asms = goal wf_thy
    "[| wf_rel(A,R);  <a,a> <: R;  a: A |] ==> P";
by (res_inst_tac [ ("P", "<a,a> <: R", Aform) ] mp 1);
by (wf_ind_tac "a" asms 1);
by (REPEAT (Class.step_tac asms 1));
val wf_anti_refl = result();


(*A well-founded relation is anti-symmetric! *)
val asms = goal wf_thy
    "[| wf_rel(A,R);  <a,b> <: R;  <b,a> <: R;  R: A*A->bool;  a: A;  b: A |] ==> P";
by (resolve_tac [tranclose_wf RS wf_anti_refl] 1);
by (DEPTH_FIRST (has_fewer_prems 1)  
	(resolve_tac (asms @ [tranclose_intr1,tranclose_intr2]) 1));
val wf_anti_sym = result();


(** Introduction/elimination rules for "restrict"*)

val restrict_intr = prove_goal wf_thy
    "[| x: A;  <x,a> <: R |] ==> x: restrict(A,R,a)"
 (fn asms=>
  [ (rewrite_goals_tac [restrict_def]),
    (REPEAT (resolve_tac ([subtype_intr]@asms) 1)) ]);

(*For type checking*)
val restrict_elim1 = prove_goal wf_thy
    "x: restrict(A,R,a) ==> x: A"
 (fn asms=>
  [ (resolve_tac (subtype_rules [restrict_def] asms) 1) ]);

val asms = goal wf_thy "x: restrict(A,R,a) ==> ([| x: A;  <x,a> <: R |] ==> P) ==> P";
by (REPEAT (resolve_tac (asms @ subtype_rules [restrict_def] asms) 1));
val restrict_elim = result();


(**** Introduction/Elimination rules for recfun ****)

val asms = goal wf_thy
    "(!x.[| x: A;  <x,a> <: idtranclose(A, R) |] ==> \
\    EX z: B(x). <x, z> <: S) ==> (!x z.[| x: A;  z: B(x);  <x,z> <: S |] ==> \
\    <x,a> <: idtranclose(A, R)) ==> \
\    (!x z.[| x: A;  z: B(x);  <x,z> <: S |] ==> \
\         (z = H(x, funof(A,B,R,S,x)): B(x) )) ==> \
\    recfun(A,B,R,H,S,a)";
by (rewrite_goals_tac [recfun_def]);
by (Pc.fast_tac asms 1);
val recfun_intr = result();


(*Tricky proof because there are two facts of the form <u,v> <: S. *)
val asms = goal wf_thy
    "[| recfun(A,B,R,H,S,a);  <x,z> <: S |] ==> \
\    ([|z=H(x,funof(A,B,R,S,x)):B(x); <x,a> <: idtranclose(A,R) |] ==> P) ==> \
\    [| a: A;  x: A;  z: B(x) |] ==> P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [recfun_def]);
by (resolve_tac asms 1);
by (Pc.best_tac asms);
val recfun_elimS = result();


(*Conclusion implies EX z:B. <x,z> <: S provided we know that
  H(x, funof(A,B,R,S,x)) is defined!  *)
val recfun_elimR = prove_goal wf_thy
    "[| recfun(A,B,R,H,S,a);  <x,a> <: idtranclose(A,R);  a: A;  x: A |] ==> \
\    <x, H(x, funof(A,B,R,S,x))> <: S"
 (fn asms=>
  [ (cut_facts_tac asms 1),
    (rewrite_goals_tac [recfun_def]),
    (REPEAT (eresolve_tac [sym RS subst] 1 ORELSE  Pc.step_tac [] 1)) ]);


val recfun_elimR_exists = prove_goal wf_thy
    "[| recfun(A,B,R,H,S,a);  <x,a> <: idtranclose(A,R);  R: A*A->bool;  \
\    a: A;  x: A |] ==> EX z: B(x). <x, z> <: S"
 (fn asms=>
  [ (cut_facts_tac asms 1),
    (rewrite_goals_tac [recfun_def]),
    (Class.fast_tac asms 1) ]);


(*For proving subgoals involving idtranclose.  Given i tries to prove
  ALL subgoals >=i *)
fun idtranclose_tac asms i = 
    BEST_FIRST (has_fewer_prems i, size_of_state) 
	((assume_tac i  APPEND
	   resolve_tac (asms @ [idtranclose_intr1,idtranclose_trans]) i)
      THEN Class.typechk_tac asms);


(*For showing that funof is defined.  A bit odd, but it encapsulates
  the transitivity reasoning in one place. *)
val recfun_elimR_trans = prove_goal wf_thy
    "[| recfun(A,B,R,H,S,a);  <y,x> <: R;  <x,a> <: idtranclose(A,R);  R: A*A->bool;  a: A;  x: A;  y: A |] ==> EX z: B(y). <y, z> <: S"
 (fn asms=>
  [ (resolve_tac [recfun_elimR_exists] 1),
    (resolve_tac asms 1),
    (idtranclose_tac asms 1) ]);


val funof_recfun_type = prove_goal wf_thy
    "[| recfun(A,B,R,H,S,a);  <x,a> <: idtranclose(A,R);  R: A*A->bool;  a: A;  x: A |] ==> funof(A,B,R,S,x) : Pi(restrict(A,R,x),B)"
 (fn asms=>
  [ (rewrite_goals_tac [funof_def]),
    (REPEAT (eresolve_tac [restrict_elim] 1 
	 ORELSE  ares_tac ([recfun_elimR_trans]@asms@type_rls) 1)) ]);


(*MAIN LEMMA.  Uniqueness of 'truncations' of the recursive function.
  Proved by wf induction.  Unfold funof(S,a)`x to H(x, funof(S,x))
  and funof(T,a)`x to H(x, funof(T,x)).  Then funof(S,x)=funof(T,x)
  by induction hypothesis. *)
val asms = goal wf_thy 
    "[| wf_rel(A,R);  recfun(A,B,R,H,S,sx);  recfun(A,B,R,H,T,tx);  \
\       R: A*A->bool;  a: A;  sx: A;  tx: A |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    (<a,sx> <: idtranclose(A,R)  &  <a,tx> <: idtranclose(A,R)) --> \
\       (ALL z:B(a). <a,z> <: S <-> <a,z> <: T)";
by (res_inst_tac [ ("a","a",Aterm) ] wf_rel_ind 1);
by (REPEAT (ares_tac (asms@[imp_intr]) 1));
by (subgoal_tac 
    "funof(A,B,R,S,ka) = funof(A,B,R,T,ka): Pi(restrict(A,R,ka),B)" 1);
by (REPEAT_FIRST (ares_tac asms  ORELSE'  Pc.onestep_tac
	ORELSE' eresolve_tac (reslist(asms,1, recfun_elimS))));
by (REPEAT_FIRST (eres_inst_tac [ ("a","kb",Aterm) ] subst));
by (eresolve_tac [sym RS subst] 2);
by (eresolve_tac [subst] 1);
by (REPEAT_FIRST (ares_tac (asms @ reslist(asms,1, recfun_elimR)) ));
by (rewrite_goals_tac [funof_def]);
by (resolve_tac [Lambda_congr] 1);
by (eresolve_tac [restrict_elim] 1);
(*transitivity: 2 instances*)
by (subgoal_tac "<kb,sx> <: idtranclose(A,R)" 1);
by (subgoal_tac "<kb,tx> <: idtranclose(A,R)" 1);
by (idtranclose_tac asms 2);
by (Pc.fast_tac ([Pick_congr,recfun_elimR_exists] @ asms) 1);
val recfun_unique = result(); 


(*Order of assumptions is carefully chosen to suit eresolve_tac --
  this instantiates S and T. *)
val asms = goal wf_thy 
    "[| <a,b> <: T;  recfun(A,B,R,H,S,sx);  recfun(A,B,R,H,T,tx);  \
\     <a,sx> <: idtranclose(A,R);  wf_rel(A,R);  R: A*A->bool;  \
\     a: A;  sx: A;  tx: A;  b: B(a) |] ==> \
\   (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\   <a,b> <: S";
by (resolve_tac [imp_elim] 1);
by (res_inst_tac [ ("sx","sx",Aterm), ("tx","tx",Aterm) ] recfun_unique 1);
by (REPEAT (ares_tac (asms@[conj_intr]) 1));
by (resolve_tac [recfun_elimS] 1  THEN  assume_tac 3);
by (REPEAT_FIRST (ares_tac asms));
by (Pc.fast_tac asms 1);
val recfun_inclusion = result(); 



(*** Properties of rec_union ***)

val asms = goal wf_thy
    "rec_union(A,B,R,H,a) : Sigma(A,B)-> bool";
by (rewrite_goals_tac [rec_union_def]);
by (Class.typechk_tac asms);
val rec_union_type = result();


val asms = goal wf_thy
    "[| recfun(A,B,R,H,S,x);  <x,a> <: R;  <u,v> <: S;  S: Sigma(A,B)->bool;  \
\       x: A;  u: A;  v:B(u) |] ==> \
\   <u,v> <: rec_union(A,B,R,H,a)";
by (rewrite_goals_tac [rec_union_def]);
by (resolve_tac [beta_conv RS subst] 1);
by (REPEAT (Class.step_tac asms 1));
val rec_union_intr = result();


val asms = goal wf_thy
    "[| ALL y: A. <y, a> <: R --> \
\	(EX s: Sigma(A,B)->bool. recfun(A,B,R,H,s,y));  \
\       <y,x> <: idtranclose(A,R);  <x,a> <: R;  R: A*A->bool;  \
\       a: A;  x: A;  y: A |] ==> \
\    EX z: B(y). <y,z> <: rec_union(A,B,R,H,a)";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [rec_union_def]);
by (eresolve_tac [all_elim] 1);
by (REPEAT (Pc.onestep_tac 1));
by (forwards_tac recfun_elimR_exists 1);
by (resolve_tac [beta_conv RS subst] 1);
by (REPEAT (ares_tac type_rls 1));
by (REPEAT (Class.step_tac [] 1));
val rec_union_exists = result();


val asms = goal wf_thy
    "[| ALL y: A. <y, a> <: R --> \
\	(EX s: Sigma(A,B)->bool. recfun(A,B,R,H,s,y));  \
\       R: A*A->bool;  a: A |] ==> \
\    funof(A, B, R, rec_union(A,B,R,H,a), a) : Pi(restrict(A,R,a),B)";
by (rewrite_goals_tac [funof_def]);
by (REPEAT (eresolve_tac [restrict_elim] 1 
      ORELSE  ares_tac ([idtranclose_refl,rec_union_exists]@asms@type_rls) 1));
val funof_rec_union_type = result();


val asms = goal wf_thy
    "<x,z> <: rec_union(A,B,R,H,a) ==> \
\    (!s y.[| s: Sigma(A,B) -> bool;  y: A;  <x,z> <: s;  <y,a> <: R;  \
\          recfun(A,B,R,H,s,y) |] ==> P) ==> \
\    [| R: A*A->bool;  a: A;  x: A;  z: B(x) |] ==> P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [rec_union_def]);
by (resolve_tac [class_elim] 1);
by (resolve_tac [beta_conv RS sym RS subst] 1 THEN assume_tac 3);
by (REPEAT (ares_tac type_rls 1));
by (REPEAT (eresolve_tac [form_elim2,exists_elim,conj_elim] 1));
by (resolve_tac asms 1);
by (assume_tac 5);
by (REPEAT (ares_tac type_rls 1));
val rec_union_elim = result();


val asms = goal wf_thy
    "[| <x,z> <: rec_union(A,B,R,H,a);  R: A*A->bool;  \
\       a: A;  x: A;  z: B(x) |] ==> \
\    <x,a> <: idtranclose(A,R)";
by (resolve_tac (reslist(asms,1,rec_union_elim)) 1);
by (resolve_tac [idtranclose_intr2] 1);
by (eresolve_tac [recfun_elimS] 1);
by (REPEAT (ares_tac asms 1));
val rec_union_elim_R = result();



val asms = goal wf_thy
    "[| recfun(A,B,R,H,S,x);  wf_rel(A,R);  <y,x> <: idtranclose(A,R);  \
\       <x,a> <: R;  R: A*A->bool;  S: Sigma(A,B)->bool;  \
\       a: A;  x: A;  y: A |] ==> \
\    (!x f.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    funof(A,B,R,S,y) = \
\           funof(A,B,R, rec_limit(A,B,R,H,a),  y) : Pi(restrict(A,R,y),B)";
by (rewrite_goals_tac [funof_def,rec_limit_def]);
by (resolve_tac [Lambda_congr] 1);
by (eresolve_tac [restrict_elim] 1);
by (resolve_tac [Pick_congr] 1);
by (DEPTH_SOLVE_1 (ares_tac (asms @ 
	[recfun_elimR_exists,idtranclose_intr1,idtranclose_trans]) 2));
by (REPEAT(Class.step_tac([rec_union_intr,rec_union_type]@asms) 1));
by (eresolve_tac [rec_union_elim] 1  THEN  Class.typechk_tac asms);
by (eresolve_tac [recfun_inclusion] 1);
by (REPEAT (ares_tac asms 1));
by (resolve_tac [idtranclose_trans] 1);
by (REPEAT (ares_tac (asms @ [idtranclose_intr1]) 1));
by (eresolve_tac [pair_inject] 1);
by (res_inst_tac [ ("a","ka",Aterm) ]  wf_anti_refl 1);
by (resolve_tac [tranclose_wf] 1  THEN  REPEAT (ares_tac asms 1));
by (resolve_tac [tranclose_trans] 1);
by (eresolve_tac [subst] 2);
by (resolve_tac (reslist(asms,1,tranclose_intr_id)) 2);
by (REPEAT (ares_tac (asms @ [pair_type,tranclose_intr1]) 1));
val funof_equality = result();


(*Another consequence of rec_union*)
val asms = goal wf_thy
    "[| <x,z> <: rec_union(A,B,R,H,a);  \
\       wf_rel(A,R);  R: A*A->bool;  a: A;  x: A;  z: B(x) |] ==> \
\   (!x f.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\   z = H(x, funof(A,B,R, rec_limit(A,B,R,H,a), x)) : B(x)";
by (resolve_tac (reslist(asms,1,rec_union_elim)) 1);
by (resolve_tac [funof_equality RS sym RS subst] 1);
by (REPEAT (ares_tac asms 1  ORELSE eresolve_tac [recfun_elimS] 1));
val rec_union_elim_equals = result();


val asms = goal wf_thy
    "[| ALL y: A. <y, a> <: R --> \
\	(EX s: Sigma(A,B)->bool. recfun(A,B,R,H,s,y));  wf_rel(A,R);  \
\      R: A*A->bool;  a: A |] ==> \
\    (!x f.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    funof(A,B,R, rec_union(A,B,R,H,a), a) = \
\	  funof(A,B,R, rec_limit(A,B,R,H,a), a) : Pi(restrict(A,R,a),B)";
by (rewrite_goals_tac [funof_def,rec_limit_def]);
by (resolve_tac [Lambda_congr] 1);
by (eresolve_tac [restrict_elim] 1);
by (resolve_tac [Pick_congr] 1);
by (REPEAT (ares_tac (asms @ [rec_union_exists,idtranclose_refl]) 2));
by (resolve_tac (reslist(asms,1,all_elim)) 1);
by (resolve_tac [beta_conv RS subst] 1);
by (REPEAT (ares_tac type_rls 1));
by (REPEAT(Class.step_tac([rec_union_intr,rec_union_type] @ asms) 1));
by (eresolve_tac [pair_inject] 1);
by (resolve_tac [wf_anti_refl] 1);
by (REPEAT (ares_tac (asms @ [pair_type]) 1 ORELSE eresolve_tac [subst] 1));
val funof_equality_uu = result();


(*** Rules for rec_limit ***)

val asms = goal wf_thy
    "rec_limit(A,B,R,H,a): Sigma(A,B) -> bool";
by (rewrite_goals_tac [rec_limit_def]);
by (Class.typechk_tac asms);
val rec_limit_type = result();

val asms = goal wf_thy
    "[| ~ <x,y> <: rec_union(A,B,R,H,a) ==>  \
\	  <x,y> = <a,H(a,funof(A,B,R,rec_union(A,B,R,H,a),a))> : Sigma(A,B); \
\	x:A;  y:B(x) |] ==> <x,y> <: rec_limit(A,B,R,H,a)";
by (rewrite_goals_tac [rec_limit_def]);
by (resolve_tac [beta_conv RS subst] 1);
by (REPEAT (ares_tac (type_rls) 1));
by (REPEAT (Class.step_tac (asms @ [rec_union_type]) 1));
val rec_limit_intr = result();

val asms = goal wf_thy
    "[| <x,z> <: rec_limit(A,B,R,H,a); \
\   	<x,z> <: rec_union(A,B,R,H,a) ==> P;  \
\   	[| x = a: A; \
\	   z = H(a, funof(A,B,R, rec_union(A,B,R,H,a),a)): B(x) |]  ==> P; \
\	x: A;  z: B(x) |] ==>    P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [rec_limit_def]);
by (resolve_tac [mp] 1 THEN assume_tac 2);
by (resolve_tac [beta_conv RS subst] 1);
by (REPEAT (ares_tac type_rls 1));
by (REPEAT (eresolve_tac (asms@[pair_inject]) 1 
	ORELSE  Class.step_tac [rec_union_type] 1));
val rec_limit_elim = result();

val asms = goal HOL_Rule.thy
    "(!u v.[| u: A;  v: B(u) |] ==> s(u,v) : S(u,v)) ==> \
\    [| a = a' : A;  b = b' : B(a) |] ==> \
\    s(a,b) = s(a',b') : S(a,b)";
by (res_subst_tac asms);
val subst2_sig = result();

val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool;  a: A |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    EX s: Sigma(A,B)->bool. recfun(A,B,R,H,s,a)";
by (wf_ind_tac "a" asms 1);
by (res_inst_tac[("a", "rec_limit(A,B,R,H,ka)", Aterm)] exists_intr 1);
by (resolve_tac [recfun_intr] 1);
by (eresolve_tac [idtranclose_elim] 1);
by (REPEAT (FIRSTGOAL (eresolve_tac 
	 	       [rec_limit_elim,rec_union_elim_R,
			rec_union_elim_equals,subst]
	ORELSE' resolve_tac (asms@[idtranclose_refl])))
	THEN Class.typechk_tac (asms@[rec_limit_type]));
by (res_inst_tac [("s","H",[Aterm,Aterm]--->Aterm)] subst2_sig 3  THEN 
	REPEAT (ares_tac ([refl,funof_equality_uu]@asms) 3));
by (resolve_tac [exists_intr] 1);
by (REPEAT (Class.step_tac ([refl,rec_limit_intr,rec_union_type,
			     funof_rec_union_type]@asms) 1));
by (REPEAT (eresolve_tac [recfun_elimR,funof_recfun_type] 1
	ORELSE Class.step_tac ([rec_union_intr]@asms) 1));
val recfun_complete = result();


val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool;  a: A |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    EX z:B(a). EX s:Sigma(A,B)->bool. recfun(A,B,R,H,s,a) & <a,z> <: s";
by (resolve_tac [exists_elim] 1); 
by (res_inst_tac[("H", "H", [Aterm,Aterm]--->Aterm)] recfun_complete 1);
by (REPEAT (ares_tac asms 1));
by (resolve_tac [recfun_elimR_exists RS exists_elim] 1);
by (REPEAT (ares_tac (asms@[idtranclose_refl]) 1));
by (Pc.fast_tac asms 1);
val wfrec_existence = result();


(*A key result: the typing rule for wfrec*)
val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool;  a: A |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    wfrec(A,B,R,H,a): B(a)";
by (rewrite_goals_tac [wfrec_def]);
by (REPEAT (ares_tac (asms@[Pick_type,wfrec_existence]) 1));
val wfrec_type = result();


val asms = goal wf_thy
    "[| recfun(A,B,R,H,S,a);  wf_rel(A,R);  R: A*A->bool;  \
\       S: Sigma(A,B)->bool;  a: A |] ==> \
\    (!x f.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    funof(A,B,R,S,a) = \
\	  (lam x: restrict(A,R,a). wfrec(A,B,R,H,x)) : Pi(restrict(A,R,a),B)";
by (rewrite_goals_tac [funof_def,wfrec_def]);
by (resolve_tac [Lambda_congr] 1);
by (eresolve_tac [restrict_elim] 1);
by (resolve_tac [Pick_congr] 1);
by (REPEAT (ares_tac (asms @ 
	[recfun_elimR_exists,idtranclose_intr1]) 2));
by (resolve_tac [exists_elim] 1); 
by (res_inst_tac[("H","H",[Aterm,Aterm]--->Aterm)] wfrec_existence 1);
by (REPEAT_FIRST (ares_tac asms));
by (REPEAT_FIRST (Class.step_tac asms));
by (REPEAT (ares_tac ([idtranclose_refl,idtranclose_intr1]@asms) 1  
    	ORELSE eresolve_tac [recfun_inclusion] 1));
val funof_equals_wfrec = result();


(*Lemma for wfrec_conv*)
val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool;  a: A;  b : B(a) |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    (b = H(a, lam x: restrict(A,R,a). wfrec(A,B,R,H,x)): B(a)) <-> \
\         (EX s: Sigma(A,B)->bool.recfun(A,B,R,H,s,a) & <a,b> <: s)";
by (resolve_tac [iff_intr] 1); 
by (resolve_tac [exists_elim] 1); 
by (res_inst_tac[("H","H",[Aterm,Aterm]--->Aterm)] wfrec_existence 1);
by (REPEAT_FIRST (Pc.step_tac asms));
by (resolve_tac [recfun_elimS] 2); 
by (resolve_tac [recfun_elimS] 1); 
by (REPEAT_FIRST (ares_tac asms));
by (REPEAT_FIRST (eres_inst_tac [ ("a","b",Aterm) ] subst));
by (res_inst_tac[("c","ka",Aterm)] subst 1  THEN  assume_tac 2);
by (eresolve_tac [subst] 1); 
by (resolve_tac [sym] 1); 
by (REPEAT (res_inst_tac [("s","H",[Aterm,Aterm]--->Aterm)] subst2_sig 1  THEN 
	REPEAT (ares_tac ([refl,funof_equals_wfrec]@asms) 1)));
val wfrec_conv_lemma = result();


(*Conversion rule for wfrec!*)
val asms = goal wf_thy
    "[| wf_rel(A,R);  R: A*A->bool;  a: A |] ==> \
\    (!f x.[| x: A;  f : Pi(restrict(A,R,x),B) |] ==> H(x,f): B(x)) ==> \
\    wfrec(A,B,R,H,a) = H(a,lam x:restrict(A,R,a).wfrec(A,B,R,H,x)) : B(a)";
by (rewrite_goals_tac [wfrec_def]);
by (resolve_tac [Pick_iff_equals] 1); 
by (fold_tac [wfrec_def]);
by (REPEAT (ares_tac ([wfrec_type,wfrec_conv_lemma]@asms@type_rls) 1
	ORELSE eresolve_tac [restrict_elim1] 1));
val wfrec_conv = result();

