(*  Title: 	FOL/int-prover
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Intuitionistic first-order logic tactics
*)


signature INT_PROVER = 
sig
  structure Int_Rule: INT_RULE
  local open Int_Rule.Pure
  in
  val all_elim: thm
  val all_imp_elim: thm
  val conj_elim: thm
  val conj_imp_elim: thm
  val contr: thm
  val disj_imp_elim: thm
  val eq_mp_tac: int -> tactic
  val exists_imp_elim: thm
  val iff_elim: thm
  val iff_imp_elim: thm
  val iff_intr: thm
  val imp_elim: thm
  val imp_imp_elim: thm
  val mp_tac: int -> tactic
  val not_elim: thm
  val not_intr: thm
  val not_to_imp: thm
  val pc_tac: int -> tactic
  val safe_brls: (bool * thm) list
  val safestep_tac: int -> tactic
  val safe_tac: int -> tactic
  val step_tac: int -> tactic
  val unsafe_brls: (bool * thm) list
end;
end;


functor Int_ProverFun(Int_Rule: INT_RULE) : INT_PROVER   = 
struct
structure Int_Rule = Int_Rule;
local open  Int_Rule  Int_Rule.Pure  Int_Rule.FOL_Syntax
in

(** Sequent-style elimination rules for & --> and ALL **)

val conj_elim = prove_goal Int_Rule.thy 
    "[| P&Q; [| P; Q |] ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (resolve_tac prems 1  ORELSE
               (resolve_tac [conjunct1, conjunct2] 1 THEN
                resolve_tac prems 1))) ]);

val imp_elim = prove_goal Int_Rule.thy 
    "[| P-->Q;  P;  Q ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems@[mp]) 1)) ]);

val all_elim = prove_goal Int_Rule.thy 
    "[| ALL x.P(x); P(a) ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems@[spec]) 1)) ]);


(*** Negation rules -- translate between ~P and P-->False ***)

val contr = prove_goal Int_Rule.thy 
   "[| ~P;  P |] ==> False"
 (fn prems=>
  [ (resolve_tac [mp] 1),
    (fold_tac [not_def]),
    (REPEAT (resolve_tac prems 1)) ]);

val not_intr = prove_goal Int_Rule.thy 
   "(P ==> False) ==> ~P"
 (fn prems=>
  [ (rewrite_goals_tac [not_def]),
    (REPEAT (ares_tac (prems@[imp_intr]) 1)) ]);

val not_elim = prove_goal Int_Rule.thy 
    "[| ~P;  P |] ==> R"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems@[contr,False_elim]) 1)) ]);

val not_to_imp = prove_goal Int_Rule.thy 
    "[| ~P;  (P-->False) ==> Q |] ==> Q"
 (fn prems=>
  [ (REPEAT (ares_tac (prems@[imp_intr,contr]) 1)) ]);



(** If-and-only-if **)

val iff_intr = prove_goal Int_Rule.thy 
   "[| P ==> Q;  Q ==> P |] ==> P<->Q"
 (fn prems=>
  [ (rewrite_goals_tac [iff_def]),
    (REPEAT (ares_tac (prems@[conj_intr, imp_intr]) 1)) ]);


(*Observe use of rewrite_rule to unfold "<->" in meta-assumptions (prems) *)
val iff_elim = prove_goal Int_Rule.thy 
    "[| P <-> Q;  [| P-->Q; Q-->P |] ==> R |] ==> R"
 (fn prems =>
  [ (resolve_tac [conj_elim] 1),
    (REPEAT (ares_tac (map (rewrite_rule [iff_def]) prems) 1)) ]);


(*Simplifications of assumed implications

MOST SHOULD BE DELETED.  INSTEAD TRY IMPLICATION RULE WITH DEPTH BOUND. *)


val conj_imp_elim = prove_goal Int_Rule.thy 
    "[| (P&Q)-->S;  P-->(Q-->S) ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (conj_intr::imp_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);


val disj_imp_elim = prove_goal Int_Rule.thy 
    "[| (P|Q)-->S;  [| P-->S; Q-->S |] ==> R |] ==> R"
  (fn prems=>
 [ (REPEAT (assume_tac 1 ORELSE  
            biresolve_tac [ (true,disj_intr1), (true,disj_intr2) ] 1  ORELSE 
            resolve_tac (imp_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);


(*Simplifies the implication.  Classical version is stronger. 
  Still UNSAFE since Q must be provable.  *)
val imp_imp_elim = prove_goal Int_Rule.thy 
     "[| (P-->Q)-->S;  [| P; Q-->S |] ==> Q;  S ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (imp_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);


(*Simplifies the implication.   UNSAFE.  *)
val iff_imp_elim = prove_goal Int_Rule.thy 
    "[| (P<->Q)-->S;  [| P; Q-->S |] ==> Q;  [| Q; P-->S |] ==> P;  \
\       S ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (iff_intr::imp_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);


(*What if (ALL x.~ ~ P(x))-->(~ ~ ALL x.P(x)) is an assumption? UNSAFE*)
val all_imp_elim = prove_goal Int_Rule.thy 
    "[| (ALL x.P(x))-->S;  !x.P(x);  S ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (all_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);


(*Probably not safe: should not thin the assumption.*)
val exists_imp_elim = prove_goal Int_Rule.thy 
    "[| (EX x.P(x))-->S;  P(a)-->S ==> R |] ==> R"
 (fn prems=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (imp_intr::exists_intr::prems) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac prems 1))) ]);



val safe_brls = sort lessb 
    [ (true,False_elim), (true,asm_rl), 
      (false,imp_intr), (false,not_intr), (false,all_intr),
      (true,conj_elim), (true,exists_elim),
      (false,conj_intr), (true,conj_imp_elim),
      (true,disj_imp_elim), (true,exists_imp_elim),
      (true,disj_elim), (false,iff_intr), (true,iff_elim), (true,not_to_imp) ];


val unsafe_brls =
    [ (false,disj_intr1), (false,disj_intr2), (false,exists_intr), 
      (true,all_elim), (true,imp_imp_elim), (true,iff_imp_elim),
      (true,all_imp_elim), (true,imp_elim) ];


(*Finds P-->Q and P in the assumptions, replaces implication by Q *)
fun mp_tac i = eresolve_tac [imp_elim,not_elim] i  THEN  assume_tac i;

(*Like mp_tac but instantiates no variables*)
fun eq_mp_tac i = eresolve_tac [imp_elim,not_elim] i  THEN  eq_assume_tac i;


(*0 subgoals vs 1 or more: the p in safep is for positive*)
val (safe0_brls, safep_brls) =
    partition (apl(0,op=) o subgoals_of_brl) safe_brls;

(*One notionally safe step. Not really safe due to mp_tac. *)
val safestep_tac =
    biresolve_tac safe0_brls  ORELSE'  mp_tac  ORELSE'
    (DETERM o biresolve_tac safep_brls);

(*Backtracks with safestep_tac.to solve one goal. *)
val safe_tac = DEPTH_SOLVE_1 o safestep_tac;

(*One safe or unsafe step. *)
val step_tac = safestep_tac  ORELSE'  biresolve_tac unsafe_brls;

(*Backtracks with step_tac.to solve one goal. *)
val pc_tac = DEPTH_SOLVE_1 o step_tac;

end;
end;

