(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* trace_gram.ml Syntax for trace requests                      	 *)
(*		 Olivier Jehl    					 *)
(*		 Pierre Weis						 *)
 

#standard arith true;;
#fast arith true;;

system module trace_syntax;;

type function_traced_syntax =
  {Function_name : ML;
   Arg_list : symbol list;
   Command_list : ML command list}

and 'a command=
   Trace_command of 'a trace_command list
  |Print_command of 'a print_command list
  |Before_command of 'a pre_postlude list
  |After_command of 'a pre_postlude list

and trace_pattern =
   Enumerate of string list
  |Ellipsis of string list * string list

and 'a trace_command == symbol list * 'a trace_predicate * 'a list 

and 'a print_command == symbol list * 'a trace_printer 

and 'a pre_postlude == symbol list * 'a trace_prelude_postlude

and reference == (int * symbol) list 

and arg_table == (string * gtype * (int list)) list

and symbol = Letter of string | Three_points | Extern_pat of ML

and 'a order =   { trace : 'a trace_predicate vect;
                   print : 'a trace_printer vect;
                   from_function : 'a list vect;
                   after : 'a trace_prelude_postlude vect;
                   before : 'a trace_prelude_postlude vect}
;;

(* error counter for ``parsing errors'' *)
let error_counter = ref 0;;

let reset_err() = error_counter:=0;;

let Err()=incr error_counter;;

(******************)
(* Trace_printers *)
(******************)

(* to print a pattern used in the tracing syntax *)
(*************************************************)
(*|
Value print_trace_pattern : (symbol list -> unit)
 CAML_system{close_box,do_list,open_hvbox,print_ML,print_space,
             print_string}
|*)
let print_trace_pattern pat =
    <:Pretty< (print Letter s -> s ; \- 
                   | Three_points -> "..."; \-
                   | Extern_pat e -> {print_ML e} (())) pat>>;;

let  print_trace_pattern_list pat_list =
     <:Pretty< [<hv> * (print pat -> print_trace_pattern pat; \-) pat_list]>>;;

(* to pretty_print a command of trace on a function *)
(****************************************************)
(*|
Value print_function : (string * symbol list -> unit)
{print_trace_pattern},
 CAML_system{close_box,open_hvbox,print_space,print_string}
|*)
(*****
let print_function (expr_Function_name, expr_Arg_list)=
    <:Pretty<[ <hv 2> expr_Function_name; \-;
                      print_trace_pattern expr_Arg_list ]>>;;
******)

(* to make the reference table *)
(*******************************)
(*|
Value bad_pattern : (symbol -> symbol list -> unit)
{print_trace_pattern},
 CAML_system{Err,close_box,force_newline,open_hovbox,print_ML,
             print_space,print_string}
|*)
let bad_pattern s pat_list =
                    <:Pretty< [ "the"; \-; "pattern"; \-;
                              print_trace_pattern_list pat_list; \-;
                              "contains"; \-; 
                              print_trace_pattern s;
                              "more"; \-; "than"; \-; "once" ]; \\ >>  ;
                    Err();;



(*|
Value mk_reference : (symbol list -> reference)
{bad_pattern},
 CAML_system{map_i,mem,pair}
|*)
let mk_reference pat = 
    let rec distinct =
        function 
           (Letter s as x) :: t 
            -> if mem x t then bad_pattern x pat;
               distinct t
          |(Extern_pat e as x) :: t
            -> if mem x t then bad_pattern x pat;
               distinct t
          | x :: t -> distinct t
          | [] -> ()
    in distinct pat ;map_i pair 0 pat ;;


(* to get the number in the reference list of a symbol *)
(* fail with "wrong pattern"                                *)
(*******************************************************)
(*|
Value num_of_symbol : (reference -> symbol -> int)
 CAML_system{Err,force_newline,print_ML,print_string,rev_assoc}
|*)
let num_of_symbol (reference:reference) s =
    begin try rev_assoc s reference with 
          failure "find" 
                -> <:Pretty< "unbound symbol ";
                              (print  
                                   Letter s -> s
                                 | Extern_pat e -> {^print_ML e ()^}
                                 | _ -> "...") s; \\>>;
                   Err();raise (failure "wrong pattern")
    end;;

(* return the list of numbers of arguments given by a pattern *)
(* fail with "wrong pattern"                                        *)
(**************************************************************)
(*|
Value pattern_to_num_list : (reference -> symbol list -> int list)
{mk_reference,num_of_symbol},
 CAML_system{@,distinct,interval,it_list,length,map,mem,system_error}
|*)
let pattern_to_num_list reference pat =
     if distinct pat
      then
        let rec before_points result  = function
            Three_points :: t -> result , map (num_of_symbol reference) t, true
          | x :: t -> before_points ((num_of_symbol reference x)::result) t
          | [] -> result , [],false
        in let union_trace = 
          it_list (fun l x  -> if mem x l then l else x::l) []
        in union_trace(
          match before_points [] pat with
            ([],[],true) -> map fst reference
          | (x::t,[],true) -> t @ (interval x ((length reference) - 1))
          | ([],x::l,true) -> (interval 0 x)@l
          | (x::t,y::l,true) -> (interval x y) @ t @ l
          | (m,n,false) -> m @ n)
      else
         (* to have the message of multiple arguments *)
         (mk_reference pat; system_error "trace")
;;

(* to get the name of an argument given by its number *)
(******************************************************)
(*|
Value name_of_arg : (reference -> int -> string)
 CAML_system{assoc,system_error}
|*)
let name_of_arg (reference:reference) x =
    match (try assoc x reference with failure _ -> system_error "name_of_arg")
    with Letter x -> x | Three_points -> "..." 
       | _ -> "argument number "^(string_of_int x);;

(*****************************************************************************)
(* to make the list of order for the trace                                   *)
(* these functions don't fail                                                *)
(* in case of error, it does nothing                                         *)
(*****************************************************************************)

(* the postlude order *)
(**********************)
let warning_last_chosen chosen on_obj =
    warning ("two "^chosen^" defined on "^on_obj^": the last is chosen");;

(*|
Value command_to_after :
      (reference -> 'a order -> 'a pre_postlude -> unit)
{name_of_arg,pattern_to_num_list},
 CAML_system{do_list,warning}
|*)
let command_to_after reference result = function
      (pat,Trace_none) -> ()
    | (pat,Trace_prelude_function f) -> 
    let after_set no = 
        if  result.after.(no) <> Trace_none  
          then warning_last_chosen "postludes" (name_of_arg reference no);
        (result.after.(no) <- Trace_prelude_function f)
    in try
           do_list after_set (pattern_to_num_list reference pat)
       with failure "wrong pattern" -> ()
;;

(* the prelude order *)
(*********************)
(*|
Value command_to_before :
      (reference -> 'a order -> 'a pre_postlude -> unit)
{name_of_arg,pattern_to_num_list},
 CAML_system{do_list,warning}
|*)
let command_to_before reference result = function
      (pat,Trace_none) -> ()
    | (pat,Trace_prelude_function f) -> 
    begin try do_list before_set (pattern_to_num_list reference pat)
    with failure "wrong pattern" -> ()
    end
    where before_set no = 
        if  result.before.(no) <> Trace_none  
          then warning_last_chosen "preludes" (name_of_arg reference no);
        (result.before.(no) <- Trace_prelude_function f)
;;

(*the printing order *)
(*********************)
(*|
Value command_to_print :
      (reference -> ML order -> ML print_command -> unit)
{name_of_arg,pattern_to_num_list},
 CAML_system{do_list,warning}
|*)
let command_to_print reference result (pat,p) =
  begin try
    do_list print_set (pattern_to_num_list reference pat)
  with failure "wrong pattern" -> ()
  end
    where print_set no = 
        if result.print.(no) <> Trace_printer <:CAML:MLvar<none>>
         then warning_last_chosen "printers" (name_of_arg reference no);
        result.print.(no) <- p  
;;

(* the tracing order *)
(*********************)

(*|
Value command_to_trace :
      (reference -> ML order -> ML trace_command -> unit)
{name_of_arg,pattern_to_num_list},
 CAML_system{@,do_list,system_error,warning}
|*)
let command_to_trace reference result = function
      pat , Trace_bool false ,  (x::t) -> system_error "trace"
    | pat , p ,  f
             -> begin try
                  let num_list = 
                    pattern_to_num_list reference pat
                  in do_list set_trace num_list;
                     do_list set_from num_list
                with failure "wrong pattern" -> ()
                end

    where set_trace no =
        if result.trace.(no) <> Trace_predicate <:CAML:MLbool<true>>
         then warning_last_chosen "tracing orders" (name_of_arg reference no);
        result.trace.(no) <- p 

    and set_from no =
        let  old_from = result.from_function.(no) 
        in if old_from = []
            then result.from_function.(no) <-  f
            else
              (warning_last_chosen "\"from\" orders" 
                                    (name_of_arg reference no);
               result.from_function.(no) <-  (f@old_from))
;;

(* to build the CAML expression of type 'trace_order' *)
(******************************************************)

(*|
Value arg_order_to_ML : (ML order -> int -> ML -> ML)
 CAML_system{it_list,mkapply,mklist,rev}
|*)
let arg_order_to_ML vect i symbol=
    <:CAML< {Trace_arg_name= #symbol;
             Trace_arg_predicate =
              {^ match vect.trace.(i) with
                  Trace_bool true -> <:CAML<Trace_bool true>>
                | Trace_bool false -> <:CAML<Trace_bool false>>
                | Trace_predicate f -> <:CAML<Trace_predicate (dynamic #f)>>^};
             Trace_arg_printer =
              {^ match vect.print.(i) with
                  Trace_printer_usuel -> <:CAML< Trace_printer_usuel >>
                | Trace_printer p -> <:CAML< Trace_printer (dynamic #p)>> 
                | Trace_printer_no -> <:CAML< Trace_printer_no>>
                | Trace_with_system -> <:CAML< Trace_with_system>> ^};
             Trace_arg_from_functions =
              {^ match vect.from_function.(i) with
                  [] -> <:CAML< [] >>
                | s::sl -> it_list
                            (fun expr s
                               -> <:CAML< #s :: #expr>>)
                            <:CAML<[#s]>>  sl ^};
             Trace_arg_prelude =
              {^ match vect.before.(i) with
                  Trace_prelude_function f
                        -> <:CAML< Trace_prelude_function (dynamic #f) >>
                | Trace_none -> <:CAML< Trace_none >> ^};
             Trace_arg_postlude =
              {^ match vect.after.(i) with
                  Trace_prelude_function f
                        -> <:CAML< Trace_prelude_function (dynamic #f) >>
                | Trace_none -> <:CAML< Trace_none >> ^}} >>
;;

(* to build the CAML expression for the all function *)
(*****************************************************)

(*|
Value function_order_to_ML : (ML -> ML order -> reference -> ML)
{arg_order_to_ML},
 CAML_system{chop_list,rev,rev_assoc,system_error}
|*)
let function_order_to_ML fun_name order_vect reference =
    let r1,r2,r3 =
        begin try
           begin match chop_list ( (rev_assoc Three_points reference) ) 
                                 reference
           with
                  l1,x::l2 -> l1,[x],l2
                | l1,_ -> l1,[],[]
           end
        with failure "find" -> reference, [],[]
        end
    in <:CAML< {Trace_function_name = #fun_name;
                Trace_function_first_args = {^ mk_rec r1 ^};
                Trace_function_ellipsis_args = {^ mk_rec r2 ^};
                Trace_function_last_args = {^ mk_rec r3 ^} } >>
    
    where rec mk_rec =
    function (i , Letter x)::l
                -> <:CAML< {^arg_order_to_ML order_vect i 
                                <:CAML:MLstring<#x>> ^} ::
                           {^mk_rec l^} >>
           | (i, Three_points)::l
                -> <:CAML< {^arg_order_to_ML order_vect i 
                                <:CAML:MLstring<"...">> ^} :: 
                           {^mk_rec l^} >>
           | (i, Extern_pat e)::l
                -> <:CAML< {^arg_order_to_ML order_vect i e ^} ::
                           {^mk_rec l^} >>
           | [] -> <:CAML< [] >>
        
(* Exhaustive | _  -> system_error "trace : order_to_ref_ML" *)
     
;;


(* to verify that the vector is equal to [| a ; a ; ... ; a|] *)
(**************************************************************)
(*|
Value vect_is_replicate : ('a -> 'a vect -> bool)
|*)
let  vect_is_replicate val vect = 
    vect_rec 0
    where rec vect_rec n =    
       n == vect_length vect || (vect.(n) = val && vect_rec(n+1))
;;

(* to build the order's vector from the command list *)
(*****************************************************)
(*|
Value command_to_order : (reference -> ML command list -> ML order)
{command_to_after,command_to_before,command_to_print,command_to_trace,
 vect_is_replicate},
 CAML_system{do_list,length}
|*)
let command_to_order reference  command=
    let result = 
    {trace = (vector (length reference) of
               Trace_predicate <:CAML:MLbool<true>>);
     print = (vector (length reference) of Trace_printer <:CAML:MLvar<none>>);
     from_function =(vector (length reference) of  []);
     after = (vector (length reference) of Trace_none); 
     before = (vector (length reference) of Trace_none)}
    in 
       do_list command_to_order (rev command) ;
       if (vect_is_replicate
            (Trace_predicate<:CAML:MLbool<true>>) result.trace)
        then  result.trace.((length reference) - 1) <- Trace_bool true;
       if (vect_is_replicate
            (Trace_printer <:CAML:MLvar<none>>) result.print)
        then modify_vect (fun x -> Trace_printer_usuel) result.print;
       modify_vect (function Trace_predicate <:CAML:MLbool<true>> ->
                                 Trace_bool false
                            |  x -> x)
                    result.trace;
       modify_vect (function Trace_printer <:CAML:MLvar<none>> ->
                                Trace_printer_no
                            |  x -> x)
                    result.print;
       result

    where 

     command_to_order =
        function
          Trace_command t ->  do_list (command_to_trace reference result) t 
        | Print_command p ->  do_list (command_to_print reference result) p 
        | Before_command b -> do_list (command_to_before reference result) b 
        | After_command a ->  do_list (command_to_after reference result) a 
   
;;

(* to build the order's vector from the concrete symtax *)
(********************************************************)

(*|
Value syntax_to_order : (function_traced_syntax list -> ML)
{command_to_order,function_order_to_ML,mk_reference},
 CAML_system{catch_type_err,enter_system,mkapply,mkraise,mkseq,
             print_string,quit_system,reset_err}
|*)
let syntax_to_order_gen l  =

 let rec rec_syntax_to_order =
  function
   expr:: l ->
     let reference = mk_reference expr.Arg_list in
         <:CAML<
           {^ function_order_to_ML
                expr.Function_name
                (command_to_order reference  expr.Command_list)
                 reference ^} ::
           {^rec_syntax_to_order l ^}>>
   | [] -> <:CAML< [] >>

 in
   reset_err();
   let result = rec_syntax_to_order l in
   if !error_counter>0
      then (message ((string_of_int !error_counter) ^
                     " error"^(if !error_counter>1 then "s" else "")^
                     " in the grammar Trace");
            failwith "trace")
      else result
;;

let syntax_to_order = do_printing_in_error_mode syntax_to_order_gen
;;


(****************************************************************************)
(* the grammar                                                              *)
(****************************************************************************)

(*|

Error in CAML system: TML
Please report it
|*)
grammar for values Trace =

rule entry top = parse system s accept <:CAML<trace_execute #s>>
        | Trace_ident f; Literal ":"; Literal "untrace"
          accept <:CAML<untrace_fun #f>>
        | Literal "untrace" accept <:CAML<untrace()>>

and entry macro = parse system s 
                  accept <:CAML< #s >>

and system = parse syntax s -> do_in_system syntax_to_order s

and syntax =
    parse Trace_function_order f ;
          ( * (parse "and" ; Trace_function_order f -> f)) fl
            -> f::fl

and Trace_function_order =
    parse Trace_ident f ; Pattern p ; Order c
            -> {Function_name = f;
                Arg_list = p;
                Command_list = c}
        | Trace_ident f ; Order c
            -> {Function_name = f;
                Arg_list = [Three_points; Letter "last argument"];
                Command_list = c}

and Pattern =
    parse Literal "all" -> [Three_points]
        | Trace_pattern p -> p

and Trace_pattern =
    parse Symbol s       -> [s]
        | Symbol s ; Trace_pattern p 
                        ->  s::p
        | Literal"."; Literal "."; Literal"."; ( *(parse Symbol s -> s)) l
                        -> Three_points :: l
        
and Symbol = parse Ident s -> Letter s
        | Literal "("; {parse_caml_expr ()} c ; Literal")" -> Extern_pat c


and Order =
    parse Literal ":"; Command_list cl -> cl
        |           -> []


and Command_list =
    parse  Command_list l ; Literal ";" ; Command c -> c::l
        |  Command c -> [c]

and Command =
    parse Literal "trace"  ; Trace_command_list t -> Trace_command t
        | Literal "print" ; Print_command_list p -> Print_command p
        | Literal "before" ; Before_command_list b -> Before_command b
        | Literal "after" ; After_command_list a -> After_command a

and Trace_command_list =
    parse Trace_command_list t ; Literal "," ; Trace_command x -> x::t
        | Trace_command x -> [x]

and Trace_command =
    parse Pattern p ; Trace t -> (p,t)

and Trace =
    parse From f ; Trace_predicate p -> p,f
        | Trace_predicate p ; From f -> p,f
        | Trace_predicate p -> p, []
        | From f -> Trace_bool true,f
        |        -> Trace_bool true, []

and From =
    parse Literal "from" ; (+ (parse Trace_ident s -> s)) f 
           -> let (x,t)=f in (x::t)
(****        | Literal "from" ; Trace_ident s;
           ( * (parse "," ; Trace_ident s -> s)) f 
           -> s::f
*)

and Trace_predicate =
    parse Literal "with" ; Literal "predicate" ; Trace_expression f 
           -> Trace_predicate f
        | Literal "false" 
           -> Trace_bool false

and Print_command_list =
    parse Print_command_list l ; Literal "," ; Print_command p -> p::l
        | Print_command p -> [p]

and Print_command =
    parse Pattern p ; Print x -> (p,x)

and Print =
    parse Literal "with" ; Literal "raw" -> Trace_with_system
        | Literal "with" ; Trace_expression f -> Trace_printer f
        | Literal "never" -> Trace_printer_no
        |  -> Trace_printer_usuel

and Before_command_list = 
    parse Before_command_list l ; Literal "," ; Before b -> b::l
        | Before b -> [b]

and After_command_list =
    parse After_command_list l ; Literal "," ; After b -> b::l
        | After b -> [b]

and Before =
    parse Pattern p ; Literal"do" ; Trace_expression f ->
                       (p,Trace_prelude_function f)
        | Pattern p -> (p,Trace_none)

and After =
    parse Pattern p ; Literal "do" ; Trace_expression f ->
                       (p,Trace_prelude_function f)
        | Pattern p -> (p,Trace_none)
                    
and Trace_ident =
    parse Ident f -> <:CAML:MLstring<#f>>
        | "#"; {parse_caml_expr0()} s -> <:CAML<(#s:string)>>

and Trace_expression =
    parse "#"; Ident f -> <:CAML< ( #(<:CAML:MLvar<#f>>) : string) >>
        | "#"; {parse_caml_expr0()} e -> <:CAML<(#e:string)>>
        | {parse_caml_expr0()} e -> e

;;

let parse_trace = (Trace "top").Parse_raw;;

let parse_trace_macro = (Trace "macro").Parse_raw;;

end module with value parse_trace and parse_trace_macro;;
