(* $Id: c_text.ml,v 8.3 91/06/19 19:46:12 ddr Exp $
 *
 * Rogloglo Toolkit: text widget class
 *
 * $Log:	c_text.ml,v $
 * Revision 8.3  91/06/19  19:46:12  ddr
 * - merge avec zinc 1.4
 * 
 * Revision 8.2  91/06/15  15:52:50  ddr
 * - evolution
 * 
 * Revision 8.1  91/06/15  09:52:15  ddr
 * - merge avec zinc
 * 
 * Revision 7.9  91/06/07  20:14:38  ddr
 * - redistrib
 *)

#standard arith false;;
#fast arith false;;

type text_global_info = {
  tfs     : font_struct;
  gc_text : GC
}

and text_local_info = {
  text_gi         : text_global_info;
  swin            : Window;
  mutable xb      : num;
  mutable yb      : num;
  mutable y_shift : num;
  mutable y_shmax : num;
  mutable txt     : string vect;
  mutable vlin    : num;
  mutable nlin    : num;
  mutable ncol    : num;
  mutable clin    : num;
  mutable ccol    : num;
  mutable mrk_lin : num;
  mutable mrk_col : num;
  mutable nscroll : num
};;

let text_global_info, get_text_global_info = dynamo_global_info
  "text_global_info" (ref None: text_global_info option ref)
and text_local_info, get_text_local_info = dynamo_local_info
  "text_local_info" (ref None: text_local_info option ref)
;;

let text_border = ref 1
and text_band = ref 2
and text_font = ref "*-courier-medium-r-*-14-*"
;;

let ascii_back_space = ascii_code "\b"
and ascii_line_feed = ascii_code "\n"
and ascii_tab = ascii_code "\t"
;;

let lin_col_of_xy li x y =
  let gi = li.text_gi in
  let lin = (y-li.yb) quo gi.tfs.fheight
  and col = (x-li.xb) quo gi.tfs.fwidth in
  (max 0 (min (li.nlin-1) lin), max 0 (min (li.ncol-1) col))

and text_expose_lines =
  let i = ref 0 and j = ref 0
  and buff = ref "" in
fun wid imin imax jmin jmax ->
  let xdm = wid.wid_xd.xdm
  and li = get_text_local_info wid.info in
  let gi = li.text_gi in
  if jmin > 0 & length_string !buff < jmax-jmin then (
    buff := make_string (jmax-jmin) ` `; ()
  );
  i := imin;
  while !i < imax do
    let txt =
      if jmin = 0 then li.txt.(!i)
      else (
        j := jmin;
        while !j < jmax do
          set_nth_ascii(!j-jmin, !buff, nth_ascii(!j, li.txt.(!i)));
          incr j
        done;
        !buff
      ) in
    XDrawImageString(xdm.dpy, li.swin, gi.gc_text,
      CINT(li.xb+jmin*gi.tfs.fwidth),
      CINT(li.yb+gi.tfs.fheight*!i+gi.tfs.ascent),
      txt, CINT(jmax-jmin)
    );
    incr i
  done
;;

let text_scroll =
  let i = ref 0 and j = ref 0 in
fun wid ->
  let xdm = wid.wid_xd.xdm
  and li = get_text_local_info wid.info in
  let gi = li.text_gi
  and b = CINT li.xb in
  XCopyArea(xdm.dpy, li.swin, li.swin, gi.gc_text,
    b, CINT(li.y_shift+gi.tfs.fheight),
    CINT(li.ncol*gi.tfs.fwidth),
    CINT(wid.height-gi.tfs.fheight),
    b, CINT li.y_shift
  );
  li.nscroll <- li.nscroll+1;
  XClearArea(xdm.dpy, li.swin,
    b, CINT(li.y_shift+wid.height-gi.tfs.fheight),
    Zero_Int, Zero_Int, Zero_Int
  );
  i := 0;
  while !i < li.nlin-1 do
    replace_string li.txt.(!i) li.txt.(!i+1) 0;
    incr i
  done;
  let ll = li.txt.(li.nlin-1) in
  let sp = ascii_code " " in
  j := 0;
  while !j < li.ncol do set_nth_ascii(!j, ll, sp); incr j done;
  let (imin, _) = lin_col_of_xy li 0 (li.y_shift+wid.height-gi.tfs.fheight)
  and (imax, _) = lin_col_of_xy li 0 (li.y_shift+wid.height-1) in
  let imax = min (imax+1) (li.nlin-1) in
  let imin = min imin imax in
  text_expose_lines wid imin imax 0 li.ncol
;;

let text_home wid =
  let li = get_text_local_info wid.info in
  li.clin <- li.nlin - li.vlin;
  li.ccol <- 0;
  ()

and text_goto(wid, lin, col) =
  let li = get_text_local_info wid.info in
  li.clin <- max 0 (min (li.nlin-1) lin);
  li.ccol <- max 0 (min (li.ncol-1) col);
  ()

and text_set_mark(wid, lin, col) =
  let li = get_text_local_info wid.info in
  li.mrk_lin <- max 0 (min (li.nlin-1) lin);
  li.mrk_col <- max 0 (min (li.ncol-1) col);
  ()

and text_clear =
  let i = ref 0 and j = ref 0 in
fun wid ->
  let xdm = wid.wid_xd.xdm
  and li = get_text_local_info wid.info in
  li.ccol <- 0; li.clin <- 0;
  XClearWindow(xdm.dpy, li.swin);
  let sp = ascii_code " " in
  i := 0;
  while !i < li.nlin do
    let str = li.txt.(!i) in
    j := 0;
    while !j < li.ncol do set_nth_ascii(!j, str, sp); incr j done;
    incr i
  done

and text_shift(wid, val) =
  let xdm = wid.wid_xd.xdm
  and li = get_text_local_info wid.info in
  li.y_shift <- max 0 (min li.y_shmax (val*li.text_gi.tfs.fheight));
  XMoveWindow(xdm.dpy, li.swin, Zero_Int, CINT(-li.y_shift));
  ()

and text_shift_value wid =
  let li = get_text_local_info wid.info in
  li.y_shift quo li.text_gi.tfs.fheight

and text_get_text(wid, lin, col) =
  let li = get_text_local_info wid.info in
  let lin1 = li.mrk_lin and col1 = li.mrk_col in
  if lin = lin1 & 0 <= col1 & col1 <= col & col < li.ncol then
    extract_string li.txt.(lin) col1 col
  else ""
;;

let text_send_string =
  let i = ref 0
  and buff = make_string 80 ` `
  and beglin = ref 0
  and begcol = ref 0
  and bufflen = ref 0 in
  let draw_char draw_flush lin col c =
    if lin <> !beglin or col <> !begcol+!bufflen
    or !bufflen = length_string buff then (
      draw_flush(); beglin := lin; begcol := col; ()
    );
    set_nth_ascii(!bufflen, buff, c);
    incr bufflen;
    ()
  in
fun (wid, text) ->
  let xdm = wid.wid_xd.xdm
  and li = get_text_local_info wid.info in
  let gi = li.text_gi in
  let len = length_string text in
  let draw_flush() =
    if !bufflen > 0 then (
      XDrawImageString(xdm.dpy, li.swin, gi.gc_text,
        CINT(li.xb+!begcol*gi.tfs.fwidth),
        CINT(li.yb+gi.tfs.fheight*!beglin+gi.tfs.ascent),
        buff, CINT !bufflen
      );
      bufflen := 0; ()
    )
  in
  i := 0; beglin := 0; begcol := 0; bufflen := 0;
  while !i < len do
    let c = nth_ascii(!i, text) in
    if c = ascii_back_space then (
      if li.ccol > 0 then (li.ccol <- li.ccol-1; ())
      else if li.clin > 0 then (
          li.ccol <- li.ncol-1;
          li.clin <- li.clin-1;
          ()
      )
    )
    else if li.ccol = li.ncol or c = ascii_line_feed then (
      li.ccol <- 0;
      li.clin <- li.clin+1;
      if li.clin = li.nlin then (
        draw_flush();
        text_scroll wid;
        li.clin <- li.clin-1;
        ()
      )
    )
    else if c = ascii_tab then (
      li.ccol <- min li.ncol (((li.ccol+8) quo 8) * 8);
      ()
    );
    if c <> ascii_line_feed & c <> ascii_back_space & c <> ascii_tab then (
      set_nth_ascii(li.ccol, li.txt.(li.clin), c);
      draw_char draw_flush li.clin li.ccol c;
      li.ccol <- li.ccol+1;
      ()
    );
    incr i
  done;
  draw_flush()
;;

let rt_set_cut_buffer(xd, str) =
  XSetSelectionOwner(xd.xdm.dpy, XA_PRIMARY, XNone, CurrentTime);
  XStoreBytes(xd.xdm.dpy, str, CINT(length_string str));
  ()

and rt_get_cut_buffer =
  let i = mallocated_var alloc_IntRef (ref None) in
fun xd ->
  let i = i() in
  XFetchBytes(xd.xdm.dpy, i)
;;

exception not_ascii;;

let TextA attr (vlin, vcol, nlin, key_callback, button_callback) =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xdm ->
    let make_global_info xdm =
      let fs = load_query_font(xdm, !text_font) in
      let mask = Long_OR(GCForeground, Long_OR(GCBackground, GCFont))
      and gstr = gstr() in
      set_XGCValues_font(fs.fid, gstr.xgcv);
      set_XGCValues_foreground(xdm.black, gstr.xgcv);
      set_XGCValues_background(xdm.white, gstr.xgcv);
      let gc_text = XCreateGC(xdm.dpy, xdm.rootw, mask, gstr.xgcv) in
      xdm.end_func <- (function () ->
        let gi = get_text_global_info(ginfo xdm "text") in
        XFreeGC(xdm.dpy, gi.gc_text);
        XFreeFont(xdm.dpy, gi.tfs.fs);
        remove_ginfo xdm "text";
        ()
      ) :: xdm.end_func;
      add_ginfo xdm "text" text_global_info {
        tfs = fs;
        gc_text = gc_text
      }
    in
    let gi =
      try get_text_global_info(ginfo xdm "text")
      with _ -> make_global_info xdm
    and (vlin, vcol) = (max 1 vlin, max 1 vcol) in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      2*!text_band+gi.tfs.fwidth*vcol)
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      2*!text_band+gi.tfs.fheight*vlin)
    and b = match szh with (_,_,Some v) -> v | _ -> !text_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let width = max width 1 and height = max height 1 in
    let xdm = xd.xdm in
    let win = XCreateSimpleWindow(xdm.dpy, pwin,
      CINT x, CINT y,
      CINT width, CINT height, CINT(max border 0),
      xdm.black, xdm.black
    ) in
    XSelectInput(xdm.dpy, win, StructureNotifyMask);
    let gi = get_text_global_info(ginfo xdm "text") in

    let vlin = max 1 ((height-2*!text_band) quo gi.tfs.fheight)
    and vcol = max 1 ((width-2*!text_band) quo gi.tfs.fwidth) in
    let ncol = vcol in
    let nlin = max vlin nlin in
    let xb = (width-vcol*gi.tfs.fwidth) quo 2
    and yb = (height-vlin*gi.tfs.fheight) quo 2 in
    let sheight = height+(nlin-vlin)*gi.tfs.fheight in
    let y_shmax = sheight - height in
    let txt = vector nlin of "" in
    modify_vect (fun _ -> make_string ncol ` `) txt;

    let swin = XCreateSimpleWindow(xdm.dpy, win,
      Zero_Int, CINT 0,
      CINT width, CINT sheight,
      Zero_Int, xdm.black, xdm.white
    ) in
    XSelectInput(xdm.dpy, swin, it_list (curry Long_OR) Zero_Long [
      ExposureMask; KeyPressMask;
      ButtonPressMask; ButtonReleaseMask
    ]);
    XMapWindow(xdm.dpy, swin);
    let info = text_local_info {
      text_gi = gi; swin = swin;
      xb = xb; yb = yb;
      y_shift = 0; y_shmax = y_shmax;
      txt = txt;
      vlin = vlin; nlin = nlin; ncol = ncol;
      clin = 0; ccol = 0; mrk_lin = 0; mrk_col = 0;
      nscroll = 0
    } in
    add_widget [] swin (add_widget attr win {
      wid_xd = xd;
      win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = info; user_info = no_info;
      children = []
    })
  )
;
  wdestroy = (function wid ->
    let li = get_text_local_info wid.info in
    remove_widget [] li.swin wid;
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev, param) ->
    let xdm = wid.wid_xd.xdm in
    let t = XEvent_type xev in
    if t = GraphicsExpose then (
      let xev = XEvent_xgraphicsexpose xev
      and li = get_text_local_info wid.info in
      let x = num_of_C_Int(XGraphicsExposeEvent_x xev)
      and y = num_of_C_Int(XGraphicsExposeEvent_y xev)
      and width = num_of_C_Int(XGraphicsExposeEvent_width xev)
      and height = num_of_C_Int(XGraphicsExposeEvent_height xev)
      and count = num_of_C_Int(XGraphicsExposeEvent_count xev) in
      if count = 0 then (li.nscroll <- li.nscroll-1; ());
      let (imin, jmin) = lin_col_of_xy li x y
      and (imax, jmax) = lin_col_of_xy li (x+width) (y+height) in
      let imin = max 0 (imin - li.nscroll)
      and imax = max 0 (imax - li.nscroll) in
      text_expose_lines wid imin (imax+1) jmin (jmax+1);
      param
    )
    else if t = NoExpose then (
      let li = get_text_local_info wid.info in
      li.nscroll <- li.nscroll-1;
      param
    )
    else if t = Expose then (
      let xev = XEvent_xexpose xev
      and li = get_text_local_info wid.info in
      let x = num_of_C_Int(XExposeEvent_x xev)
      and y = num_of_C_Int(XExposeEvent_y xev)
      and width = num_of_C_Int(XExposeEvent_width xev)
      and height = num_of_C_Int(XExposeEvent_height xev) in
      let (imin, jmin) = lin_col_of_xy li x y
      and (imax, jmax) = lin_col_of_xy li (x+width) (y+height) in
      text_expose_lines wid imin (imax+1) jmin (jmax+1);
      param
    )
    else if t = KeyPress then (
      let xev = XEvent_xkey xev in
      let c = XKeycodeToKeysym(xdm.dpy, XKeyEvent_keycode xev,
        CINT(if XKeyEvent_state xev = ShiftMask then 1 else 0)
      ) in
      let cc = num_of_C_Long c in
      try
        let a =
          match cc with
            65288 | 65535 -> "\b" | 65293 -> "\n"
          | 65361 -> "Left" | 65362 -> "Up"  | 65363 -> "Right"
          | 65364 -> "Down" | 65379 -> "Ins"
          | 65456 -> "K0" | 65457 -> "K1" | 65458 -> "K2" | 65459 -> "K3"
          | 65460 -> "K4" | 65461 -> "K5" | 65462 -> "K6" | 65463 -> "K7"
          | 65464 -> "K8" | 65465 -> "K9"
          | 65496 -> "R7"  | 65498 -> "R9"  | 65500 -> "R11"
          | 65502 -> "R13" | 65504 -> "R15"
          | _ -> try ascii cc with _ -> raise not_ascii in
        key_callback(wid, a)
      with not_ascii -> param
    )
    else if t = ButtonPress then (
      let xev = XEvent_xbutton xev
      and li = get_text_local_info wid.info in
      let b = num_of_C_Int(XButtonEvent_button xev)
      and x = num_of_C_Int(XButtonEvent_x xev)
      and y = num_of_C_Int(XButtonEvent_y xev) in
      let i,j = lin_col_of_xy li x y in
      button_callback(wid, b, i, j)
    )
    else if t = ButtonRelease then (
      if xdm.win_but <> C'WB_None then param
      else (
        let xev = XEvent_xbutton xev
        and li = get_text_local_info wid.info in
        let b = num_of_C_Int(XButtonEvent_button xev)
        and x = num_of_C_Int(XButtonEvent_x xev)
        and y = num_of_C_Int(XButtonEvent_y xev) in
        let i,j = lin_col_of_xy li x y in
        button_callback(wid, -b, i, j)
      )
    )
    else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      let width = num_of_C_Int(XConfigureEvent_width xev)
      and height = num_of_C_Int(XConfigureEvent_height xev) in
      if width <> wid.width or height <> wid.height then (
        wid.width <- width; wid.height <- height;
        let li = get_text_local_info wid.info in
        let gi = li.text_gi
        and wdesc = wid.wdesc in

    let vlin = max 1 ((height-2*!text_band) quo gi.tfs.fheight)
    and vcol = max 1 ((width-2*!text_band) quo gi.tfs.fwidth) in
    let ncol = vcol in
    let nlin = max vlin nlin in
    let xb = (width-vcol*gi.tfs.fwidth) quo 2
    and yb = (height-vlin*gi.tfs.fheight) quo 2 in
    let sheight = height+(nlin-vlin)*gi.tfs.fheight in
    let y_shmax = sheight - height in
    let txt = vector nlin of "" in
    modify_vect (fun _ -> make_string ncol ` `) txt;

        modify_vect_i (fun i s ->
          if i < li.nlin then (replace_string s li.txt.(i) 0; s) else s
        ) txt;
        let y_shift = min li.y_shift y_shmax in
        XMoveResizeWindow(xdm.dpy, li.swin,
          Zero_Int, CINT(-y_shift),
          CINT width, CINT sheight
        );
        li.xb <- xb; li.yb <- yb;
        if li.clin >= nlin or li.ccol >= ncol then (li.ccol <- ncol; ());
        if li.clin >= nlin then (li.clin <- nlin-1; ());
        li.vlin <- vlin; li.nlin <- nlin; li.ncol <- ncol;
        li.txt <- txt;
        li.y_shmax <- y_shmax; li.y_shift <- y_shift;
        ()
      );
      param
    ) else param
  )
;
  filler = mem C'FillerAtt attr
}
;;

let TextD = TextA []
;;
