(*\
\chapter{CAML in action}
\paragraph\ 
We start this manual by a short demonstration of the capacity of the language
to help the user to construct a prototype of an application. Our example
will be an interactive desk calculator. We will introduce the basic 
notions of the language from scratch. But, this is not intended to be a 
tutorial, rather
 a brief recall of the main features (for a tutorial see the CAML PRIMER).

First, some advertising!

\section{Interfaces and facilities}
\paragraph\ 
CAML is a rich system:

Facilities include
\begin{itemize}
\item Secure and easy parser generation with the ``grammar'' construct of
 the language.
\item Separate compilation with a simple modules feature.
\item Completely integrated and typechecked macro facilities.
\item Compiler pragmas with automatic environment checking.
\item Debugging tools (a sophisticated trace package).
\item Automatic documentation of source files.
\item Scanner generation.
\item Easy development of application systems with high level tools.
\end{itemize}

Numerous interfaces are provided:
\begin{itemize}
\item Algebraic computation can be done by a specialised process
via an interface with the ``maple'' system.
\item Graphics use an interface with Postscript:
 CAML outputs code for the postcript interpreter.
\item An Xdr interface allows communication between CAML and
external C functions.
\item CAML interacts nicely with Latex, since the language is able to expand
examples written in CAML inside a latex file. Moreover a CAML program can be
turned into a latex file via another existing utility (in the spirit of WEB).
\end{itemize}


\section{Starting}
\paragraph\ 
 CAML is an interactive system:
\begin{itemize}
\item The user types in an expression and
\item CAML responds by printing the value of that expression, together with its type:
\begin{caml_example}
1;;
\end{caml_example}
\end{itemize}

 Predefined values are: 
\begin{itemize}
 \item integers which belong to the type \verb"num"
 (in fact \verb"num" is
\{integers\} $\cup$ \{rational numbers\} $\cup$ \{floats\})
 \item boolean values (type "bool") 
 \item strings (type "string")
\end{itemize}

\begin{caml_example}
1+2;;
1+2=2+1;;
"The CAML"^" language";;
\end{caml_example}

\section{Pairs and Functions}
\paragraph\ 
 Some type constructors are given to build new types from
 basic types.
 The only predefined type constructors are \verb"*" (cartesian
 product) and \verb"->" (infix notation) (building functional
 types).
 (In fact, the constructor \verb"vect" is also predefined).

\begin{caml_example}
1+2,3+4;;
1+2,3=4;;
fst (1,true);;
snd (1,true);;
1,2,3;;
fst (1,2,3);;
snd (1,2,3);;
fst (snd (1,2,3));;
\end{caml_example}

\section{Declarations}
\subsection{Global declarations}
Declarations are bindings of identifiers to values:

\begin{caml_example}
let x=1+2;;
let y=x+4;;
x*y;;
\end{caml_example}

Variables that are declared using the \verb"let" construct are
usable in expressions. Their binding is not modifiable
(unless they are of the type of modifiable values). They may
only be redefined by a new \verb"let". But this defines a new
variable, putting the previous one ``out of scope''.


\subsection{Local declarations}
\paragraph\ 
These disappear at the end of the evaluation of the phrase they appeared in:
\begin{caml_example}
let x="The CAML " and y="language" in x^y;;
y*4 where y=20;;
x;;
\end{caml_example}

In this case, the global value of \verb"x" is still \verb"3", and has
not been modified by the local declaration of \verb"x".

\section{Functions as values}

\subsection{Defining a function}

To declare a function is simple and natural:
\begin{caml_example}
let f (x) = 2*x+1;;
let g(x,y) = 2*x+y;;
\end{caml_example}

The system doesn't know how to print functional values
because their internal representation is machine code.
Application of a function to its argument is easy as well:
\begin{caml_example}
f (2);;
g (2,3);;
\end{caml_example}
A relaxed syntax exists to use and define functions. Bound arguments and 
applications are
denoted by simple juxtaposition:
\begin{caml_example}
let f x = 2*x+1;;
f 2;;
\end{caml_example}


\subsection{Anonymous functions}

A function is a first class citizen:
\begin{caml_example}
function x -> 2*x+1;;
\end{caml_example}

\verb"function" may be abbreviated by \verb"fun":

\begin{caml_example}
fun x -> 2*x+1;;
function (x,y) -> 2*x+y;;
\end{caml_example}
These are alternative ways to define \verb"f" and \verb"g", as identifiers
bound to values which are functions:
\begin{caml_example}
let f = function x -> 2*x+1;;
let g = function (x,y) -> 2*x+y;;
\end{caml_example}

\subsection{Higher-order functions}
\paragraph\ 
The \verb"->" type constructor may take any type arguments \verb"ty1"
and \verb"ty2" to build a legal type \verb"ty1 -> ty2".
 \verb"ty1" and \verb"ty2" may be functional themselves, and this leads to
functions having functions as arguments and (or) results.

\begin{caml_example}
let h = function x -> (function y -> 2*x+y);;
let k = h 1;;
k 3;;
h 2 3;;
\end{caml_example}


The above function named \verb"h" is the {\em curried} version of the
function we named \verb"g" before.
Expression \verb"h 2 3" has to be read as \verb"(h 2) 3" following the
usual $\lambda-$calculus convention.

\subsection{Recursive functions}
\paragraph\ 
The \verb"let rec" and \verb"where rec" constructs allows the definition of
``recursive'' functions, that is, functions defined in terms of themselves:

\begin{caml_example}
let rec fact x = if x=0 then 1 else x*fact(x-1);;
fact 5;;
\end{caml_example}

\subsection{Definition by ``case''}
\paragraph\ 
One may use case analysis on the value of the argument to define the same
function as in:
\begin{caml_example}
let rec fact = function 0 -> 1 | n -> n*fact(n-1);;
\end{caml_example}

There are alternative syntaxes for defining recursive functions:
\begin{caml_example}
let rec fact 0 = 1
      | fact n = n*fact(n-1);;
fact 20;;
\end{caml_example}

The special symbol \verb"_" may be used to stand for ``any other case'':
\begin{caml_example}
let is_one = function 1 -> true | _ -> false;;
is_one 1;;
is_one 0;;
\end{caml_example}

\subsection{Infix identifiers}
\paragraph\ 
It is possible to decide that an identifier
should be parsed as infix (that is between two tokens as \verb"+" in
 \verb"1+2"). There are
already predefined such identifiers: \verb"+", \verb"*", \ldots but the user
may declare his own infixes.

After an infix declaration, one has to either
write the infix identifier as an infix, or to use the
\verb"prefix" keyword, to use it as prefix
identifier.

\begin{caml_example}
#infix power;;
let rec x power y = 
    if x=0 then 0
    if y=0 then 1
    else x*(x power (y-1));;
2 power 8;;
\end{caml_example}

\section{Typing and polymorphism}
\paragraph\ 
The CAML toplevel tries to give a type to each
expression typed in by the user. The typing phase
does not need any type annotation in the program to
give it a type. Knowing types of basic values and
primitive operations and using typing rules for
application and functions, the type-checker produces
the most general type for each expression (Milner,
1978).

\begin{caml_example}
let identity x = x;;
\end{caml_example}

\verb"'a", \verb"'b", \ldots are {\em type variables}.
 A type variable stands for {\em any} legal type. As an example we can
{\em instantiate} \verb"'a" with \verb"'b -> 'b", and thus apply the function
\verb"identity" to itself:
\begin{caml_example}
let id = identity identity;;
id 1;;
(* Parsed as (identity (identity)) (1) *)
identity identity 1;;
\end{caml_example}

\begin{caml_example}
#infix o;;
let f o g = function x -> f (g x);;
\end{caml_example}

 The type of the
\verb"o" function (mathematical function composition)
means that the domain of the \verb"f" argument has to be the
same as the codomain of the \verb"g" argument, and that the
last argument (\verb"x") has to belong to the domain of the
\verb"g" argument.
If these constraints are not satisfied in an
expression, then the type-checker will reject that
expression.

\begin{caml_example}
f;;
g;;
f o g;;
g o f;;
\end{caml_example}

The type of the function \verb"o" is said to be ``polymorphic'': type variables
occur in it.

A general functional to define the curried version of a function having
 a pair of arguments is easily defined
\begin{caml_example}
let curry f = function x -> (function y -> f (x,y));;
\end{caml_example}

The converse function is not more complex:
\begin{caml_example}
let uncurry f = function (x,y) -> f x y;;
\end{caml_example}

Notice that these functions interact nicely to give instances of the identity
function:
\begin{caml_example}
curry o uncurry;;
uncurry o curry;;
\end{caml_example}

\section{Defining new types}
\paragraph\ 
It is possible to define new
types which may be mutually recursive, by defining their constructors and giving
the types of their arguments.

Assume that in the language we are modelling, an expression can be either
\begin{itemize}
\item a number (thus a constant expression).
\item a variable name
\item a binding introduced by a \verb"let" keyword
\item an unary operation
\item a binary operation
\end{itemize}
Then the data type for these expressions may be:
\begin{caml_example}
type expression =
     Constexp of num
   | Varexp of string
   | Letexp of string * expression * expression
   | Unopexp of string * expression
   | Binopexp of string * expression * expression
;;
\end{caml_example}

These types may be polymorphic, for example,
the type \verb"list" is defined in the CAML
system by:
\begin{caml}
type 'a list = [] | (prefix ::) of 'a * 'a list
\end{caml}

Thus the empty list is a polymorphic object:

\begin{caml_example}
[];;
\end{caml_example}

With our new constructors we may define new values:
\begin{caml_example}
Constexp 1;;
Constexp 2.0e+10;;
Binopexp("+", Constexp 1, Constexp 2);;
\end{caml_example}

\subsection{Function call ``by-pattern''}
\paragraph\ 
We define now a function which formally simplifies the expressions of our
tiny language. This function does a case analysis on its ``expression''
argument, and according to the shape of this expression returns a simplified
expression: this defines a set of simplification rules for expressions.

For instance we may translate the mathematical rule
\( -0 == 0 \)
with the CAML pattern matching case:
\begin{caml}
 Unopexp ("-",Constexp 0) -> Constexp 0
\end{caml}

Here is the whole definition of the simplificator
\begin{caml_example}
let rec simpl = function
(* UNARY MINUS *)
   Unopexp ("-",Constexp 0) -> Constexp 0
 | Unopexp ("-",Unopexp ("-",x)) -> simpl x
 | Unopexp ("-",x) as arg -> check (arg,Unopexp ("-",simpl x))
(* ADDITION *)
 | Binopexp ("+",Constexp m,Constexp n) -> Constexp (m+n)
 | Binopexp ("+",Constexp 0,x) -> simpl x
 | Binopexp ("+",x,Constexp 0) -> simpl x
 | Binopexp ("+",x,y) as arg ->
    check (arg,Binopexp ("+",simpl x,simpl y))
(* SUBTRACTION *)
 | Binopexp ("-",Constexp m,Constexp n) -> Constexp (m-n)
 | Binopexp ("-",Constexp 0,x) -> Unopexp ("-",simpl x)
 | Binopexp ("-",x,Constexp 0) -> simpl x
 | Binopexp ("-",x,y) as arg ->
    check (arg,Binopexp ("-",simpl x,simpl y))
(* MULTIPLICATION *)
 | Binopexp ("*",Constexp m,Constexp n) -> Constexp (m*n)
 | Binopexp ("*",Constexp 0,x) -> Constexp 0
 | Binopexp ("*",x,Constexp 0) -> Constexp 0
 | Binopexp ("*",Constexp 1,x) -> simpl x
 | Binopexp ("*",x,Constexp 1) -> simpl x
 | Binopexp ("*",x,y) as arg ->
    check (arg,Binopexp ("*",simpl x,simpl y))
(* DIVISION *)
 | Binopexp ("/",Constexp m,Constexp n) -> Constexp (m/n)
 | Binopexp ("/",Constexp 0,x) -> Constexp 0
 | Binopexp ("/",x,Constexp 0) -> Binopexp
                                   ("/",Constexp 1,Constexp 0)
 | Binopexp ("/",Constexp 1,x) -> Binopexp
                                   ("/",Constexp 1,simpl x)
 | Binopexp ("/",x,Constexp 1) -> simpl x
 | Binopexp ("/",x,y) as arg ->
    check (arg,Binopexp ("/",simpl x,simpl y))
(* LOCAL DECLARATIONS *)
 | Letexp (var,exp1,exp2) ->
            let e1 = simpl exp1 and e2 = simpl exp2
            in Letexp (var,e1,e2)
 | x -> x

and check (e1,e2) = if e1 = e2 then e1 else simpl e2
;;
\end{caml_example}

There is also a \verb"case ... of ..." construct.
One may have written the previous function with: 
\begin{caml}
let rec simpl expr =
case expr of Unopexp ("-", Constexp 0) -> Constexp 0
   | ...
     ...;;
\end{caml}
\verb"case ... of ..." is syntactically equivalent to
\verb"match ... with ...":

\begin{caml_example}
match simpl (Binopexp("+", Constexp 1, Constexp 1))
with Constexp 2 -> true
 | _ -> false
;;
\end{caml_example}

It is also possible to define product types
and, indeed, cartesian product is just a special
case of product type and may be defined by:
\begin{caml}
type 'a * 'b = {Fst:'a; Snd:'b};;
\end{caml}

and projections are:

\begin{caml}
let fst p = p.Fst and snd p = p.Snd;;
\end{caml}


\section{Exceptions}
\paragraph\ 
Dually with type definitions and
pattern-matching on constructed values, it is also
possible to define exceptions, and to use ``handlers''
to catch and treat them.

\begin{caml_example}
exception MY_EXCEPTION of expression;;
\end{caml_example}

Raising of exceptions is done using the \kwd{raise}
construct and handlers are set up with the \kwd{try} \ldots \kwd{with} 
construct:

\begin{caml_example}
let never_defined expr = raise MY_EXCEPTION expr;;
try never_defined (Constexp 0)
with MY_EXCEPTION (Constexp n) -> n*10000
 | failure s -> -1
 | _ -> -1000;;
\end{caml_example}

\section{Other features}
\paragraph\ 
\begin{itemize}
\item Modifiable values (assignation is possible).
\item Sequencing of expressions (expr1; expr2).
\end{itemize}
These permit programming in procedural style.

Iterative version of the factorial function:

\begin{caml_example}
let fact_iter n =
    let acc = ref 1 in frec n where rec
    frec = function 0 -> !acc | n -> acc:=!acc*n; frec(n-1)
;;
fact_iter 30;;
\end{caml_example}

\begin{itemize}
\item ``Lazy'' values which allow computation with potentially infinite
data structures
\end{itemize}
\begin{caml_example}
type 'a Stream = {lazy Head : 'a; lazy Tail : 'a Stream}
and 'a Frozen = lazy Freeze of 'a;;
let rec (Freeze Nat) = (Freeze {Head = 0; Tail = map succ Nat}
 where rec map f = function
  {Head=h; Tail=t} -> {Head=f h; Tail=map f t});;
Nat.Tail.Head;;
(* Updating is automatic *)
Nat;;
\end{caml_example}

\begin{itemize}
\item I/O (channels): the basic notion of I/Os in CAML is the stream of
characters.
\end{itemize}

\begin{caml_example}
try open_in "tto";() with io_failure s -> message s;;
\end{caml_example}

\begin{itemize}
\item Autoload functions whose code is loaded from a file at the first call to
the function.
\end{itemize}

\begin{caml_example}
autoload toto: num -> num and tutu: void -> bool
from "my_file"
;;
\end{caml_example}

\section{Concrete syntax manipulation}

Using a concrete syntax for ``expressions'':

\begin{caml_example}
let MLconst_to_MLvar =
function MLconst (mlstring s) -> MLvar s
 | _ -> failwith "wrong \"IDENT\" lexeme"
;;
grammar Expr =

 delimitor
   string is "\""
   comment is "%"
 ;
 precedences
   left "+";
   left "*"
 ;

rule entry Expr =
    parse NUM n -> Constexp n
        | IDENT s -> Varexp s
        | Literal "let"; Ident_or_escape v; Literal "=";
          Expr e1; Literal "in"; Expr e2 -> Letexp (v,e1,e2)
        | Unopexp e -> e
        | Binopexp e -> e
        | Literal "("; Expr e; Literal ")" -> e
        | Escape e -> e

and Ident_or_escape =
    parse IDENT s -> s
        | Escape e -> e

and Unopexp =
    parse Literal ("-" as minus); Expr e -> Unopexp (minus,e)

and Binopexp =
    parse Expr e1; Binop bop; Expr e2 -> Binopexp (bop,e1,e2)

and Binop =
    parse Literal ("+" as bop) -> bop
        | Literal ("*" as bop) -> bop
        | Literal ("-" as bop) -> bop
        | Literal ("/" as bop) -> bop
        | Escape e -> e

and Escape =
    parse Literal "^"; IDENT s; {MLconst_to_MLvar s} v -> v
        | Literal "{^"; {parse_caml_expr()} e; Literal "^}" -> e
;;
\end{caml_example}
Then naming the grammar and an entry of it, calls the corresponding parser:
\begin{caml_example}
<:Expr:Expr<1+2>>;;
let x=<<1>>;;
let y=<<x+2*3+y>>;;
\end{caml_example}
If an escape has been defined in the grammar, interaction with the metalanguage
is allowed:
\begin{caml_example}
<<^x + ^y>>;;
\end{caml_example}

One can even use its concrete syntax within patterns:
\begin{caml_example}
let is_zero = function
      <<0>> -> true
    | <<^x+^y>> -> message "this is an addition!"; false
    | _ -> false;;
\end{caml_example}



 A small pretty-printer for elements of type \verb"expression":
\begin{caml_example}
let print_expression =

let rec print_exp = function
     <<{^Constexp n^}>> -> print_num n
   | <<{^Varexp v^}>> -> print_string v
   | <<let ^v = ^expr1 in ^expr2>> ->
       print_string "let"; print_break(1,0);
       print_string v; print_break(1,0);
       print_string "="; print_break(1,0);
       print_exp expr1; print_break(1,0);
       print_string "in"; print_break (1,0);
       print_exp expr2
   | <<^exp1 ^op ^exp2>> -> print_string"(";
       print_exp exp1; print_break(1,0);
       print_string op; print_break(1,0);
       print_exp exp2; print_string ")"
   | <<- ^exp>> -> print_string "-"; print_exp exp
   | _ -> failwith "Incorrect expression"
    (* When given another unary operator *)

 in function exp -> open_hovbox 0; print_exp exp; close_box()
;;
print_expression (simpl <<2*2-(2*3)+x*y+(-1)/(2-3)>>);
print_newline();;
\end{caml_example}

 Make our expressions pretty-printer available for the CAML system itself:
\begin{caml_example}
#printer print_expression;;
simpl <<let x=2*2-(2*3)+x*y+(-1)/(2-3) in x*x-2>>;;
\end{caml_example}

We now rewrite \verb"simpl" in a more convenient way, using concrete syntax
in each sides of match rules:
\begin{caml_example}
let rec simpl = function
(* UNARY MINUS *)
      <<-0>> -> <<0>>
    | <<-(- ^x)>> -> simpl x
    | <<-^x>> as arg -> check(arg,<<-{^simpl x^}>>)
(* ADDITION *)
    | <<{^Constexp m^}+{^Constexp n^}>> -> Constexp (m+n)
    | <<0+^x>> -> simpl x
    | <<^x+0>> -> simpl x
    | <<^x+^y>> as arg -> check(arg,<<{^simpl x^}+{^simpl y^}>>)
(* SUBTRACTION *)
    | <<{^Constexp m^}-{^Constexp n^}>> -> Constexp (m-n)
    | <<0-^x>> -> <<-{^simpl x^}>>
    | <<^x-0>> -> simpl x
    | <<^x-^y>> as arg -> check(arg,<<{^simpl x^}-{^simpl y^}>>)
(* MULTIPLICATION *)
    | <<{^Constexp m^}*{^Constexp n^}>> -> Constexp (m*n)
    | <<0*^x>> -> <<0>>
    | <<^x*0>> -> <<0>>
    | <<1*^x>> -> simpl x
    | <<^x*1>> -> simpl x
    | <<^x*^y>> as arg -> check(arg,<<{^simpl x^}*{^simpl y^}>>)
(* DIVISION *)
    | <<{^Constexp m^}/{^Constexp n^}>> -> Constexp (m/n)
    | <<0/^x>> -> <<0>>
    | <<^x/0>> -> <<1/0>>
    | <<1/^x>> -> <<1/{^simpl x^}>>
    | <<^x/1>> -> simpl x
    | <<^x/^y>> as arg -> check(arg,<<{^simpl x^}/{^simpl y^}>>)
(* LOCAL DECLARATIONS *)
    | <<let ^var = ^exp1 in ^exp2>> ->
             let e1 = simpl exp1 and e2 = simpl exp2
             in <<let ^var = ^e1 in ^e2>>
    | x -> x

and check(e1,e2) = if e1=e2 then e1 else simpl e2;;
\end{caml_example}

\section{Debugging tools}

Trace facilities are provided
\begin{caml_example}
trace "fact";;
fact 5;;
\end{caml_example}
Numerous options are available, through a predefined concrete syntax for trace
orders:
\begin{caml_example}
<:Trace<
 fact x : trace x from fact with predicate (fun x -> x > 3);
          print x with
           (fun x -> print_string ("Number "^(string_of_num x)));
          before x do
           (fun x -> message "Fact is called from fact !");
          after x do
           (fun x result ->
            message ("Result was: "^(string_of_num result)))
>>;;
fact 5;;
\end{caml_example}

Information about identifiers can be given by the system:
\begin{caml_example}
info"fact";;
\end{caml_example}
and identifiers can be looked for in the global symbol table:
\begin{caml_example}
search_variable"act";;
\end{caml_example}

\section{A simple calculator}

\begin{itemize}
\item We use abstract syntax trees of arithmetic expressions.
\item We define abstract syntax trees for toplevel expressions.
\item Then we define their denotational semantics.
\item At last, a toplevel loop is defined.
\end{itemize}

 Let us recall definition of abstract syntax trees:
\begin{caml}
type expression =
     Constexp of num
   | Varexp of string
   | Letexp of string * expression * expression
   | Unopexp of string * expression
   | Binopexp of string * expression * expression
;;
\end{caml}

We add now the abstract syntax of toplevel expressions, which may be either
expressions to be evaluated or declarations of globals:
\begin{caml_example}
(* Type of toplevel expressions *)
type toplevel_expression =
 Exp of expression
 | Global_let of string * expression
 | End
;;
\end{caml_example}

And the complete grammar associated with the desk calculator is:
\begin{caml_example}
grammar for values Dc =

 delimitor
string is "\""
comment is "%"
 ;
 precedences
left "+";
left "*"
 ;

rule

entry Top_exp = parse Top_exp1 te; Literal "?" -> te

and Top_exp1 = parse
    Literal "let"; IDENT s; Literal "="; Expr e -> Global_let(s,e)
  | Expr e -> Exp e
  | Literal "." -> End

and Expr = parse
    NUM n -> Constexp n
  | IDENT s -> Varexp s
  | Literal "let"; IDENT s; Literal "=";
    Expr e1; Literal "in"; Expr e2 -> Letexp (s,e1,e2)
  | Unopexp e -> e
  | Binopexp e -> e
  | Literal "("; Expr e; Literal ")" -> e

and Unopexp = parse
    Literal ("-" as minus); Expr e -> Unopexp (minus,e)

and Binopexp = parse
    Expr e1; Binop bop; Expr e2 -> Binopexp (bop,e1,e2)

and Binop = parse
    Literal ("+" as bop) -> bop
  | Literal ("*" as bop) -> bop
  | Literal ("-" as bop) -> bop
  | Literal ("/" as bop) -> bop
;;
\end{caml_example}
From this grammar we define a parser:
\begin{caml_example}
let parse_dc_top_exp () =
 match eval_syntax (((Dc "Top_exp").Parse) ()) with
  dynamic (d : toplevel_expression) -> d;;
\end{caml_example}

\section{Denotational semantics of a small desk calculator}
\paragraph\ 
We define the global environment of the calculator and its 
associated manipulation tools:
\begin{caml_example}
let global_env = ref ([]: (string * num) list);;
let init_env () = global_env:=[]; ()
and store_in_global_env (var,val as binding) =
    global_env:=binding::!global_env;
    print_string (var^" gets bound to ");
    val
;;
\end{caml_example}

Now we define the semantics of expressions:
\begin{caml_example}
exception UNBOUND of string;;
let rec semexp env = function
    Constexp n -> n
  | Varexp var -> assoc var env ? raise UNBOUND var
  | Letexp (var,exp,exp') ->
           let n = semexp env exp in
           semexp ((var,n)::env) exp'
  | Unopexp (name,exp) ->
           semunop name (semexp env exp)
  | Binopexp (name,exp,exp') ->
           sembinop name (semexp env exp,semexp env exp')

and semunop = function
    "-" -> minus | s -> failwith ("unknown unop"^s)

and sembinop = function
    "+" -> prefix +
  | "*" -> prefix *
  | "-" -> prefix -
  | "/" -> prefix /
  | s -> failwith ("unknown unop"^s)
;;
\end{caml_example}

and the semantics of toplevel expressions
\begin{caml_example}
exception END_DC;;
let global_sem env = function
    Exp exp -> semexp env exp
  | Global_let (var,exp) -> let n = semexp env exp in
                            store_in_global_env (var,n)
  | End -> raise END_DC
;;
(* To compute the value of a toplevel expression *)
let compute exp = global_sem !global_env exp;;
\end{caml_example}

The calculator itself and its toplevel loop:
\begin{caml_example}
let dc_step e =
 try
 print_num (compute e);
 print_newline();print_newline()
 with UNBOUND var -> message ("*** "^var^" is unbound! ***");
                     print_newline()
| END_DC -> message "Bye" reraise
| failure s -> message s; print_newline()
| break -> message "Bye"; raise END_DC
| _ -> message "ERROR"; print_newline();;
let dc_top e = 
 try dc_step e with _ -> ();;
let dc_eval = dc_step o parse_dc_top_exp;;
let dc_loop () =
 try while true do dc_eval () done
 with
  END_DC -> set_prompt "#";;
let dc () =
 print_newline(); set_prompt "-> "; init_env();
 dc_loop()
;;
(* Create a load function for dc, using the function
   load_with_loop, available from the CAML system *)
let load_dc_in_channel = load_with_loop dc_loop;;
let load_dc s =
    let is = open_in s in
    load_dc_in_channel is;
    close_in is;
    message ("dc file "^s^" loaded");;
\end{caml_example}
\*)

(*\begin{caml_ignore}*)
(*
let dc () =
 print_newline(); set_prompt "-> "; init_env();

 let rec dc_go_on () =
 try while true do 
 print_num(compute(parse_dc_top_exp ()));
 print_newline();print_newline()
 done
 with UNBOUND var -> message ("*** "^var^" is unbound! ***");
 print_newline(); dc_go_on()
| END_DC -> set_prompt "#"; message "Bye"
| failure s -> message s; print_newline(); dc_go_on ()
| break -> set_prompt "#"; message "Bye"
| _ -> message "ERROR"; print_newline(); dc_go_on ()

 in dc_go_on()
;;
*)
(*\end{caml_ignore}*)

(*\
Let's give a ``dc'' session as an example:
\begin{caml}
(* Entering calculator's toplevel loop *)
dc();;
\end{caml}

\begin{caml_eval}
user_syntax_prefix := "dc_top (<:Dc:Top_exp<";;
user_syntax_postfix := ">>);;";;
user_syntax_prompt := "-> ";;
user_syntax_terminator := "?";;
\end{caml_eval}

\begin{caml_user_syntax_example}
3 ?
x ?
let x=3 ?
let x=10 in x*x ?
x-x ?
let x=x+1 ?
let y = 111111111111111111111111111111111111111111+1 ?
y*1.5 ?
(* Exiting (notice that comments are still available) *)
.?
\end{caml_user_syntax_example}
\begin{caml_example}
1;;
\end{caml_example}

When then load the following simple ``dc'' file:

\begin{caml_include}
./aux/dc
\end{caml_include}
using:
\begin{caml_example}
load_dc "./aux/dc";;
\end{caml_example}

\*)

(*\begin{caml_ignore}*)
(*\
\section{Lazy data structures}

\begin{caml_example}
(* Lazy data structures *)

type 'a stream = {lazy Lhd:'a; lazy Ltl:'a stream};;

type ('a,'b) lpair = {lazy Lfst: 'a; lazy Lsnd:'b};;

let hd {Lhd=x;Ltl=_} =x
and tl {Lhd=_;Ltl=x} = x;;

type 'a frozen = lazy Freeze of 'a
;;

let Unfreeze (Freeze x) = x;;

(* Forcing them from 1 to n simply by accessing to each element *)

let eval x = x;;

let Force L n = evalrec L n;L
  where rec evalrec {Lhd=x;Ltl=l} =
             fun 1 -> x
               | n -> eval x; evalrec l (n-1);;

(* Mapping function over infinite lists *)

let rec map_stream f {Lhd=x;Ltl=t} =
        {Lhd=f x; Ltl= map_stream f t};;

let sf_uminus = map_stream (fun x -> (-x));;

(* Formal series integration fonction *)

let integer l = (intrec 0 l)
  where rec intrec n {Lhd=x;Ltl=l} =
                {Lhd=(x/(n+1));Ltl=(intrec (n+1) l)}
;;

let print_expo =
function 0 -> ()
       | 1 -> print_string " X"
       | n -> print_string " X^"; print_num n
;;

let print_monome coeff n =
match coeff
with 0 -> ()
   | 1 -> print_string "+"; print_expo n
   | (-1) -> print_string "-"; print_expo n
   | c -> if c<0 then (print_string "- "; print_num (-c))
          else (print_string "+ "; print_num c);
          print_expo n
;;

let print_sf sf expo =
let rec print sf n =
if n<=0 then () else
(print_space(); print_monome sf.Lhd (expo-n); print sf.Ltl (n-1))
in
print_num sf.Lhd; open_hovbox 1;
print sf.Ltl (expo-1);
close_box(); print_newline()
;;

(* Examples *)

(* Necessary to protect evaluation of exp for it not to be undefined *)

let rec (Freeze exp) = Freeze {Lhd=1;Ltl=integer exp};;

print_sf exp 10;;

let rec (Freeze l) = Freeze {Lhd=1;Ltl=(sf_uminus l)};;

let log1 = {Lhd=0;Ltl=integer l};;

print_sf log1 10;;

let rec (Freeze {Lfst=cos;Lsnd=sin}) =
Freeze {Lfst={Lhd=1;Ltl=integer (sf_uminus sin)};
Lsnd={Lhd=0;Ltl=integer (cos)}};;

print_sf cos 10;;
print_sf sin 10;;
\end{caml_example}

That's all folks!
\*)
(*\end{caml_ignore}*)
