(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* semant.ml	Denotational semantics of a simple langauge.             *)
(*              Guy Cousineau (original code)                            *)
(*		Pierre Weis (new version for CAML V2-5)                  *)

(****************************** SEMANT ***********************************)


(* Le langage Semant : langage proce'dural avec fonctions et passage
   de parame`tres par re'fe'rence, sans go to *)


(**************************** LA SYNTAXE *********************************)


(* La syntaxe abstraite de Semant *)

(*|
Type program defined
     Program : (declaration * command -> program)
Type declaration defined
     Var_decl : (string -> declaration)
   | Proc_decl :
     (string * string * string * declaration * command -> declaration)
   | Fun_decl : (string * string * expression -> declaration)
   | Comp_decl : (declaration list -> declaration)
Type command defined
     Ass_com : (string * expression -> command)
   | Proc_com : (string * string * expression -> command)
   | If_com : (expression * command * command -> command)
   | While_com : (expression * command -> command)
   | Write_com : (expression -> command)
   | Read_com : (string -> command)
   | Comp_com : (command list -> command)
Type expression defined
     Num_exp : (num -> expression)
   | Bool_exp : (bool -> expression)
   | Unop_exp : (string * expression -> expression)
   | Binop_exp : (expression * string * expression -> expression)
   | Var_exp : (string -> expression)
   | Fun_exp : (string * expression -> expression)
   | If_exp : (expression * expression * expression -> expression)
|*)
type program = Program of declaration & command

and  declaration =
                Var_decl of string
              | Proc_decl of string & string & string &
                             declaration & command
              | Fun_decl of string & string & expression
              | Comp_decl of declaration list

and  command =
                Ass_com of string & expression
              | Proc_com of string & string & expression
              | If_com of expression & command & command
              | While_com of expression  & command
              | Write_com of expression
              | Read_com of string
              | Comp_com of command list

and expression =
                Num_exp of num  
              | Bool_exp of bool  
              | Unop_exp of string & expression
              | Binop_exp of expression & string & expression
              | Var_exp of string 
              | Fun_exp of string & expression
              | If_exp of expression & expression & expression;;


(************************** LA SYNTAXE CONCRETE *************************)


(*|
Directive () : unit
|*)
#use "semant_gram2";;


(***************************** LA SEMANTIQUE ****************************)


(* Les valeurs manipule'es par le langage *)
(*|
Type Value defined
     num : (num -> Value)
   | bool : (bool -> Value)
   | file : (num list -> Value)
|*)
type Value = num of num | bool of bool | file of num list;;

(* Les domaines de la semantique *)
(*|
Type Env defined
     Env : ((string -> Val) -> Env)
Type Val defined
     Val : (Value -> Val)
   | Loc_as_val : (Loc -> Val)
   | Proc : ((Val * Val -> Store -> Store) -> Val)
   | Fun : ((Val -> Store -> Val) -> Val)
Type Loc defined
     Loc : (num -> Loc)
Type Store defined
     Store : ((Loc -> Val) -> Store)
|*)
type     Env = Env of (string -> Val)

and      Val = Val of Value 
             | Loc_as_val of Loc
             | Proc of (Val & Val -> Store -> Store)
             | Fun of  (Val -> Store -> Val)

and      Loc = Loc of num
and      Store = Store of Loc -> Val;;


(* Les adresses me'moires constantes *)
(*|
Value first_unused_Loc : Loc
Value input_Loc : Loc
Value output_Loc : Loc
|*)
let first_unused_Loc = Loc 0
and input_Loc = Loc 1
and output_Loc = Loc 2;;

(* L'environnement vide *)
(*|
Value init_env : Env
|*)
let init_env =
 Env (fun ident -> failwith (ident ^ " unbound"));;

(* La me'moire "vide" *)
(*|
Value init_store : Store
|*)
let init_store =
 Store (fun (Loc 0) -> Loc_as_val (Loc 3)
         |  (Loc 1|Loc 2) -> Val (file [])
         | _ -> failwith "unused location");;

(* Le contenu d'une adresse *)
(*|
Value contents : (Loc -> Store -> Val)
|*)
let contents loc (Store store) = store loc;;

(* Ve'rifie que la premie`re adresse libre est bien une adresse *)
(*|
Value get_first_free_Loc : (Store -> Loc)
{contents,first_unused_Loc}
|*)
let get_first_free_Loc S =
 match contents first_unused_Loc S
  with (Loc_as_val (Loc l as loc)) -> loc
    |   _ -> failwith "First free location is not a location!";;

(* La valeur d'un identificateur dans un environnement *)
(*|
Value envalof : (string -> Env -> Val)
|*)
let envalof ident (Env env) = env ident;;

(* La valeur d'un identificateur est le contenu de l'adresse
   qui lui est lie'e dans l'environnement *)
(*|
Value loc_of_ident : (Env -> string -> Loc)
{envalof}
|*)
let loc_of_ident env ident =
 match envalof ident env
  with (Loc_as_val loc) -> loc
    |  _ -> failwith "Bad environment gestion";;

(*|
Value valof : (Env -> string -> Store -> Val)
{contents,loc_of_ident}
|*)
let valof env ident = contents (loc_of_ident env ident);;

(* Le moteur de la se'mantique :
   la fonction qui modifie les fonctions *)
(* extend : 'a -> ('a -> 'b) -> 'b -> ('a -> 'b) *)
(*|
Value extend : ('a -> ('a -> 'b) -> 'b -> 'a -> 'b)
|*)
let extend tag f val =
 fun x -> if x = tag then val else f x;;

(* Alloue une case me'moire : pousse d'un la premie`re adresse libre *)
(*|
Value push_store : (Store -> Store)
{extend,first_unused_Loc,get_first_free_Loc}
|*)
let push_store (Store store as S) =
    let new_free_Loc =
     Loc_as_val (Loc (l+1 where (Loc l) = get_first_free_Loc S)) in
    Store (extend first_unused_Loc store new_free_Loc);;
 
(* Change le contenu d'une adresse me'moire *)
(* Renvoie une nouvelle me'moire avec v a` l'adresse loc *)
(*|
Value update_store : (Loc -> Store -> Val -> Store)
{extend}
|*)
let update_store loc (Store store) v = Store (extend loc store v);;

(* Change la liaison d'un identificateur *)
(*|
Value update_binding : (string -> Env -> Store -> Val -> Store)
{loc_of_ident,update_store}
|*)
let update_binding env ident = update_store (loc_of_ident ident env);;

(* Change la valeur lie'e a` un identificateur ds l'environnement *)
(*|
Value update_env : (Env -> string -> Val -> Env)
{extend}
|*)
let update_env (Env env) ident v = Env (extend ident env v);;
    
(* De'clare un nouvel identificateur *)
(*|
Value push_binding : (string -> Env * Store -> Env * Store)
{contents,first_unused_Loc,push_store,update_env}
|*)
let push_binding ident (env,store) =
 let loc = contents first_unused_Loc store in
 (update_env env ident loc, push_store store);;

(* De'clare un identificateur et lui affecte une valeur ds la me'moire *)
(*|
Value new_binding : (string -> Val -> Env * Store -> Env * Store)
{get_first_free_Loc,push_store,update_env,update_store}
|*)
let new_binding ident v (env,s) =
 let loc = get_first_free_Loc s in
  (update_env env ident (Loc_as_val loc),
   update_store loc (push_store s) v);;

(* Utilitaires pour les entre'es-sorties *)
(*|
Value get_output_file : (Store -> num list)
{contents,output_Loc}
|*)
let get_output_file s =
 match contents output_Loc s
 with (Val (file prevoutput)) -> prevoutput
  |   _ -> failwith "Output is not a file";;

(*|
Value write : (Val -> Store -> Store)
{get_output_file,output_Loc,update_store}
|*)
let write = fun
  (Val (num v)) s ->
    update_store output_Loc s (Val (file (v :: (get_output_file s))))
  | _ -> failwith "can write only numbers";;

(*|
Value get_input_file : (Store -> num list)
{contents,input_Loc}
|*)
let get_input_file s =
 match contents input_Loc s
 with (Val (file previnput)) -> previnput
  |   _ -> failwith "Input is not a file";;

(*|
Value read : (Store -> Store * Val)
{get_input_file,input_Loc,update_store}
|*)
let read s =
 match get_input_file s with
  [] -> failwith "empty input"
 | (x::l) -> (update_store input_Loc s (Val (file l)),(Val (num x)));;

(* L'extraction des valeurs (Val -> "CAML") *)
(*|
Value get_bool : (Val -> bool)
|*)
let get_bool v =
 match v with (Val (bool b)) -> b | _ -> failwith "Type clash Bool";;

(*|
Value get_proc : (Val -> Val * Val -> Store -> Store)
|*)
let get_proc v =
 match v with (Proc p) -> p | _ -> failwith "Type clash Proc";;

(*|
Value get_fun : (Val -> Val -> Store -> Val)
|*)
let get_fun v =
 match v with (Fun f) -> f | _ -> failwith "Type clash Fun";;


(************************ LA SEMANTIQUE DE SEMANT ************************)


(* Se'mantique des ope'rations *)
(*|
Value sembinop : (string -> Val * Val -> Val)
|*)
let sembinop = fun
      "+" -> (fun (Val (num v1),Val (num v2)) -> Val (num (v1 + v2))
                 | _ -> failwith "wrong args to operator +")
    | "*" -> (fun (Val (num v1),Val (num v2)) -> Val (num (v1 * v2))
                 | _ -> failwith "wrong args to operator *")
    | "-" -> (fun (Val (num v1),Val (num v2)) -> Val (num (v1 - v2))
                 | _ -> failwith "wrong args to operator -")

    | "=" -> (fun (Val (num v1),Val (num v2))  -> Val (bool (v1=v2))
               | (Val (bool v1),Val (bool v2)) -> Val (bool (v1=v2))
               | _ -> failwith "wrong args to operator =")

    | "<" -> (fun (Val (num v1),Val (num v2)) -> Val (bool (v1<v2))
               | _ -> failwith ("wrong args to operator <"))
    | ">" -> (fun (Val (num v1),Val (num v2)) -> Val (bool (v1>v2))
               | _ -> failwith ("wrong args to operator >"))
    |  t  -> failwith ("wrong operator : " ^ t);;

(*|
Value semunop : (string -> Val -> Val)
|*)
let  semunop = fun "suc" -> (fun (Val (num v)) -> Val (num (1+v))
                              |  _ -> failwith "wrong arg to suc")
                  |  t   -> failwith ("wrong operator : " ^ t);;


(***************************)
(* La se'mantique des expressions *)
(*|
Default grammar is now semant:expression
Pragma () : unit
|*)
#set default grammar semant : expression;;

(*|
Value semexp : (expression -> Env -> Store -> Val)
{envalof,get_bool,get_fun,sembinop,semunop,valof}
|*)
let rec semexp = function
    << Num ^n >>  ->  (fun e s -> Val (num n))
 |  << Bool ^b >>  ->  (fun e s -> Val (bool b))

 |  << Unop ^op_name ^E >>  ->
          let opr = semunop op_name
          and Sem_E = semexp E in
          (fun e s -> opr (Sem_E e s))
 | << ^E1 ^op_name ^E2 >> ->
          let opr = sembinop op_name
          and Sem_E1 = semexp E1
          and Sem_E2 = semexp E2 in
          (fun e s -> opr (Sem_E1 e s,Sem_E2 e s))

 |  <<Ident ^ident >>  ->  (fun e s -> valof e ident s)

 |  << ^fun_name (^arg) >> ->
     let Sem_arg = semexp arg in
     (fun e s ->
      let f = get_fun (envalof fun_name e) in
       f (Sem_arg e s) s)

   | << if ^test then ^E1 else ^E2 >> ->
     let Sem_test = semexp test
     and Sem_E1 = semexp E1
     and Sem_E2 = semexp E2 in
     (fun e s -> (if get_bool (Sem_test e s) then Sem_E1 else Sem_E2) e s)
;;



(*|
Default grammar is now semant:command
Pragma () : unit
|*)
#set default grammar semant : command;;

(*|
Value semcom : (command -> Env -> Store -> Store)
{envalof,get_bool,get_proc,read,semexp,update_binding,write}
|*)
let rec semcom = function
      << ^ident := ^exp >> ->
      let Sem_exp = semexp exp in
      (fun e s -> update_binding ident e s (Sem_exp e s))

 | << ^proc_name (^ident,^exp) >> ->
      let Sem_exp = semexp exp in
      (fun e s ->
       let proc = get_proc (envalof proc_name e) in
        proc (envalof ident e,Sem_exp e s) s)

 | << if ^test then ^com1 else ^com2 >> ->
      let Sem_test = semexp test
      and Sem_com1 = semcom com1
      and Sem_com2 = semcom com2 in
      (fun e s -> if get_bool (Sem_test e s) then Sem_com1 e s else
                                                  Sem_com2 e s)

 | << while ^test do ^comm >> ->
      let Sem_comm = semcom comm
      and Sem_test = semexp test in
      (fun e  -> wh
       where rec wh s =
       if get_bool (Sem_test e s) then wh (Sem_comm e s) else s)

 | << write ^exp >> ->
      let Sem_exp = semexp exp in
      (fun e s -> write (Sem_exp e s) s)

 | << read ^ident >> ->
      (fun e s -> 
       let s',val = read s in update_binding ident e s' val)

 | << begin ^comms end >> ->
      match comms with
         [] -> (fun e s -> s)
      |  comm :: coml ->
      let Sem_com = semcom comm
      and Sem_coml = semcom (Comp_com coml) in
      (fun e s -> Sem_coml e (Sem_com e s));;



(* La se'mantique des de'clarations *)

(*|
Default grammar is now semant:declaration
Pragma () : unit
|*)
#set default grammar semant : declaration;;

(*|
Value semdecl : (declaration -> Env * Store -> Env * Store)
{new_binding,push_binding,semcom,semexp,update_env}
|*)
let rec semdecl =
 fun << var ^ident >>  ->
      (fun (e,s) -> push_binding ident (e,s))

   | << function ^fun_name (^id); ^exp >>  -> 
       let Sem_exp = semexp exp in
       (fun (e,s) ->
        let rec f val1 s1 =
         let env_id,store_id = new_binding id val1 (e,s1) in
         let env_fun = update_env env_id fun_name (Fun f) in
         Sem_exp env_fun store_id

        in update_env e fun_name (Fun f), s)

   | << procedure ^proc_name (^id1,^id2);
         ^decl ^comm>>  ->
      let Sem_comm = semcom comm
      and Sem_decl = semdecl decl in
      (fun (e,s) ->
       let rec proc (ref1,val1) s1 =
        let env_id1 = update_env e id1 ref1 in
        let env_id2,store_id2 = new_binding id2 val1 (env_id1,s1) in
        let env_decl,store_decl = Sem_decl (env_id2,store_id2) in
        let env_proc = update_env env_decl proc_name (Proc proc) in
        Sem_comm env_proc store_decl

       in (update_env e proc_name (Proc proc) , s))

   | << ^d >> -> match d with
      [] -> I
    | decl :: decll ->
      let Sem_decl = semdecl decl
      and Sem_decll = semdecl (Comp_decl decll) in
      (fun (e,s) -> Sem_decll (Sem_decl (e,s)));;


(* La se'mantique des programmes *)

(*|
Default grammar is now semant:program
Pragma () : unit
|*)
#set default grammar semant : program;;

(*|
Value semprog : (program -> num list -> num list)
{get_output_file,init_env,init_store,input_Loc,semcom,semdecl,
 update_store},
 CAML_system{rev}
|*)
let semprog << program ^decl ^comm . >> =
 let (e,s) = semdecl decl (init_env,init_store) in
 let Sem_come = semcom comm e in
 (fun input_list ->
  let s' = update_store input_Loc s (Val (file input_list)) in
  let final_store = Sem_come s' in
  rev (get_output_file final_store));;

(* Exemple : la fonction de fibonacci *)

(* fib : num list -> num list *)

(*|
Value fib : (num list -> num list)
{semprog}
|*)
let fib = semprog
<<
program
 var x ;
 function fib (x);
  if x = 1 then 1 else
   if x = 2 then 1 else
    fib (x-1) + fib (x-2);
begin
 read x ; write (fib (x))
end.
>>;;

(*|
* : unit
|*)
timers true;;
(*|
* : num list
|*)
fib[10];;
