(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* pretty.ml     A pretty_printing package for CAML                 	 *)
(*		 Olivier Jehl						 *)

(* Abstract syntax of pretty_printing *)
#module Pretty;;

type Pretty_Order == Element list

and Sub_box = {
  Type:Type_box;
  Offset:ML;
  Actions_list:Element list}

and Type_box = h | v | hv | hov 

and Element = 
   
   Ident of string
 | Extern of ML
 | String of string
 | Space
 | Cut
 | Break of ML
 | Iterate of Iteration
 | Define_printer of Local_printer
 | Sequence of (Element list)
 | Sub_box of Sub_box 
 | Define_pretty_function of Pretty_function
 | New_line

and Pretty_function == (MLpat & Element list) list

and Iteration = {
  Symbol:MLpat;
  Action_iterated:Element list;
  Extern_list:ML }

and Local_printer = {
  Printer_function:ML;
  Actions_list:Element list}

and Current_function = Undefined | Function of ML

and Pretty_env ==Current_function & string list;;


 
(* the parser of pretty-syntax ;
 it needs : a current printing function, 
            a list of unbound variables,
            a expression of type Element;            
 it returns en expression of type ML, which is a sequence of printing orders;
*)
         

let rec Element_to_ML =

(* to apply MLseq only if it is useful :  if an element of the argument list *)
(* is equal to (MLseq <expr>) then we don't keep this MLseq                  *)
  let mk_MLseq= 
    let rec avoid_MLseq = function
         ((MLseq x)::l) -> x @ avoid_MLseq(l)
       | (x::l) -> x::avoid_MLseq(l)
       | [] -> []
    in  function [] -> MLconst mlnull
             | [x] -> x
             | l -> MLseq (avoid_MLseq (l))
  in
   function
    Undefined,_,Ident x 
     -> warning ("Printer defined for "^x^" is print_string");
        MLapply ((MLvar "print_string"),MLvar x,[])
  | Function f,Extern_list,Ident x
     ->  MLapply (f,(MLvar x),[])
  | _,_,String s 
     -> MLapply ((MLvar "print_string"),(MLconst (mlstring s)),[])
  | Undefined, _,Extern x 
     -> open_hovbox 0; print_dlist print_string " " ["Warning";"Printer";
                           "defined";"for "];
                       print_ML x ();
                       print_dlist print_string " " [" is";"print_string"];
        message " !";
        MLapply ((MLvar "print_string"),x,[])
  | Function f,Extern_list,Extern x 
     -> MLapply (f, x,[])
  | _,_,Space    
     -> (<:CAML:Expr<print_space ()>>)
  | (_,_,Cut)      
     -> <:CAML:Expr<print_cut()>>
  | _,_,Break paire
     -> MLapply (<:CAML<print_break>>,paire,[])
  | _,Extern_list,Define_printer app 
     -> Element_to_ML (Function app.Printer_function, Extern_list, Sequence app.Actions_list)
  | f,Extern_list,Sequence l 
     -> mk_MLseq   (map (fun element -> Element_to_ML (f,Extern_list,element))
                     l)
  | f,Extern_list,Define_pretty_function l
     -> MLmatch (map (fun (pat,element) -> (pat,
                                            Element_to_ML 
                                              (f,
                                               Extern_list,
                                               Sequence element)))
                     l)
  | f,Extern_list,Iterate ite    
     -> MLapply
        ((MLvar "do_list"),
          (MLmatch
           [( ite.Symbol,
              Element_to_ML
                (f,ite.Symbol::Extern_list,Sequence ite.Action_iterated))]),
          [ite.Extern_list])
  | f,l,Sub_box b
     -> mk_MLseq (((MLapply((MLvar( (fun  h -> "open_hbox"
                                   |   v -> "open_vbox"
                                   |  hv -> "open_hvbox"
                                   | hov -> "open_hovbox") b.Type)),
                            b.Offset,[])
                 ) ::
                (map (fun element -> Element_to_ML (f,l,element)) b.Actions_list)
                 )@ [ <:CAML:Expr< close_box() >>])
  | _,_,Return 
     -> <:CAML<force_newline()>>
;;


(* the parser of pretty-syntax *)

let Pretty_Order_to_ML expr=
    Element_to_ML (Undefined, [],Sequence expr);;
        

(* Grammar of the pretty concret syntax *)

            
#pragma echo_abbrevs false;;

grammar  for values Pretty =

rule entry exec =
        parse Prog s accept s

and entry top =
        parse Syntax s ; Literal ";;"  -> Pretty_Order_to_ML s

and entry Prog =
        parse Syntax s -> Pretty_Order_to_ML s

and Syntax =
        parse [*(parse Order ord -> ord)] l ; Return_order r -> l@r

and Return_order =
        parse Literal "return"  -> [New_line]
            |                   -> []

and Order =
        parse Ident i -> Ident i
            | String s -> String s

            | Literal "-" -> Space

            | Literal "\\" ; Break b -> b

            | Literal "with" ;
              Escape pf ;
              Literal "do";
              Syntax sy ;
              Literal "done"
                    -> Define_printer {Printer_function = pf;Actions_list = sy}

            | Literal "iterate";
              Pattern p ;
              Literal "->" ;
              Syntax sy ; 
              Literal "on" ;
              Caml_expr el
                    -> Iterate {Symbol=p;Action_iterated=sy;Extern_list=el}

            | Literal "(" ; Escape es ; Literal ")" -> Extern es

            | Sub_box b -> Sub_box b


and Sub_box =
        parse Literal "[" ; Type_box ty ; Syntax sy ; Literal "]" 
                    -> {Actions_list=sy;Offset=ty.Offset;Type=ty.Type}

and Type_box =
        parse Literal "(" ; Name_Sub_box n ; Escape e ; Literal ")"
                    -> {Actions_list=[];Offset=e;Type=n}
            | Name_Sub_box n 
                    -> {Actions_list=[];Offset=<:CAML<0>>;Type=n}
            |       -> {Actions_list=[];Offset=<:CAML<0>>;Type=hov}

and Name_Sub_box =
        parse "h" -> h
            | "v" -> v 
            | "hv" -> hv
            | "hov" -> hov

and Break =
        parse Literal "(" ; Escape n1  ; Literal ")"
                    -> Break (n1)
            |       -> Cut

and Caml_expr =    
        parse Ident s -> MLvar s
            | Literal "(" ; Escape f ; Literal ")" -> f

and Escape =
        parse { parse_caml_expr ()} expr -> expr

and Pattern =
        parse { parse_caml_pat ()} pat -> pat
;;

let parse_pretty = (Pretty "exec").Parse_raw;;

#pragma echo_abbrevs true;;
#end module with value parse_pretty 
                   and Pretty;;
