
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	cons.c
 *
 *	PVM-console task.  Gives the user a shell to manage the virtual
 *	machine and other tasks.
 *
$Log$
 */

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include "pvm3.h"

#define	TALLOC(n,t)	(t*)malloc((n)*sizeof(t))

#define	PVMERRMSG(n)	((n) <= 0 && (n) > -pvm_nerr \
						? pvm_errlist[-(n)] : "Unknown Error")

extern char *pvm_errlist[];
extern int pvm_nerr;

int add_cmd();
int conf_cmd();
int delete_cmd();
int halt_cmd();
int help_cmd();
int id_cmd();
int insert_cmd();
int kill_cmd();
int lookup_cmd();
int mstat_cmd();
int quit_cmd();
int pstat_cmd();
int remove_cmd();
int reset_cmd();
int sig_cmd();
int spawn_cmd();
int start_cmd();
int tasks_cmd();
int tickle_cmd();
int version_cmd();


int mytid = -1;
int nhosts = 0;
int narchs = 0;
struct hostinfo *hostlist = 0;


struct cmdsw {
	char *cmd;		/* command name */
	int a1;			/* min number of args */
	int a2;			/* max number of args */
	int (*fun)();	/* function */
} commands[] = {
	{ "add",    2, 99, add_cmd },
	{ "conf",   1, 1,  conf_cmd },
	{ "delete", 2, 99, delete_cmd },
	{ "halt",   1, 1,  halt_cmd },
	{ "help",   1, 2,  help_cmd },
	{ "id",     1, 1,  id_cmd },
	{ "insert",  4, 4,  insert_cmd },
	{ "kill",   2, 99, kill_cmd },
	{ "lookup",  3, 3,  lookup_cmd },
	{ "mstat",  2, 99, mstat_cmd },
	{ "ps",     1, 2,  tasks_cmd },
	{ "pstat",  2, 99, pstat_cmd },
	{ "quit",   1, 1,  quit_cmd },
	{ "remove",  3, 3,  remove_cmd },
	{ "reset",  1, 1,  reset_cmd },
	{ "sig",    3, 99, sig_cmd },
	{ "spawn",  2, 99, spawn_cmd },
	{ "tickle", 2, 11,  tickle_cmd },
	{ "version", 1, 1,  version_cmd },
	{ 0, 0, 0, 0 }
};

static char *aliases[] = {
	"?",  "help",
	"h",  "help",
	"t",  "ps",
	"tk", "tickle",
	0
};


main(argc, argv)
	int argc;
	char **argv;
{
	char *prompt = "pvm> ";
	char cmd[512];
	int ac;
	char *av[128];
	char **p;
	struct cmdsw *csp;
	int cc;
	int i;
	int src, cod, len;

	if (argc > 1 && argv[1][0] == '-' && argv[1][1] == 'd')
		pvm_setdebug(xtoi(argv[1] + 2));

	pvm_advise(PvmDontRoute);

	ac = 1;
	av[0] = 0;
	av[1] = 0;
	i = pvm_serror(0);
	cc = pvm_start_pvmd(ac, av);
	if (cc < 0) {
		if (cc != PvmDupHost) {
			fputs("Can't start pvmd.\n", stderr);
			exit(1);
		}
		printf("pvmd already running.\n");
	}
	pvm_serror(i);
	mytid = pvm_mytid();

	(void)signal(SIGINT, SIG_IGN);

/*
	ttpcb_dumpall();
*/
	while (1) {

		if (mytid > 0) {
			while ((i = pvm_nrecv(-1, -1)) > 0) {
				pvm_bufinfo(i, &len, &cod, &src);
				printf("console: msg from t%x code %d length %d ?\n",
						src, cod, len);
				pvm_freebuf(i);
			}
		}

		printf(prompt);
		fflush(stdout);

		if (!fgets(cmd, sizeof(cmd)-1, stdin)) {
			printf("quit");
			quit_cmd();
		}

		ac = sizeof(av)/sizeof(av[0]) - 1;
		if (acav(cmd, &ac, av)) {
			fputs("command too long\n", stdout);
			continue;
		}
		if (!ac)
			continue;
		av[ac] = 0;
		for (p = aliases; *p; p += 2)
			if (!strcasecmp(av[0], *p)) {
				p++;
				break;
			}
		if (!*p)
			p = &av[0];
		for (csp = commands; csp->cmd; csp++) {
			if (!strcasecmp(csp->cmd, *p)) {
				if (ac >= csp->a1 && ac <= csp->a2)
					(csp->fun)(ac, av);
				else
					fputs("incorrect arg count\n", stdout);
				break;
			}
		}
		if (!csp->cmd)
			fputs("unknown command\n", stdout);
	}
}


/*	acav()
*
*	Parse a string into words separated by whitespace.
*	Max number of words is original value of *acp.
*
*	Trashes out the original string.
*	Returns 0 with av[0]..av[*acp - 1] pointing to the words.
*	Returns 1 if too many words.
*/

int
acav(s, acp, av)
	char *s;		/* the string to parse */
	int *acp;		/* max words in, ac out */
	char **av;		/* pointers to args */
{
	register int ac;
	register char *p = s;
	register n = *acp;

	/* separate out words of command */

	ac = 0;
	while (*p) {
		while (isspace(*p)) p++;
		if (*p) {
			if (ac >= n) {
	/* command too long */
				*acp = ac;
				return 1;
			}
			av[ac++] = p;
			while (*p && !isspace(*p)) p++;
			if (*p) *p++ = 0;
		}
	}
	*acp = ac;
	return 0;
}


/*	xtoi()
*
*	Yet another version of ascii hex to integer
*/

xtoi(p)
	char *p;
{
	int i = 0;
	char c;

	while (isxdigit(c = *p++)) {
		i = (i << 4) + c - (isdigit(c) ? '0' : (isupper(c) ? 'A' : 'a') - 10);
	}
	return i;
}


/*	axtoi()
*
*	ascii hex or decimal to integer.
*/

axtoi(p)
	char *p;
{
	if (p[0] == '0' && p[1] == 'x')
		return xtoi(p + 2);
	else
		return atoi(p);
}


int
tidtoi(p)
	char *p;
{
	if (*p == 't')
		p++;
	return xtoi(p);
}


/****************
 **  Commands  **
 **            **
 ****************/

add_cmd(ac, av)
	int ac;
	char **av;
{
	int cc;
	int *sv;
	int i;

	av++;
	ac--;
	sv = TALLOC(ac, int);
	if ((cc = pvm_addhosts(av, ac, sv)) >= 0) {
		printf("%d successful\n", cc);
		fputs("                    HOST     DTID\n", stdout);
		for (i = 0; i < ac; i++)
			if ((cc = sv[i]) < 0)
				printf("%24s %8s\n", av[i], PVMERRMSG(cc));
			else
				printf("%24s %8x\n", av[i], cc);
	}
	free(sv);
	pvm_config(&nhosts, &narchs, &hostlist);
}


conf_cmd(ac, av)
	int ac;
	char **av;
{
	int i;

	ac = ac;
	av = av;
	if (!pvm_config(&nhosts, &narchs, &hostlist)) {
		printf("%d host%s, %d data format%s\n",
			nhosts, (nhosts > 1 ? "s" : ""), narchs, (narchs > 1 ? "s" : ""));
		fputs("                    HOST     DTID     ARCH   MTU SPEED\n", stdout);
		for (i = 0; i < nhosts; i++)
			printf("%24s %8x %8s%6d%6d\n",
					hostlist[i].hi_name,
					hostlist[i].hi_tid,
					hostlist[i].hi_arch,
					hostlist[i].hi_mtu,
					hostlist[i].hi_speed);
	}
}


delete_cmd(ac, av)
	int ac;
	char **av;
{
	int cc;
	int i;
	int *sv;

	av++;
	ac--;
	sv = TALLOC(ac, int);
	if ((cc = pvm_delhosts(av, ac, sv)) >= 0) {
		printf("%d successful\n", cc);
		fputs("                    HOST  STATUS\n", stdout);
		for (i = 0; i < ac; i++) {
			printf("%24s  ", av[i]);
			if ((cc = sv[i]) < 0)
				printf("%8s\n", PVMERRMSG(cc));
			else
				printf("deleted\n");
		}
	}
	free(sv);
}


halt_cmd(ac, av)
	int ac;
	char **av;
{
	ac = ac;
	av = av;
	pvm_halt();
}


static char *helptx[] = {

	"add ADD - add hosts to virtual machine",
	"add syntax: add hostname ...",

	"conf CONF - list virtual machine configuration",
	"conf syntax: conf",
	"conf output fields:",
	"conf   HOST  - host name",
	"conf   DTID  - tid base of pvmd",
	"conf   ARCH  - xhost architecture",
	"conf   MTU   - UDP max packet size",
	"conf   SPEED - host relative speed",

	"delete DELETE - delete hosts from virtual machine",
	"delete syntax: delete hostname ...",

	"halt HALT - stop pvmds",
	"halt syntax: halt",

	"help HELP - print helpful information about a command",
	"help syntax: help [ command ]",
	"help aliases: h, ?",
	"help commands are:",
	"help   add     - add hosts to virtual machine",
	"help   conf    - list virtual machine configuration",
	"help   delete  - delete hosts from virtual machine",
	"help   halt    - stop pvmds",
	"help   id      - print console task id",
	"help   kill    - terminate tasks",
	"help   mstat   - show status of hosts",
	"help   ps      - list tasks",
	"help   pstat   - show status of tasks",
	"help   quit    - exit console",
	"help   reset   - kill all tasks",
	"help   sig     - send signal to task",
	"help   spawn   - spawn task",
	"help   version - show libpvm version",

	"id ID - print console task id",
	"id syntax: id",

	"insert INSERT - add entry to database",
	"insert syntax: insert index|-1 hexvalue",

	"kill KILL - terminate tasks",
	"kill syntax: kill tid ...",
	"kill args: -c  - kill children of pid",

	"lookup LOOKUP - find entry in database",
	"lookup syntax: lookup index|-1",

	"mstat MSTAT - show status of hosts",
	"mstat syntax: mstat name ...",

	"ps PS - list tasks",
	"ps syntax: ps [ -axh ]",
	"ps alias: t",
	"ps args: -a     - all hosts (default is local)",
	"ps       -hhost - specific host tid",
	"ps       -nhost - specific host name",
	"ps       -x     - show console task (default is not)",
	"ps output fields:",
	"ps   HOST  - host name",
	"ps   A.OUT - executable name",
	"ps   TID   - task id",
	"ps   PTID  - parent task id",
	"ps   FLAG  - status",
	"ps FLAG values:",
	"ps   c - task connected to pvmd",
	"ps   a - task waiting authorization",
	"ps   o - task connection being closed",

	"pstat PSTAT - show status of tasks",
	"pstat syntax: pstat tid ...",

	"quit QUIT - exit console",
	"quit syntax: quit",

	"remove REMOVE - delete entry from database",
	"remove syntax: remove index",

	"reset RESET - kill all tasks",
	"reset syntax: reset",

	"sig SIG - send signal to task",
	"sig syntax: sig signum task ...",

	"spawn SPAWN - spawn task",
	"spawn syntax: spawn [ options ] file [ arg ... ]",
	"spawn options: -(count)  number of tasks, default is 1",
	"spawn          -(host)   spawn on host, default is any",
	"spawn          -(ARCH)   spawn on hosts of ARCH",
	"spawn          -?        enable debugging",

	"tickle TICKLE - tickle pvmd",
	"tickle syntax: tickle how [ arg ... ]",
	"tickle alias: tk",
	"tickle how:",
	"tickle   0 dump heap",
	"tickle   1 dump host table",
	"tickle   2 dump local task table",
	"tickle   3 dump waitc list",
	"tickle   4 dump class-name list",
	"tickle   5 get debugmask",
	"tickle   6 set debugmask",

	"version VERSION - show libpvm version",
	"version syntax: version",
	0
};


help_cmd(ac, av)
	int ac;
	char **av;
{
	char **p;
	char *topic;
	int l;

	/* if not specified, topic = help */
	if (ac > 1)
		topic = av[1];
	else
		topic = "help";

	for (p = aliases; *p; p += 2)
		if (!strcasecmp(topic, *p)) {
			topic = *++p;
			break;
		}
	l = strlen(topic);

	/* search through messages for ones matching topic */
	for (p = helptx; *p; p++) {
		if (!strncasecmp(topic, *p, l) && (*p)[l] == ' ')
			printf("  %s\n", (*p) + l + 1);
	}
	return;
}


id_cmd(ac, av)
	int ac;
	char **av;
{
	ac = ac;
	av = av;
	printf("t%x\n", mytid);
}


kill_cmd(ac, av)
	int ac;
	char **av;
{
	int i;
	int tid, tid2;
	char *p;
	int host = 0;
	int cflg = 0;
	struct taskinfo *tip;
	int ntask;

	if (ac > 1 && av[1][0] == '-') {
		ac--;
		for (p = *++av; *p; p++)
			switch (*p) {

#if 0
			case 'h':
				if (isalpha(*++p)) {
					for (hip = hostlist + nhosts; --hip >= hostlist; )
						if (!(strcmp(p, hip->hi_name))) {
							host = hip->hi_tid;
							break;
						}
					if (hip < hostlist)
						printf("unknown host %s\n", p);
				} else {
					host = tidtoi(p);
				}
				*p-- = 0;
				break;
#endif

			case 'c':
				cflg = 1;
				break;

			case '-':
				break;

			default:
				printf("unknown flag -%c\n", *p);
				break;
			}
	}

	if (ac < 2) {
		fputs("incorrect arg count\n", stdout);
		return;
	}

	if (cflg && pvm_tasks(host, &ntask, &tip) < 0)
		return 1;

	while (ac > 1) {
		ac--;
		tid = tidtoi(*++av);

		if (cflg) {
			for (i = 0; i < ntask; i++)
				if (tip[i].ti_ptid == tid) {
					tid2 = tip[i].ti_tid;
					if (tid2 != mytid)
						pvm_kill(tid2);
				}

		} else {
			if (tid == mytid)
				printf("t%x: that's me.\n", tid);
			else
				pvm_kill(tid);
		}
	}
}


mstat_cmd(ac, av)
	int ac;
	char **av;
{
	int i;
	int cc;

	for (i = 1; i < ac; i++) {
		cc = pvm_mstat(av[i]);
		printf("%24s  %s\n", av[i], (cc < 0 ? PVMERRMSG(cc) : "ok"));
	}
}


pstat_cmd(ac, av)
	int ac;
	char **av;
{
	int i;
	int tid;
	int cc;

	for (i = 1; i < ac; i++) {
		tid = tidtoi(av[i]);
		cc = pvm_pstat(tid);
		printf("t%8x  %s\n", tid, (cc < 0 ? PVMERRMSG(cc) : "run"));
	}
}


quit_cmd()
{
	printf("\n");
	if (mytid > 0) {
		pvm_exit();
		printf("pvmd still running.\n");
	}
	exit(0);
}


reset_cmd(ac, av)
	int ac;
	char **av;
{
	struct taskinfo *tip;
	int ntask;
	int i;
	int j;

	ac = ac;
	av = av;
	if (!pvm_tasks(0, &ntask, &tip))
		for (i = 0; i < ntask; i++)
			if (tip[i].ti_tid && tip[i].ti_tid != mytid)
				pvm_kill(tip[i].ti_tid);

	/* XXX this is sooooo bad... */
	while ((i = pvm_lookup("pvmgs", -1, &j)) >= 0)
		pvm_delete("pvmgs", i);
}


sig_cmd(ac, av)
	int ac;
	char **av;
{
	int i;
	int signum;
	int tid;

	signum = atoi(av[1]);
	for (i = 2; i < ac; i++) {
		tid = tidtoi(av[i]);
		pvm_sendsig(tid, signum);
	}
}


spawn_cmd(ac, av)
	int ac;
	char **av;
{
	int *tids = 0;
	char *where = 0;
	int flags = 0;
	int count = 1;
	int i;

	while (av[1][0] == '-') {
		if (ac < 3) {
			fputs("incorrect arg count\n", stdout);
			return;
		}
		if (islower(av[1][1])) {
			where = av[1] + 1;
			flags |= PvmTaskHost;
		}
		if (isupper(av[1][1])) {
			where = av[1] + 1;
			flags |= PvmTaskArch;
		}
		if (av[1][1] == '?')
			flags |= PvmTaskDebug;
		if (isdigit(av[1][1]))
			count = atoi(av[1] + 1);
		av++;
		ac--;
	}
	tids = TALLOC(count > 1 ? count : 1, int);
	if ((i = pvm_spawn(av[1], &av[2], flags, where, count, tids)) >= 0) {
		printf("%d successful\n", i);
		for (i = 0; i < count; i++)
			if (tids[i] < 0)
				printf("%s\n", PVMERRMSG(tids[i]));
			else
				printf("t%x\n", tids[i]);
	}
	free(tids);
}


static char *tflgs[] = {
	"?", "?", "c",
	"a", "o",
};

char *
task_flags(f)
	int f;
{
	static char buf[64];
	int bit, i;

	sprintf(buf, "0x%02x/", f);
	i = sizeof(tflgs)/sizeof(tflgs[0]) - 1;
	bit = 1 << i;
	while (i >= 0) {
		if (f & bit) {
			strcat(buf, tflgs[i]);
			strcat(buf, ",");
		}
		bit /= 2;
		i--;
	}
	buf[strlen(buf) - 1] = 0;
	return buf;
}


tasks_cmd(ac, av)
	int ac;
	char **av;
{
	struct taskinfo *tip;
	int ntask;
	int i;
	struct hostinfo *hip = 0;
	int xflg = 0;
	char *p;
	int host = pvm_tidtohost(mytid);

	if (ac > 1) {
		for (p = av[1]; *p; p++)
			switch (*p) {

			case 'n':
				p++;
				for (hip = hostlist + nhosts; --hip >= hostlist; )
					if (!(strcmp(p, hip->hi_name))) {
						host = hip->hi_tid;
						break;
					}
				if (hip < hostlist)
					printf("unknown host %s\n", p);
				*p-- = 0;
				break;

			case 'h':
				host = tidtoi(++p);
				*p-- = 0;
				break;

			case 'a':
				host = 0;
				break;

			case 'x':
				xflg = 1;
				break;

			case '-':
				break;

			default:
				printf("unknown flag -%c\n", *p);
				break;
			}
	}

	if (!hostlist)
		pvm_config(&nhosts, &narchs, &hostlist);

	if (!pvm_tasks(host, &ntask, &tip)) {
		fputs("                    HOST        A.OUT      TID     PTID FLAG\n",
				stdout);
		for (i = 0; i < ntask; i++) {
			if (tip[i].ti_tid == mytid && !xflg)
				continue;
			if (hostlist) {
				for (hip = hostlist + nhosts - 1; hip >= hostlist; hip--)
					if (hip->hi_tid == tip[i].ti_host)
						break;
				if (hip < hostlist)
					hip = 0;
			}
			if (hip)
				printf("%24s", hip->hi_name);
			else
				printf("%24x", tip[i].ti_host);
			printf(" %12s", tip[i].ti_a_out[0] ? tip[i].ti_a_out : "-");
			if (tip[i].ti_tid == mytid)
				printf("   (cons)");
			else
				printf(" %8x", tip[i].ti_tid);
			if (tip[i].ti_ptid == mytid)
				printf("   (cons)");
			else
				if (tip[i].ti_ptid)
					printf(" %8x", tip[i].ti_ptid);
				else
					printf("        -");
			printf(" %s\n", task_flags(tip[i].ti_flag));
		}
	}
}


tickle_cmd(ac, av)
	int ac;
	char **av;
{
	int nar;
	int arg[10];
	int i;

	ac--;
	av++;
	for (nar = 0; nar < ac; nar++)
		arg[nar] = axtoi(av[nar]);

	if (!pvm_tickle(nar, arg, &nar, arg)) {
		printf("(");
		for (i = 0; i < nar; i++)
			printf(" %d", arg[i]);
		printf(" )\n");
	}
	return 0;
}


version_cmd(ac, av)
	int ac;
	char **av;
{
	ac = ac;
	av = av;
	printf("%s\n", pvm_version());
	return 0;
}


insert_cmd(ac, av)
	int ac;
	char **av;
{
	int cc;
	int data;

	ac = ac;
	data = axtoi(av[3]);
	if ((cc = pvm_insert(av[1], axtoi(av[2]), data)) >= 0)
		printf("%s, %d = 0x%08x\n", av[1], cc, data);
	else
		if (cc == PvmDupEntry)
			printf("already exists\n");
}


remove_cmd(ac, av)
	int ac;
	char **av;
{
	int cc;

	ac = ac;
	if ((cc = pvm_delete(av[1], axtoi(av[2]))) > 0)
		printf("deleted %s, %d\n", av[1], cc);
	else
		if (cc == PvmNoEntry)
			printf("no such entry\n");
}


lookup_cmd(ac, av)
	int ac;
	char **av;
{
	int cc, data = 0;

	ac = ac;
	if ((cc = pvm_lookup(av[1], axtoi(av[2]), &data)) >= 0)
		printf("%s, %d = 0x%08x\n", av[1], cc, data);
	else
		if (cc == PvmNoEntry)
			printf("no such entry\n");
}


