
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	tdpro.c
 *
 *	Entry points for messages from local tasks.
 *
$Log$
 */


#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#ifdef	SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif
#include <errno.h>
#include <stdio.h>

#ifdef IMA_I860
#include <cube.h>
#endif

#include "global.h"
#include "fromlib.h"
#include "tdpro.h"
#include "ddpro.h"
#include "protoglarp.h"
#include "myalloc.h"
#include "mesg.h"
#include "task.h"
#include "host.h"
#include "waitc.h"
#include "listmac.h"

#ifdef IMA_I860
#include "pvm3.h"
#include "mycube.h"
#endif

#ifndef	min
#define	min(a,b)	((a)<(b)?(a):(b))
#endif


/***************
 **  Globals  **
 **           **
 ***************/

extern void task_dump();

extern int debugmask;				/* from pvmd.c */
extern struct htab *hosts;			/* from pvmd.c */
extern struct task *locltasks;		/* from task.c */
extern int myndf;					/* from pvmd.c */
extern int myhostpart;				/* from pvmd.c */
extern int mytid;					/* from pvmd.c */
extern int ourudpmtu;				/* from pvmd.c */
extern int runstate;				/* from pvmd.c */
extern int tidhmask;				/* from pvmd.c */
extern int tidlmask;				/* from pvmd.c */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";
static char etext[512];		/* scratch for error log */

int tm_addhost();
int tm_config();
int tm_conn2();
int tm_connect();
int tm_db();
int tm_delhost();
int tm_exit();
int tm_halt();
int tm_mca();
int tm_mstat();
int tm_notify();
int tm_pstat();
int tm_sendsig();
int tm_spawn();
int tm_task();
int tm_tickle();

#ifdef IMA_I860
int tm_spawnack();
#endif

int (*loclswitch[])() = {
	0,
	tm_connect,
	tm_conn2,
	tm_exit,
	tm_addhost,
	tm_delhost,
	tm_config,
	tm_mstat,
	tm_halt,
	tm_tickle,
	tm_spawn,
	tm_pstat,
	tm_sendsig,
	tm_task,
	tm_mca,
	tm_notify,
	tm_db,
#ifdef IMA_I860
    tm_spawnack,
#endif
};


char *tm_names[] = {
	"tm_connect",
	"tm_conn2",
	"tm_exit",
	"tm_addhost",
	"tm_delhost",
	"tm_config",
	"tm_mstat",
	"tm_halt",
	"tm_tickle",
	"tm_spawn",
	"tm_pstat",
	"tm_sendsig",
	"tm_task",
	"tm_mca",
	"tm_notify",
	"tm_db",
#ifdef IMA_I860
	"tm_spawnack",
#endif
};

char *
tmname(code)
	int code;
{
	if (code < 1 || code > sizeof(tm_names)/sizeof(tm_names[0]))
		return "unknown";
	return tm_names[code - 1];
}


int
loclentry(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int c = mp->m_cod;

	if (debugmask & PDMMESSAGE) {
		sprintf(etext, "loclentry() from t%x code %s\n", mp->m_src,
				tmname(mp->m_cod));
		log_error(etext);
/*
		hdump(mp->m_cfrag->fr_dat, mp->m_cfrag->fr_len, "frag: ");
*/
	}

	if (mp->m_enc != 1) {
		sprintf(etext, "loclentry() message from t%x with bad enc %d\n",
				tp->t_tid, mp->m_enc);
		log_error(etext);
		goto bail;
	}

	if (c < 1 || c >= sizeof(loclswitch)/sizeof(loclswitch[0])) {
		sprintf(etext, "loclentry() message from t%x with bogus code %d\n",
				tp->t_tid, c);
		log_error(etext);
		goto bail;
	}

	if ((!(tp->t_flag & TF_CONN) && c != TM_CONNECT && c != TM_CONN2)
	|| ((tp->t_flag & TF_AUTH) && c != TM_CONN2)) {
		log_error("loclentry() non-connect message from anon task\n");
		tp->t_flag |= TF_CLOSE;
		goto bail;
	}

	(loclswitch[c])(tp, mp);

bail:
	mesg_unref(mp);
	return 0;
}


/*********************************
 **  Task request entry points  **
 **                             **
 *********************************/


/*	tm_connect()
*
*	First message from a task.
*	We write in t-auth file to prove our ident,
*	then make d-auth file for task to prove itself.
*/

#ifdef	L_tmpnam
#define	LEN_OF_TMP_NAM	L_tmpnam
#else
#define	LEN_OF_TMP_NAM	64
#endif

int
tm_connect(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int ver;						/* task's libpvm t-d proto version */
	char authfn[LEN_OF_TMP_NAM];	/* t-auth file name */
	int d;

	if (upkint(mp, &ver) || upkstr(mp, authfn, sizeof(authfn))) {
		log_error("tm_connect() bad msg format\n");
		goto bail;
	}

	/*
	*	if protocols are not compatible, send nack
	*	context will get flushed after reply is sent
	*/

	if (ver != TDPROTOCOL) {
		sprintf(etext, "tm_connect() t-d protocol mismatch (%d/%d)\n",
			ver, TDPROTOCOL);
		log_error(etext);

		mp = mesg_new(0);
		pkint(mp, TDPROTOCOL);
		pkint(mp, 0);
		pkstr(mp, "");
		mp->m_cod = TM_CONNECT;
		mp->m_flag |= MM_PRIO;
		mesg_to_task(tp, mp);
		mesg_unref(mp);
		goto bail;
	}

	/*
	*	write in t-auth file, create empty d-auth file that task
	*	must write and we'll check later
	*/

	if ((d = open(authfn, O_WRONLY, 0)) == -1) {
		log_perror("tm_connect() can't open t-auth file");
		goto bail;
	}
	(void)write(d, authfn, 1);
	(void)close(d);

	tp->t_authnam = TALLOC(LEN_OF_TMP_NAM, char, "auth");
	(void)tmpnam(tp->t_authnam);

	if ((tp->t_authfd = open(tp->t_authnam, O_RDONLY|O_CREAT|O_TRUNC, 0600))
	== -1) {
		log_perror("tm_connect() can't create d-auth file");
		goto bail;
	}

	/*
	*	task's turn to auth
	*/

	tp->t_flag |= TF_AUTH;

	mp = mesg_new(0);
	pkint(mp, TDPROTOCOL);
	pkint(mp, 1);
	pkstr(mp, tp->t_authnam);
	mp->m_cod = TM_CONNECT;
	mp->m_flag |= MM_PRIO;
	mesg_to_task(tp, mp);
	mesg_unref(mp);
	return 0;

bail:
	tp->t_flag |= TF_CLOSE;
	return 0;
}


/*	tm_conn2()
*
*	Second message from a task.
*	We check d-auth file and give it config info.
*/

int
tm_conn2(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int pid;						/* real pid of task */
	int ipid;						/* fake pid to identify task or 0 */
	struct task *tp2;				/* to search for existing context */
	int cc;
	char c;

	if (upkint(mp, &pid) || upkint(mp, &ipid)) {
		log_error("tm_conn2() bad msg format\n");
		goto bail;
	}
	if (!ipid)
		ipid = pid;

	if (!(tp->t_flag & TF_AUTH)) {
		sprintf(etext, "tm_conn2() message from t%x, TF_AUTH not set\n",
				tp->t_tid);
		log_error(etext);
		return 0;
	}

	/*
	*	check that task could write d-auth file
	*/

	if ((cc = read(tp->t_authfd, &c, 1)) == -1) {
		log_perror("tm_conn2() can't read d-auth file");
		return 0;
	}

	if (cc != 1) {
		log_error("tm_conn2() task didn't validate itself\n");
		goto bail;
	}

	(void)close(tp->t_authfd);
	tp->t_authfd = -1;
	(void)unlink(tp->t_authnam);
	PVM_FREE(tp->t_authnam);
	tp->t_authnam = 0;

	/*
	*	if task spawned by us, already has a context
	*/

#ifdef IMA_I860
  /* do this for host procs only */
  if (pid > 0) {
#endif
	if (tp2 = task_find(myhostpart + ipid)) {

	/* brundle-fly the contexts together */

		tp2->t_sock = tp->t_sock;
		tp2->t_sad = tp->t_sad;
		tp2->t_salen = tp->t_salen;
		tp2->t_pid = pid;
		tp2->t_rxp = tp->t_rxp;
		tp->t_sock = -1;	/* tp will be freed by loclinput() */
		tp->t_rxp = 0;
		tp = tp2;
		if (debugmask & PDMTASK) {
			sprintf(etext, "tm_conn2() reconnect task t%x pid %d\n",
					tp->t_tid, tp->t_pid);
			log_error(etext);
		}

	} else {

	/* otherwise set up its context */

		tp->t_tid = myhostpart + pid;
		tp->t_ptid = 0;
		tp->t_pid = pid;
		if (debugmask & PDMTASK) {
			sprintf(etext, "tm_connect() new task t%x pid %d\n",
					tp->t_tid, tp->t_pid);
			log_error(etext);
		}
	}
#ifdef IMA_I860
  }
#endif
	tp->t_flag &= ~TF_AUTH;
	tp->t_flag |= TF_CONN;

	mp = mesg_new(0);
	pkint(mp, 1);
#ifdef IMA_I860
	if (pid > 0)
#endif
		pkint(mp, tp->t_tid);
	pkint(mp, tp->t_ptid);
	pkint(mp, ourudpmtu);
	pkint(mp, myndf);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_CONN2;
	mp->m_flag |= MM_PRIO;
	sendmessage(mp);
	return 0;

bail:
	tp->t_flag |= TF_CLOSE;
	return 0;
}


/*	tm_exit()
*
*	Last message from a task.  This is the nice way of disconnecting.
*/

int
tm_exit(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	mp = mesg_new(0);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_EXIT;
	sendmessage(mp);
	task_cleanup(tp);
	tp->t_flag |= TF_CLOSE;
	return 0;
}


int
tm_pstat(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int tid;
	struct hostd *hp;
	struct waitc *wp;

	/* unpack and sanity check tid */

	if (upkint(mp, &tid)) {
		log_error("tm_pstat() bad msg format\n");
		return 0;
	}
	if (!TIDISTASK(tid)) {
		sprintf(etext, "tm_pstat() bad tid %x\n", tid);
		log_error(etext);
		return 0;
	}

	/* nack if no such host */

	if (!(hp = tidtohost(hosts, tid))) {
		mp = mesg_new(0);
		mp->m_dst = tp->t_tid;
		mp->m_cod = TM_PSTAT;
		pkint(mp, PvmNoTask);
		sendmessage(mp);
		return 0;
	}

	/* else make a wait context and send query */

	wp = wait_new(WT_PSTAT);
	wp->wa_tid = tp->t_tid;
	wp->wa_on = hp->hd_hostpart;

	mp = mesg_new(0);
	mp->m_dst = hp->hd_hostpart | TIDPVMD;
	mp->m_cod = DM_PSTAT;
	mp->m_wid = wp->wa_wid;
	pkint(mp, tid);
	sendmessage(mp);
	return 0;
}


/*	tm_addhost()
*
*	Task requesting to add hosts to virtual machine.  Exit point is
*	here or dm_addhostack().
*/

int
tm_addhost(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int count;
	char buf[128];
	struct waitc *wp;

	/* sanity check the message */

	if (upkint(mp, &count))
		goto bad;
	if (count < 1 || count > (tidhmask >> ffs(tidhmask) - 1))
		goto bad;
	while (count-- > 0)
		if (upkstr(mp, buf, sizeof(buf)))
			goto bad;

	/* make a wait channel for the task */

	wp = wait_new(WT_ADDHOST);
	wp->wa_tid = tp->t_tid;
	wp->wa_on = hosts->ht_hosts[hosts->ht_master]->hd_hostpart;

	/* forward message to master pvmd */

	mp->m_dst = hosts->ht_hosts[hosts->ht_master]->hd_hostpart | TIDPVMD;
	mp->m_cod = DM_ADDHOST;
	mp->m_wid = wp->wa_wid;
	mp->m_ref++;
	sendmessage(mp);
	return 0;

bad:
	sprintf(etext, "tm_addhost() from t%x bad msg format\n", mp->m_src);
	log_error(etext);
	return 0;
}


free_wait_spawn(wxp)
	struct waitc_spawn *wxp;
{
	int i;

	if (wxp->w_file)
		PVM_FREE(wxp->w_file);
	if (wxp->w_argv) {
		for (i = 0; i < wxp->w_argc; i++)
			if (wxp->w_argv[i])
				PVM_FREE(wxp->w_argv[i]);
		PVM_FREE(wxp->w_argv);
	}
	if (wxp->w_ht)
		ht_free(wxp->w_ht);
	if (wxp->w_vec)
		PVM_FREE(wxp->w_vec);
	PVM_FREE(wxp);
	return 0;
}


/*	tm_spawn()
*
*	Task requesting to spawn other tasks.  Exit point for this
*	request is here or dm_spawnack().
*/

int
tm_spawn(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	char *file = 0;				/* executable */
	int flags;					/* options */
	char *where = 0;			/* location from req */
	int count;					/* num of tasks */
	int argc;					/* num of args */
	char **argv = 0;			/* args */

	struct waitc *wp;			/* 'seed' waitc */
	struct waitc_spawn *wxp;	/* extra data */
	int *vec;					/* result/status vector */
	struct htab *htp;			/* set of usable hosts */
	struct hostd *hp;
	int hh;
	int i;

	/*
	* unpack spawn command from task
	*/

	if (upkstralloc(mp, &file)
	|| upkint(mp, &flags)
	|| upkstralloc(mp, &where)
	|| upkint(mp, &count)
	|| upkint(mp, &argc))
		goto bad;

	if (count < 1)
		goto bad;

	argc++;
	bzero((char*)(argv = TALLOC(argc, char*, "argv")),
			argc * sizeof(char*));
	argc--;
	for (i = 0; i < argc; i++)
		if (upkstralloc(mp, &argv[i]))
			goto bad;

	/*
	* make host set containing hosts (matching where option)
	*/

	if ((flags & (PvmTaskHost|PvmTaskArch)) && !where)
		goto bad;

	htp = ht_new(1);

	if (flags & PvmTaskHost) {			/* given host */
		if (hp = nametohost(hosts, where))
			ht_insert(htp, hp);

	} else {
		if (flags & PvmTaskArch) {		/* given arch */
			for (hh = hosts->ht_last; hh > 0; hh--)
				if ((hp = hosts->ht_hosts[hh]) && !strcmp(where, hp->hd_arch))
					ht_insert(htp, hp);

		} else {						/* anywhere */
			ht_merge(htp, hosts);
		}
	}
/*
	ht_dump(htp);
*/

	/*
	* assign each task to a host
	*/

	wp = wait_new(WT_SPAWN);
	wp->wa_tid = tp->t_tid;

	wxp = TALLOC(1, struct waitc_spawn, "waix");
	bzero((char*)wxp, sizeof(struct waitc_spawn));
	wp->wa_spec = (void*)wxp;

	wxp->w_file = file;
	file = 0;
	wxp->w_flags = flags;
	wxp->w_argc = argc;
	wxp->w_argv = argv;
	argv = 0;
	wxp->w_ht = htp;
	vec = TALLOC(count, int, "vec");
	bzero((char*)vec, count * sizeof(int));
	wxp->w_vec = vec;
	wxp->w_togo = wxp->w_veclen = count;

	assign_tasks(wp);

	/* if already done, reply to task */

	if (wp->wa_peer == wp) {
		assign_tasks(wp);
	}

	wait_delete(wp);
	goto cleanup;

bad:
	sprintf(etext, "tm_spawn() from t%x bad msg format\n", mp->m_src);
	log_error(etext);

cleanup:
	if (file)
		PVM_FREE(file);
	if (where)
		PVM_FREE(where);
	if (argv) {
		for (i = 0; i < argc; i++)
			if (argv[i])
				PVM_FREE(argv[i]);
		PVM_FREE(argv);
	}
	return 0;
}


/*	assign_tasks()
*
*	This is only called when no replies are pending (at the beginning
*	of the operation or when all pvmds have checked back in).
*/

assign_tasks(wp)
	struct waitc *wp;		/* (any) waitc in peer group for this op */
{
	struct waitc_spawn *wxp = (struct waitc_spawn*)wp->wa_spec;
	struct htab *htp;				/* set of hosts to use */
	int *vec;						/* result/status vector */
	int veclen;						/* length of vector */
	int count = 0;					/* num of tasks to be assigned */
	int nh;							/* num of hosts to assign tasks */
	int a = 0;						/* accum for finding hosts */
	int na = 0;						/* num tasks assigned to a host */
	struct waitc *wp2;
	struct hostd *hp;
	struct mesg *mp;
	int t;
	int i;

	static int lasthh = 0;		/* for assigning hosts */

	if (!wxp)
		return 0;

	htp = wxp->w_ht;
	vec = wxp->w_vec;
	veclen = wxp->w_veclen;

	/*
	* if no hosts left, fill unassigned entries with PvmNoHost
	*/

	if (!htp->ht_cnt)
		for (t = veclen; t-- > 0; )
			if (!vec[t])
				vec[t] = PvmNoHost;

	/*
	* count tasks to be assigned, if none left reply to task
	*/

	for (t = veclen; t-- > 0; )
		if (!vec[t])
			count++;

	if (!count) {
		mp = mesg_new(0);
		mp->m_dst = wp->wa_tid;
		mp->m_cod = TM_SPAWN;
		pkint(mp, wxp->w_veclen);
		for (t = 0; t < wxp->w_veclen; t++)
			pkint(mp, wxp->w_vec[t]);
		sendmessage(mp);

		free_wait_spawn(wxp);
		wp->wa_spec = 0;
		return 0;
	}

	/*
	* assign tasks to hosts
	*/

	nh = min(htp->ht_cnt, count);

	/* find first host to assign */

	if (lasthh > htp->ht_last)
		lasthh = 0;
	while (!htp->ht_hosts[lasthh])
		if (++lasthh > htp->ht_last)
			lasthh = 1;
	hp = htp->ht_hosts[lasthh];

	for (t = 0; t < veclen && vec[t]; t++);

	while (t < veclen) {
/*
		sprintf(etext, "assign_tasks() %s <- %d\n", hp->hd_name, t);
		log_error(etext);
*/

		vec[t] = hp->hd_hostpart;
		na++;

	/* when enough tasks for this host, move to next */

		if ((a += nh) >= count) {
			a -= count;

			wp2 = wait_new(WT_SPAWN);
			wp2->wa_tid = wp->wa_tid;
			wp2->wa_on = hp->hd_hostpart;
			wp2->wa_spec = wp->wa_spec;
			LISTPUTBEFORE(wp, wp2, wa_peer, wa_rpeer);

			mp = mesg_new(0);
			pkint(mp, wp->wa_tid);
			pkstr(mp, wxp->w_file);
			pkint(mp, wxp->w_flags);
			pkint(mp, na);
			pkint(mp, wxp->w_argc);
			for (i = 0; i < wxp->w_argc; i++)
				pkstr(mp, wxp->w_argv[i]);
			mp->m_dst = hp->hd_hostpart | TIDPVMD;
			mp->m_cod = DM_SPAWN;
			mp->m_wid = wp2->wa_wid;

			do {
				if (++lasthh > htp->ht_last)
					lasthh = 1;
			} while (!htp->ht_hosts[lasthh]);

			sendmessage(mp);

			hp = htp->ht_hosts[lasthh];
			na = 0;
		}
		for (t++ ; t < veclen && vec[t]; t++);
	}
	return 0;
}


/*	tm_sendsig()
*
*	Task sending a signal to another task.
*/

int
tm_sendsig(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int tid;

	if (upkint(mp, &tid)) {
		log_error("tm_sendsig() bad msg format\n");
		return 0;
	}
	if (!TIDISTASK(tid)) {
		sprintf(etext, "tm_sendsig() bad tid %x\n", tid);
		log_error(etext);
		return 0;
	}
	mp->m_dst = (tid & TIDHOST) | TIDPVMD;
	mp->m_cod = DM_SENDSIG;
	mp->m_wid = 0;
	mp->m_ref++;
	sendmessage(mp);

	mp = mesg_new(0);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_SENDSIG;
	sendmessage(mp);
	return 0;
}


/*	tm_config()
*
*	Task wants machine configuration
*/

int
tm_config(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int hh;
	struct hostd *hp;

	mp = mesg_new(0);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_CONFIG;

	pkint(mp, hosts->ht_cnt);
	pkint(mp, hosts->ht_narch);
	for (hh = 1; hh <= hosts->ht_last; hh++) {
		if (hp = hosts->ht_hosts[hh]) {
			pkint(mp, hp->hd_hostpart);
			pkstr(mp, hp->hd_name);
			pkstr(mp, hp->hd_arch ? hp->hd_arch : "");
			pkint(mp, hp->hd_mtu);
			pkint(mp, hp->hd_speed);
		}
	}
	sendmessage(mp);
	return 0;
}


/*	tm_halt()
*
*	Command from task to stop master pvmd.
*/

int
tm_halt(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	mp = mesg_new(0);
	mp->m_cod = DM_HALT;
	mp->m_dst = hosts->ht_hosts[hosts->ht_master]->hd_hostpart | TIDPVMD;
	sendmessage(mp);

	mp = mesg_new(0);
	mp->m_cod = TM_HALT;
	mp->m_dst = tp->t_tid;
	sendmessage(mp);
	return 0;
}


/*	tm_task()
*
*	Task wants a list of tasks.
*/

int
tm_task(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int where;
	struct task *tp2;
	struct mesg *mp2;
	struct waitc *wp, *wp2 = 0;
	int hh;

	upkint(mp, &where);

	mp = mesg_new(0);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_TASK;

	/* if where nonzero, find the host */

	if (where && where != myhostpart) {
		for (hh = hosts->ht_last; hh > 0; hh--)
			if (hosts->ht_hosts[hh]
			&& hosts->ht_hosts[hh]->hd_hostpart == where)
				break;

		if (hh < 1) {				/* err, no such host */
			pkint(mp, PvmNoHost);
			sendmessage(mp);
			return 0;
		}

	/* dial direct */
		pkint(mp, 0);
		wp = wait_new(WT_TASK);
		wp->wa_mesg = mp;
		wp->wa_tid = tp->t_tid;
		wp->wa_on = hosts->ht_hosts[hh]->hd_hostpart;

		mp = mesg_new(0);
		mp->m_cod = DM_TASK;
		mp->m_dst = hosts->ht_hosts[hh]->hd_hostpart | TIDPVMD;
		mp->m_wid = wp->wa_wid;
		sendmessage(mp);
		return 0;
	}

	/* pack list of local tasks */

	pkint(mp, 0);	/* no error */
	for (tp2 = locltasks->t_link; tp2 != locltasks; tp2 = tp2->t_link) {
		pkint(mp, tp2->t_tid);
		pkint(mp, tp2->t_ptid);
		pkint(mp, myhostpart);
		pkint(mp, tp2->t_flag);
		pkstr(mp, tp2->t_a_out ? tp2->t_a_out : "");
	}

	/* if no other hosts, reply right away */

	if (hosts->ht_cnt == 1 || where == myhostpart) {
		sendmessage(mp);
		return 0;
	}

	/* send request to each other host */

	mp2 = mesg_new(0);
	mp2->m_cod = DM_TASK;

	for (hh = hosts->ht_last; hh > 0; hh--) {
		if (hh == hosts->ht_local || !hosts->ht_hosts[hh])
			continue;

		wp = wait_new(WT_TASK);
		mp->m_ref++;
		wp->wa_mesg = mp;
		wp->wa_tid = tp->t_tid;
		wp->wa_on = hosts->ht_hosts[hh]->hd_hostpart;

		if (wp2) {
			LISTPUTBEFORE(wp2, wp, wa_peer, wa_rpeer);
		} else
			wp2 = wp;

		mp2->m_dst = hosts->ht_hosts[hh]->hd_hostpart | TIDPVMD;
		mp2->m_wid = wp->wa_wid;
		mp2->m_ref++;
		sendmessage(mp2);
	}
	mesg_unref(mp2);
	mesg_unref(mp);
	return 0;
}


int
tm_tickle(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int how;

	upkint(mp, &how);
	sprintf(etext, "tm_tickle() #%d\n", how);
	log_error(etext);

	switch (how) {

	case 0:
		i_dump(1);
		break;

	case 1:
		ht_dump(hosts);
		break;

	case 2:
		task_dump();
		break;

	case 3:
		wait_dumpall();
		break;

    case 4:
        nmd_dumpall();
        break;

	default:
		sprintf(etext, "tm_tickle() don't know #%d\n", how);
		log_error(etext);
		break;
	}

	mp = mesg_new(0);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_TICKLE;
	sendmessage(mp);
	return 0;
}


/*	tm_delhost()
*
*	Task requesting to add hosts to virtual machine.  Exit point is
*	here or dm_delhostack().
*/

int
tm_delhost(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int count;
	char buf[128];
	struct waitc *wp;

	/* sanity check the message */

	if (upkint(mp, &count))
		goto bad;
	if (count < 1 || count > (tidhmask >> ffs(tidhmask) - 1))
		goto bad;
	while (count-- > 0)
		if (upkstr(mp, buf, sizeof(buf)))
			goto bad;

	/* make a wait channel for the task */

	wp = wait_new(WT_DELHOST);
	wp->wa_tid = tp->t_tid;
	wp->wa_on = hosts->ht_hosts[hosts->ht_master]->hd_hostpart;

	/* forward message to master pvmd */

	mp->m_dst = hosts->ht_hosts[hosts->ht_master]->hd_hostpart | TIDPVMD;
	mp->m_cod = DM_DELHOST;
	mp->m_wid = wp->wa_wid;
	mp->m_ref++;
	sendmessage(mp);
	return 0;

bad:
	sprintf(etext, "tm_delhost() from t%x bad msg format\n", mp->m_src);
	log_error(etext);
	return 0;
}


static int
int_compare(i, j)
	int *i, *j;
{
	return *i - *j;
}


int
tm_mca(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	static int lastmca = 0;		/* last-assigned mca local part */
	struct mca *mcap;			/* mca descriptor */
	int ndst;					/* num of dst tids */
	int *dsts;					/* dst tids */
	int tid;
	int i1, i2;

	/* generate new mca local part */

	if (++lastmca > tidlmask)
		lastmca = 0;

	/*
	* unpack list of dst tids from message.
	* normalize local dsts to nonzero hostpart.
	* discard tids for nonexistent local tasks or foreign hosts.
	*/

	mcap = mca_new();
	mcap->mc_tid = myhostpart | TIDGID | lastmca;
	upkint(mp, &ndst);
	dsts = TALLOC(ndst, int, "dsts");
	for (i1 = 0; i1 < ndst; i1++) {
		upkint(mp, &tid);
		if (!(tid & tidhmask) || (tid & tidhmask) == myhostpart)
			tid |= myhostpart;

		else
			if (!tidtohost(hosts, tid)) {
				i1--;
				ndst--;
				continue;
			}
		dsts[i1] = tid;
	}

	if (ndst < 1)		/* if no dsts left then just reply to task */
		goto noguys;

	/*
	* sort dst tids
	* send DM_MCA messages containing tids to dst hosts
	* make list of dst hosts for us
	*/

	qsort((char*)dsts, ndst, sizeof(int), int_compare);

	mcap->mc_dsts = TALLOC(ndst, int, "mcad");	/* XXX cheat, too much space */
	mcap->mc_ndst = 0;

	for (i2 = 0; (i1 = i2) < ndst; ) {
		tid = dsts[i1] & tidhmask;
		while (++i2 < ndst && tid == (dsts[i2] & tidhmask)) ;
		mp = mesg_new(0);
		mp->m_dst = (tid |= TIDPVMD);
		mp->m_cod = DM_MCA;
		pkint(mp, mcap->mc_tid);
		pkint(mp, i2 - i1);
		while (i1 < i2)
			pkint(mp, dsts[i1++]);
		sendmessage(mp);
		mcap->mc_dsts[mcap->mc_ndst++] = tid;
	}

noguys:
	PVM_FREE(dsts);

	/*
	* tag task descriptor with mca desc and send mca back to task
	*/

	tp->t_mca = mcap;

	if (debugmask & PDMMESSAGE) {
		sprintf(etext, "tm_mca() made mca %x for t%x\n",
				mcap->mc_tid, tp->t_tid);
		log_error(etext);
	}

	mp = mesg_new(0);
	pkint(mp, mcap->mc_tid);
	mp->m_dst = tp->t_tid;
	mp->m_cod = TM_MCA;
	sendmessage(mp);
	return 0;
}


int
tm_notify(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	int what, code, count, tid;
	struct hostd *hp;
	struct mesg *mp2;
	struct waitc *wp;
	int i;

	if (upkint(mp, &what) || upkint(mp, &code) || upkint(mp, &count)
	|| count < 0) {
		log_error("tm_notify() bad msg format\n");
		return 0;
	}

	mp2 = mesg_new(0);
	mp2->m_dst = tp->t_tid;
	mp2->m_cod = TM_NOTIFY;
	sendmessage(mp2);

	for (i = 0; i < count; i++) {
		if (upkint(mp, &tid)) {
			log_error("tm_notify() bad msg format\n");
			return 0;
		}
		if (!(hp = tidtohost(hosts, tid)))
			goto dead;

		if (hp->hd_hostpart == myhostpart) {
			if (!task_find(tid))
				goto dead;
			wp = wait_new(WT_TASKX);
			wp->wa_tid = tp->t_tid;
			wp->wa_on = tid;
			wp->wa_dep = code;

		} else {
			wp = wait_new(WT_TASKX);
			wp->wa_tid = tp->t_tid;
			wp->wa_on = tid;
			wp->wa_dep = code;
			mp2 = mesg_new(0);
			pkint(mp2, what);
			pkint(mp2, tid);
			mp2->m_dst = hp->hd_hostpart | TIDPVMD;
			mp2->m_cod = DM_NOTIFY;
			mp2->m_wid = wp->wa_wid;
			sendmessage(mp2);
		}
		continue;

dead:
		mp2 = mesg_new(0);
		mp2->m_dst = tp->t_tid;
		mp2->m_cod = code;
		pkint(mp2, tid);
		sendmessage(mp2);
	}
	return 0;
}


int
tm_mstat(tp, mp)
	struct task *tp;
	struct mesg *mp;
{
	char *name;
	struct hostd *hp;
	int hh;
	struct waitc *wp;

	if (upkstralloc(mp, &name)) {
		log_error("tm_mstat() bad msg format\n");
		return 0;
	}

	for (hh = hosts->ht_last; hh > 0; hh--)
		if ((hp = hosts->ht_hosts[hh]) && !strcmp(name, hp->hd_name))
			break;

	PVM_FREE(name);

	if (hh < 1) {
		mp = mesg_new(0);
		mp->m_dst = tp->t_tid;
		mp->m_cod = TM_MSTAT;
		pkint(mp, PvmNoHost);
		sendmessage(mp);
		return 0;
	}

	wp = wait_new(WT_MSTAT);
	wp->wa_tid = tp->t_tid;
	wp->wa_on = hp->hd_hostpart;

	mp = mesg_new(0);
	mp->m_dst = hp->hd_hostpart | TIDPVMD;
	mp->m_cod = DM_PSTAT;
	mp->m_wid = wp->wa_wid;
	pkint(mp, hp->hd_hostpart | TIDPVMD);
	sendmessage(mp);
	return 0;
}

#ifdef IMA_I860
int
tm_spawnack(tp, mp)
    struct task    *tp;
    struct mesg    *mp;
{
    int             err, *tids;
    register int    i;
    struct cube    *cp;
    char            cname[CUBENAMELEN];

    if (upkint(mp, &err) || upkstr(mp, cname, sizeof(cname))) {
        sprintf(etext, "tm_spawnack() from t%x bad msg format\n", mp->m_src);
        log_error(etext);
        return 0;
    }
    if (!(cp = cube_byname(cname))) {
        sprintf(etext, "tm_spawnack() cube %s not on active list\n", cname);
        log_error(etext);
        return 0;
    }
    if (!err) {
        int             cid = -1;

        if (upkint(mp, &cid)) {
            sprintf(etext,"tm_spawnack() from t%x bad msg format\n", mp->m_src);
            log_error(etext);
            return 0;
        }
        if (debugmask & PDMCUBE) {
            sprintf(etext, "tm_spawnack() got cube %s id=%d\n", cname, cid);
            log_error(etext);
        }
        cp->c_id = cid;
    }
    tids = TALLOC(cp->c_size, int, "tids");
    if (!err) {
        register struct task *tkp = 0;
        register int   *tid;

        for (i = 0, tid = tids; i < cp->c_size; i++) {
            tkp = task_new(0);
            tkp->t_ptid = cp->c_ptid;
            upkint(mp, tid++);
            tkp->t_tid = tids[i];
            tkp->t_a_out = STRALLOC(cp->c_file);
        }
        if (debugmask & PDMTASK)
            for (i = 0; i < cp->c_size; i++) {
                sprintf(etext, "tm_spawnack() started t%x\n", tids[i]);
                log_error(etext);
            }
    } else
        for (i = 0; i < cp->c_size; i++)
            tids[i] = err;

    mp = mesg_new(0);
    pkint(mp, cp->c_size);
    for (i = 0; i < cp->c_size; i++)
        pkint(mp, tids[i]);
    mp->m_dst = cp->c_msrc;
    mp->m_cod = DM_SPAWNACK;
    mp->m_wid = cp->c_mwid;
    sendmessage(mp);

    mp = mesg_new(0);
    mp->m_dst = tp->t_tid;
    mp->m_cod = TM_SPAWNACK;
    sendmessage(mp);

    if (tids)
        PVM_FREE(tids);
	if (err)
		cp->c_size = 0;

    return 0;
}
#endif	/*IMA_I860*/


int
tm_db(tp, mp)
    struct task *tp;
    struct mesg *mp;
{
    struct waitc *wp;

    wp = wait_new(WT_DB);
    wp->wa_tid = tp->t_tid;
    wp->wa_on = hosts->ht_hosts[hosts->ht_master]->hd_hostpart;

    mp->m_ref++;
    mp->m_dst = hosts->ht_hosts[hosts->ht_master]->hd_hostpart | TIDPVMD;
    mp->m_cod = DM_DB;
    mp->m_wid = wp->wa_wid;
    sendmessage(mp);
    return 0;
}
