
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	pack.c
 *
 *	Typed packing/unpacking, message buffer manip.
 *
$Log$
 */

#include <stdio.h>
#ifdef IMA_BSD386
#include <machine/endian.h>
#endif
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "pvm3.h"
#include "global.h"
#include "tdpro.h"
#include "ddpro.h"
#include "frag.h"
#include "umbuf.h"
#include "listmac.h"

struct dat_blk {
	int type;		/* data type */
	void *base;		/* base address */
	int cnt;		/* number of elements */
	int std;		/* lead to next element */
};

/* struct dat_blk type codes */
#define	INP_BYTE	1
#define	INP_INT		2
#define	INP_SHORT	3
#define	INP_LONG	4
#define	INP_FLOAT	5
#define	INP_DFLOAT	6
#define	INP_CPLX	7
#define	INP_DCPLX	8


/***************
 **  Globals  **
 **           **
 ***************/

extern struct mhp *midh;		/* from umbuf.c */
extern int midhsiz;				/* from umbuf.c */
extern int myndf;				/* from lpvm.c */
extern int mypvmtid;			/* from lpvm.c */
extern int ourudpmtu;			/* from lpvm.c */

int rbufmid = 0;				/* current recv/unpack mid */
int sbufmid = 0;				/* current send/pack mid */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";


/***************************
 **  Raw-format encoders  **
 **                       **
 ***************************/

/*	bytepk()
*
*	Insert a stream of bytes into sbuf.  Allocate more fragments as
*	necessary.
*	Returns 0 else PvmNoMem if malloc fails.
*/

static int
bytepk(cp, num, siz, lnc)
	char *cp;	/* base of data */
	int num;	/* num of chunks */
	int siz;	/* size of chunk */
	int lnc;	/* lead to next chunk */
{
	struct umbuf *up = midtobuf(sbufmid);	/* working message */
	struct frag *fp;			/* working frag */
	int togo;					/* bytes left in chunk */
	int r;						/* bytes (space) left in frag */

	if (siz == lnc) {		/* if contiguous, treat as single chunk */
		lnc = (siz *= num);
		num = 1;
	}
	lnc -= siz;		/* now bytes between chunks */

	while (num-- > 0) {		/* copy chunks until done */

		for (togo = siz; togo > 0; ) {
			fp = up->ub_frag->fr_rlink;
			r = fp->fr_max - (fp->fr_dat - fp->fr_buf) - fp->fr_len;

			if (togo <= r) {	/* space in frag for entire chunk */
				bcopy(cp, fp->fr_dat + fp->fr_len, togo);
				fp->fr_len += togo;
				cp += togo;
				togo = 0;

			} else {
				if (r > 0) {	/* space for part of chunk */
					bcopy(cp, fp->fr_dat + fp->fr_len, r);
					fp->fr_len += r;
					togo -= r;
					cp += r;

				} else {		/* no space, add new frag */
					if (r = enc_step())
						return r;
				}
			}
		}
		cp += lnc;
	}
	return 0;
}


/*	byteupk()
*
*	Extract bytes from rbuf.
*	Returns 0 else PvmNoData if end of message reached early.
*/

static int
byteupk(cp, num, siz, lnc)
	char *cp;	/* base of data */
	int num;	/* num of chunks */
	int siz;	/* size of chunk */
	int lnc;	/* lead to next chunk */
{
	struct umbuf *up = midtobuf(rbufmid);	/* working message */
	struct frag *fp;			/* working frag */
	int togo;					/* bytes left in chunk */
	int r;						/* bytes (data) left in frag */

	if (siz == lnc) {		/* if contiguous, treat as single chunk */
		lnc = (siz *= num);
		num = 1;
	}
	lnc -= siz;		/* now bytes between chunks */

	while (num-- > 0) {		/* copy chunks until done */

		for (togo = siz; togo > 0; ) {
			fp = up->ub_cfrag;
			r = fp->fr_len - up->ub_cpos;

			if (togo <= r) {	/* frag contains rest of chunk */
				bcopy(fp->fr_dat + up->ub_cpos, cp, togo);
				up->ub_cpos += togo;
				cp += togo;
				togo = 0;

			} else {
				if (r > 0) {	/* frag contains part of chunk */
					bcopy(fp->fr_dat + up->ub_cpos, cp, r);
					up->ub_cpos += r;
					togo -= r;
					cp += r;

				} else {		/* no space, add new frag */
					if (r = dec_step())
						return r;
				}
			}
		}
		cp += lnc;
	}
	return 0;
}


static int
enc_raw_init()
{
	return 0;
}


static int
dec_raw_init()
{
	struct umbuf *up = midtobuf(rbufmid);

	up->ub_cfrag = up->ub_frag->fr_link;
	up->ub_cpos = 0;
	return 0;
}


static int
enc_raw_any(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	return bytepk((char*)vp, cnt, siz, std * siz);
}


static int
dec_raw_any(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	return byteupk((char*)vp, cnt, siz, std * siz);
}


/***************************
 **  XDR-format encoders  **
 **                       **
 ***************************/

static int
enc_xdr_init()
{
	struct umbuf *up = midtobuf(sbufmid);
	struct frag *fp = up->ub_frag->fr_link;

	xdrmem_create(&up->ub_encxdr,
			fp->fr_dat,
			(unsigned)(fp->fr_max - (fp->fr_dat - fp->fr_buf)),
			XDR_ENCODE);
	return 0;
}


static int
enc_xdr_step()
{
	struct umbuf *up = midtobuf(sbufmid);
	struct frag *fp = up->ub_frag->fr_rlink;
	int cc;

	fp->fr_len = xdr_getpos(&up->ub_encxdr);
	if (cc = enc_step())
		return cc;
	fp = fp->fr_link;
	xdrmem_create(&up->ub_encxdr,
			fp->fr_dat,
			(unsigned)(fp->fr_max - (fp->fr_dat - fp->fr_buf)),
			XDR_ENCODE);
	return 0;
}


static int
enc_xdr_byte(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	struct frag *fp;
	int cc;

	fp = up->ub_frag->fr_rlink;
	if (cc = bytepk((char*)vp, cnt, 1, std))
		return cc;
	if (fp != up->ub_frag->fr_rlink) {
		fp = up->ub_frag->fr_rlink;
		xdrmem_create(&up->ub_encxdr,
				fp->fr_dat,
				(unsigned)(fp->fr_max - (fp->fr_dat - fp->fr_buf)),
				XDR_ENCODE);
	}
	fp->fr_len = (fp->fr_len + 3) & ~3;
	xdr_setpos(&up->ub_encxdr, fp->fr_len);
	return 0;
}


static int
enc_xdr_short(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register short *np;
	int cc = 0;

	for (np = (short*)vp; cnt-- > 0; np += std)
		if (!xdr_short(xob, np)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_short(xob, np)) {
					cc = PvmNoMem;
					break;
				}
		}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_int(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register int *np;
	int cc = 0;

	for (np = (int*)vp; cnt-- > 0; np += std)
		if (!xdr_int(xob, np)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_int(xob, np)) {
					cc = PvmNoMem;
					break;
				}
		}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_long(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register long *np;
	int cc = 0;

	for (np = (long*)vp; cnt-- > 0; np += std)
		if (!xdr_long(xob, np)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_long(xob, np)) {
					cc = PvmNoMem;
					break;
				}
		}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_float(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register float *fp;
	int cc = 0;

	for (fp = (float*)vp; cnt-- > 0; fp += std)
		if (!xdr_float(xob, fp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_float(xob, fp)) {
					cc = PvmNoMem;
					break;
				}
		}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_double(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register double *dp;
	int cc = 0;

	for (dp = (double*)vp; cnt-- > 0; dp += std)
		if (!xdr_double(xob, dp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_double(xob, dp)) {
					cc = PvmNoMem;
					break;
				}
		}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_cplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register float *xp;
	int cc = 0;

	std = std * 2 - 1;
	for (xp = (float*)vp; cnt-- > 0; xp += std) {
		if (!xdr_float(xob, xp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_float(xob, xp)) {
					cc = PvmNoMem;
					break;
				}
		}
		xp++;
		if (!xdr_float(xob, xp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_float(xob, xp)) {
					cc = PvmNoMem;
					break;
				}
		}
	}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
enc_xdr_dcplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(sbufmid);
	XDR *xob = &up->ub_encxdr;
	register double *zp;
	int cc = 0;

	std = std * 2 - 1;
	for (zp = (double*)vp; cnt-- > 0; zp += std) {
		if (!xdr_double(xob, zp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_double(xob, zp)) {
					cc = PvmNoMem;
					break;
				}
		}
		zp++;
		if (!xdr_double(xob, zp)) {
			if (cc = enc_xdr_step())
				break;
			else
				if (!xdr_double(xob, zp)) {
					cc = PvmNoMem;
					break;
				}
		}
	}
	up->ub_frag->fr_rlink->fr_len = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_init()
{
	struct umbuf *up = midtobuf(rbufmid);
	struct frag *fp = up->ub_frag->fr_link;

	up->ub_cfrag = fp;
	up->ub_cpos = 0;
	xdrmem_create(&up->ub_decxdr, fp->fr_dat, (unsigned)(fp->fr_len),
			XDR_DECODE);
	return 0;
}


static int
dec_xdr_step()
{
	struct umbuf *up = midtobuf(rbufmid);
	struct frag *fp;
	int cc;

	if (cc = dec_step())
		return cc;
	fp = up->ub_cfrag;
	xdrmem_create(&up->ub_decxdr, fp->fr_dat, (unsigned)(fp->fr_len),
			XDR_DECODE);
	return 0;
}


static int
dec_xdr_byte(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	struct frag *fp;
	int cc;

	fp = up->ub_cfrag;
	if (cc = byteupk((char*)vp, cnt, 1, std))
		return cc;
	if (fp != up->ub_cfrag) {
		fp = up->ub_cfrag;
		xdrmem_create(&up->ub_decxdr, fp->fr_dat, (unsigned)(fp->fr_len),
				XDR_DECODE);
	}
	up->ub_cpos = (up->ub_cpos + 3) & ~3;
	xdr_setpos(&up->ub_decxdr, up->ub_cpos);
	return 0;
}


static int
dec_xdr_short(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register short *np;
	int cc = 0;

	for (np = (short*)vp; cnt-- > 0; np += std)
		if (!xdr_short(xob, np)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_short(xob, np)) {
					cc = PvmNoData;
					break;
				}
		}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_int(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register int *np;
	int cc = 0;

	for (np = (int*)vp; cnt-- > 0; np += std)
		if (!xdr_int(xob, np)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_int(xob, np)) {
					cc = PvmNoData;
					break;
				}
		}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_long(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register long *np;
	int cc = 0;

	for (np = (long*)vp; cnt-- > 0; np += std)
		if (!xdr_long(xob, np)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_long(xob, np)) {
					cc = PvmNoData;
					break;
				}
		}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_float(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register float *fp;
	int cc = 0;

	for (fp = (float*)vp; cnt-- > 0; fp += std)
		if (!xdr_float(xob, fp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_float(xob, fp)) {
					cc = PvmNoData;
					break;
				}
		}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_double(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register double *dp;
	int cc = 0;

	for (dp = (double*)vp; cnt-- > 0; dp += std)
		if (!xdr_double(xob, dp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_double(xob, dp)) {
					cc = PvmNoData;
					break;
				}
		}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_cplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register float *xp;
	int cc = 0;

	std = std * 2 - 1;
	for (xp = (float*)vp; cnt-- > 0; xp += std) {
		if (!xdr_float(xob, xp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_float(xob, xp)) {
					cc = PvmNoData;
					break;
				}
		}
		xp++;
		if (!xdr_float(xob, xp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_float(xob, xp)) {
					cc = PvmNoData;
					break;
				}
		}
	}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


static int
dec_xdr_dcplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct umbuf *up = midtobuf(rbufmid);
	XDR *xob = &up->ub_decxdr;
	register double *zp;
	int cc = 0;

	std = std * 2 - 1;
	for (zp = (double*)vp; cnt-- > 0; zp += std) {
		if (!xdr_double(xob, zp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_double(xob, zp)) {
					cc = PvmNoData;
					break;
				}
		}
		zp++;
		if (!xdr_double(xob, zp)) {
			if (cc = dec_xdr_step())
				break;
			else
				if (!xdr_double(xob, zp)) {
					cc = PvmNoData;
					break;
				}
		}
	}
	up->ub_cpos = xdr_getpos(xob);
	return cc;
}


/*************************
 **  In-place encoders  **
 **                     **
 *************************/

static int
enc_inp_init()
{
	return 0;
}


static int
enc_inp_byte(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_BYTE;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_short(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_SHORT;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_int(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_INT;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_long(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_LONG;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_float(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_FLOAT;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_double(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_DFLOAT;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_cplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_CPLX;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


static int
enc_inp_dcplx(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	struct dat_blk d;

	d.type = INP_DCPLX;
	d.base = vp;
	d.cnt = cnt;
	d.std = std;
	return bytepk((char*)&d, 1, sizeof(d), 0);
}


/*******************
 **  Foo encoders **
 **               **
 *******************/

static int
enc_foo_init()
{
	return 0;
}


static int
enc_foo_byte(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	return bytepk((char*)vp, cnt, siz, std * siz);
}


static int
enc_foo_int(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	register int *np;
	char buf[4];
	int n;
	int cc;

	for (np = (int*)vp; cnt-- > 0; np += std) {
		n = *np;
		buf[0] = n >> 24;
		buf[1] = n >> 16;
		buf[2] = n >> 8;
		buf[3] = n;
		if (cc = bytepk(buf, 4, 1, 1))
			return cc;
	}
	return 0;
}


static int
dec_foo_init()
{
	struct umbuf *up = midtobuf(rbufmid);

	up->ub_cfrag = up->ub_frag->fr_link;
	up->ub_cpos = 0;
	return 0;
}


static int
dec_foo_byte(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	return byteupk((char*)vp, cnt, siz, std * siz);
}


static int
dec_foo_int(vp, cnt, std, siz)
	void *vp;
	int cnt, std, siz;
{
	register int *np;
	char buf[4];
	int cc;

	for (np = (int*)vp; cnt-- > 0; np += std) {
		if (cc = byteupk(buf, 4, 1, 1))
			return cc;
		*np = (0x80 & buf[0] ? ~0xffffffff : 0)
		+ ((0xff & buf[0]) << 24)
		+ ((0xff & buf[1]) << 16)
		+ ((0xff & buf[2]) << 8)
		+ (0xff & buf[3]);
	}
	return 0;
}


static int
enc_foo_err()
{
	return PvmNotImpl;	/* don't implement this :-) */
}


/*******************************************
 **  Alien (can't do it) en/decoder stub  **
 **                                       **
 *******************************************/

static int
enc_alien()
{
	return PvmBadMsg;
}


/****************************
 **  Encoder switch table  **
 **                        **
 ****************************/

static struct encvec encoders[] = {

	{	enc_raw_init, dec_raw_init,		/* Raw (native) 0 */
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any,
		enc_raw_any, dec_raw_any },

	{	enc_xdr_init, dec_xdr_init,		/* XDR 1 */
		enc_xdr_byte, dec_xdr_byte,
		enc_xdr_short, dec_xdr_short,
		enc_xdr_int, dec_xdr_int,
		enc_xdr_long, dec_xdr_long,
		enc_xdr_float, dec_xdr_float,
		enc_xdr_double, dec_xdr_double,
		enc_xdr_cplx, dec_xdr_cplx,
		enc_xdr_dcplx, dec_xdr_dcplx },

	{	enc_inp_init, enc_alien,		/* In place (encode only) 2 */
		enc_inp_byte, enc_alien,
		enc_inp_short, enc_alien,
		enc_inp_int, enc_alien,
		enc_inp_long, enc_alien,
		enc_inp_float, enc_alien,
		enc_inp_double, enc_alien,
		enc_inp_cplx, enc_alien,
		enc_inp_dcplx, enc_alien },

	{	enc_foo_init, dec_foo_init,		/* Foo (for pvmd) 3 */
		enc_foo_byte, dec_foo_byte,
		enc_foo_err, enc_foo_err,
		enc_foo_int, dec_foo_int,
		enc_foo_err, enc_foo_err,
		enc_foo_err, enc_foo_err,
		enc_foo_err, enc_foo_err,
		enc_foo_err, enc_foo_err,
		enc_foo_err, enc_foo_err },

	{	enc_alien, enc_alien,			/* Alien (can't deal) 4 */
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien,
		enc_alien, enc_alien },
};


/*	enctovec()
*
*	Map data encoding value to encode/decode function vector
*/

struct encvec *
enctovec(enc)
	int enc;
{
	if (enc == 0)			/* xdr */
		return &encoders[1];
	if (enc == 1)			/* foo */
		return &encoders[3];
	if (enc == myndf)		/* native */
		return &encoders[0];
	return &encoders[4];	/* alien */
}


int
enc_step()
{
	struct frag *fp = midtobuf(sbufmid)->ub_frag;
	struct frag *nfp;

	if (!(nfp = fr_new(ourudpmtu)))
		return PvmNoMem;
	nfp->fr_dat += MAXHDR;
	LISTPUTBEFORE(fp, nfp, fr_link, fr_rlink);
	return 0;
}


int
dec_step()
{
	struct umbuf *up = midtobuf(rbufmid);

	up->ub_cpos = 0;
	if (up->ub_cfrag == up->ub_frag)		/* no message left */
		return PvmNoData;
	up->ub_cfrag = up->ub_cfrag->fr_link;
	if (up->ub_cfrag == up->ub_frag)
		return PvmNoData;
	return 0;
}


/************************
 **  Libpvm Functions  **
 **                    **
 ************************/

int
pvm_mkbuf(enc)
	int enc;		/* data format */
{
	int mid;
	struct umbuf *up;
	struct frag *fp;

	switch (enc) {						/* verify encoding */
	case PvmDataDefault:
		enc = 0;		/* XXX this just forces XDR */
		break;

	case PvmDataRaw:
		enc = myndf;
		break;

	case PvmDataFoo:
		enc = 1;
		break;

	case PvmDataInPlace:
		return uliberr("pvm_mkbuf", PvmNotImpl);
		break;

	default:
		return uliberr("pvm_mkbuf", PvmBadParam);
	}
	if ((mid = umbuf_new()) <= 0)
		return uliberr("pvm_mkbuf", mid);

	up = midtobuf(mid);					/* set encoders */
	up->ub_enc = enc;
	up->ub_codef = enctovec(enc);
	up->ub_src = mypvmtid;

	fp = fr_new(ourudpmtu);				/* make a blank frag to write in */
	fp->fr_dat += MAXHDR;
	LISTPUTBEFORE(up->ub_frag, fp, fr_link, fr_rlink);

	return mid;
}


int
pvm_freebuf(mid)
	int mid;		/* message handle */
{
	int cc;

	if (mid <= 0 || mid >= midhsiz)
		return uliberr("pvm_freebuf", PvmBadParam);
	if (sbufmid == mid)
		sbufmid = 0;
	if (rbufmid == mid)
		rbufmid = 0;
	if ((cc = umbuf_free(mid)) < 0)
		return uliberr("pvm_freebuf", cc);
	return 0;
}


int
pvm_setsbuf(mid)
	int mid;
{
	int omid;
	struct umbuf *up = 0;

	if (mid < 0 || mid >= midhsiz)
		return uliberr("pvm_setsbuf", PvmBadParam);
	if (mid) {
		if (!(up = midh[mid].m_umb))
			return uliberr("pvm_setsbuf", PvmNoSuchBuf);
		if (mid == rbufmid)
			rbufmid = 0;
	}
	omid = sbufmid > 0 ? sbufmid : 0;
	sbufmid = mid;
	if (up && !(up->ub_flag & UB_PACK)) {
		(up->ub_codef->enc_init)();
		up->ub_flag &= ~UB_UPACK;
		up->ub_flag |= UB_PACK;
	}
	return omid;
}


int
pvm_getsbuf()
{
	return sbufmid > 0 ? sbufmid : 0;
}


int
pvm_setrbuf(mid)
	int mid;
{
	int omid;
	struct umbuf *up = 0;

	if (mid < 0 || mid >= midhsiz)
		return uliberr("pvm_setrbuf", PvmBadParam);
	if (mid) {
		if (!(up = midh[mid].m_umb))
			return uliberr("pvm_setrbuf", PvmNoSuchBuf);
		if (mid == sbufmid)
			sbufmid = 0;
	}
	omid = rbufmid > 0 ? rbufmid : 0;
	rbufmid = mid;
	if (up && !(up->ub_flag & UB_UPACK)) {
		(up->ub_codef->dec_init)();
		up->ub_flag &= ~UB_PACK;
		up->ub_flag |= UB_UPACK;
	}
	return omid;
}


int
pvm_getrbuf()
{
	return rbufmid > 0 ? rbufmid : 0;
}


int
pvm_initsend(enc)
	int enc;
{
	int mid;

	if ((mid = pvm_mkbuf(enc)) < 0)
		return uliberr("pvm_initsend", mid);
	if (sbufmid > 0)
		pvm_freebuf(sbufmid);
	pvm_setsbuf(mid);
	return mid;
}


int
pvm_bufinfo(mid, len, code, tid)
	int mid;
	int *len;
	int *code;
	int *tid;
{
	struct umbuf *up;

	if (mid <= 0 || mid >= midhsiz)
		return uliberr("pvm_bufinfo", PvmBadParam);
	if (!(up = midh[mid].m_umb))
		return uliberr("pvm_bufinfo", PvmNoSuchBuf);
	if (len)
		*len = up->ub_len;
	if (code)
		*code = up->ub_cod;
	if (tid)
		*tid = up->ub_src;
	return PvmOk;
}


int
pvm_pkbyte(cp, cnt, std)
	char *cp;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkbyte", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_byte)((void*)cp, cnt, std, 1))
		return uliberr("pvm_pkbyte", cc);
	return PvmOk;
}


int
pvm_upkbyte(cp, cnt, std)
	char *cp;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkbyte", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_byte)((void*)cp, cnt, std, 1))
		return uliberr("pvm_upkbyte", cc);
	return PvmOk;
}


int
pvm_pkcplx(xp, cnt, std)
	float *xp;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkcplx", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_cplx)
			((void*)xp, cnt, std, sizeof(float)*2))
		return uliberr("pvm_pkcplx", cc);
	return PvmOk;
}


int
pvm_upkcplx(xp, cnt, std)
	float *xp;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkcplx", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_cplx)
			((void*)xp, cnt, std, sizeof(float)*2))
		return uliberr("pvm_upkcplx", cc);
	return PvmOk;
}


int
pvm_pkdcplx(zp, cnt, std)
	double *zp;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkdcplx", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_dcplx)
			((void*)zp, cnt, std, sizeof(double)*2))
		return uliberr("pvm_pkdcplx", cc);
	return PvmOk;
}


int
pvm_upkdcplx(zp, cnt, std)
	double *zp;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkdcplx", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_dcplx)
			((void*)zp, cnt, std, sizeof(double)*2))
		return uliberr("pvm_upkdcplx", cc);
	return PvmOk;
}


int
pvm_pkdouble(dp, cnt, std)
	double *dp;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkdouble", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_double)
			((void*)dp, cnt, std, sizeof(double)))
		return uliberr("pvm_pkdouble", cc);
	return PvmOk;
}


int
pvm_upkdouble(dp, cnt, std)
	double *dp;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkdouble", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_double)
			((void*)dp, cnt, std, sizeof(double)))
		return uliberr("pvm_upkdouble", cc);
	return PvmOk;
}


int
pvm_pkfloat(fp, cnt, std)
	float *fp;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkfloat", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_float)
			((void*)fp, cnt, std, sizeof(float)))
		return uliberr("pvm_pkfloat", cc);
	return PvmOk;
}


int
pvm_upkfloat(fp, cnt, std)
	float *fp;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkfloat", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_float)
			((void*)fp, cnt, std, sizeof(float)))
		return uliberr("pvm_upkfloat", cc);
	return PvmOk;
}


int
pvm_pkint(np, cnt, std)
	int *np;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkint", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_int)
			((void*)np, cnt, std, sizeof(int)))
		return uliberr("pvm_pkint", cc);
	return PvmOk;
}


int
pvm_upkint(np, cnt, std)
	int *np;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkint", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_int)
			((void*)np, cnt, std, sizeof(int)))
		return uliberr("pvm_upkint", cc);
	return PvmOk;
}


int
pvm_pklong(np, cnt, std)
	long *np;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pklong", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_long)
			((void*)np, cnt, std, sizeof(long)))
		return uliberr("pvm_pklong", cc);
	return PvmOk;
}


int
pvm_upklong(np, cnt, std)
	long *np;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upklong", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_long)
			((void*)np, cnt, std, sizeof(long)))
		return uliberr("pvm_upklong", cc);
	return PvmOk;
}


int
pvm_pkshort(np, cnt, std)
	short *np;
	int cnt, std;
{
	int cc;

	if (sbufmid <= 0)
		return uliberr("pvm_pkshort", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_short)
			((void*)np, cnt, std, sizeof(short)))
		return uliberr("pvm_pkshort", cc);
	return PvmOk;
}


int
pvm_upkshort(np, cnt, std)
	short *np;
	int cnt, std;
{
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkshort", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_short)
			((void*)np, cnt, std, sizeof(short)))
		return uliberr("pvm_upkshort", cc);
	return PvmOk;
}


int
pvm_pkstr(cp)
	char *cp;
{
	int cc;
	int l = strlen(cp) + 1;

	if (sbufmid <= 0)
		return uliberr("pvm_pkstr", PvmNoBuf);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_int)
			((void*)&l, 1, 1, sizeof(int)))
		return uliberr("pvm_pkstr", cc);
	if (cc = (midtobuf(sbufmid)->ub_codef->enc_byte)
			((void*)cp, l, 1, 1))
		return uliberr("pvm_pkstr", cc);
	return PvmOk;
}


int
pvm_upkstr(cp)
	char *cp;
{
	int l;
	int cc;

	if (rbufmid <= 0)
		return uliberr("pvm_upkstr", PvmNoBuf);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_int)
			((void*)&l, 1, 1, sizeof(int)))
		return uliberr("pvm_upkstr", cc);
	if (cc = (midtobuf(rbufmid)->ub_codef->dec_byte)
			((void*)cp, l, 1, 1))
		return uliberr("pvm_upkstr", cc);
	return PvmOk;
}


