
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	lpvm.c
 *
 *	Libpvm core for unix environment.
 *
$Log$
 */

#include <stdio.h>
#ifdef IMA_BSD386
#include <machine/endian.h>
#endif
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#ifdef IMA_RS6K
#include <sys/select.h>
#endif
#include <sys/stat.h>
#include <fcntl.h>
#ifdef	SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif
#include <errno.h>
#include <signal.h>
#include "pvm3.h"
#include "global.h"
#include "tdpro.h"
#include "ddpro.h"
#include "myalloc.h"
#include "frag.h"
#include "umbuf.h"
#include "listmac.h"

/* task debug mask */

#define	TDMPACKET	1		/* packet tracing */
#define	TDMMESSAGE	2		/* message tracing */


#define	TVCLEAR(tvp)	((tvp)->tv_sec = (tvp)->tv_usec = 0)

char *getenv();

extern struct encvec *enctovec();



/***************
 **  Globals  **
 **           **
 ***************/

extern int errno;					/* from libc */
extern char *sys_errlist[];
extern int sys_nerr;

extern int rbufmid;					/* from pack.c */
extern int sbufmid;					/* from pack.c */

char *pvm_errlist[] = {					/* error messages for -pvm_errno */
		"Error 0",
		"Error 1",
	"Bad parameter",
	"Count mismatch",
		"Error 4",				/* not used */
	"End of buffer",
	"No such host",
	"No such file",
		"Error 8",				/* not used */
		"Error 9",				/* not used */
	"Malloc failed",
		"Error 11",				/* not used */
	"Can't decode message",
		"Error 13",				/* not used */
	"Can't contact local daemon",
	"No current buffer",
	"No such buffer",
	"Null group name",
	"Already in group",
	"No such group",
	"Not in group",
	"No such instance",
	"Host failed",
	"No parent task",
	"Not implemented",
	"Pvmd system error",
	"Version mismatch",
	"Out of resources",
	"Duplicate host",
	"Can't start pvmd",
	"Already in progress",
	"No such task",
	"No such entry",
	"Duplicate entry",
    "Can't get cube (try again later)",
};

int myndf = 0;							/* host native data enc, init XDR */
int mypvmptid = -1;						/* parent task id */
int mypvmtid = -1;						/* this task id */
int myunixpid = -1;						/* process unix pid */
int ourudpmtu = UDPMAXLEN;				/* local UDP MTU */
int pvm_errno = 0;						/* last libpvm error code */
int pvm_nerr = sizeof(pvm_errlist)
			/sizeof(pvm_errlist[0]);	/* exported num of errors */
struct umbuf *rxlist = 0;				/* not-recvd msg list */
int tidhmask = TIDHOST;					/* mask for host field of tids */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";
static int debugmask = 0;				/* which debugging info */
static char etext[512];					/* scratch for error log */
static struct sockaddr_in pvmdsadr;		/* address of pvmd socket */
static int loclsock = -1;				/* pvmd-task tcp socket */
static int pvmdslen = 0;				/* pvmdsadr length */
static struct umbuf *rxfrag = 0;		/* not-assembled incm msgs */
static int useruid = -1;				/* user's unix uid */


/**************************
 **  Internal Functions  **
 **                      **
 **************************/

/*	bailout()
*
*	Called by low-level stuff in f.e. frag.c.  Don't really want to
*	bail in libpvm.
*/

void
bailout(n)
	int n;
{
	n = n;	/* ayn rand was here */
}


/*	log_error()
*
*	Log a libpvm error message.  Prepends a string identifying the task.
*/

log_error(s)
	char *s;
{
	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: %s", myunixpid, s);
	else
		fprintf(stderr, "libpvm [t%x]: %s", mypvmtid, s);
}


/*	log_perror()
*
*	Log a libpvm error message.  Prepends a string identifying the
*	task and appends the system error string for _errno.
*/

log_perror(s)
	char *s;
{
	char *em;

	em = ((errno >= 0 && errno < sys_nerr)
		? sys_errlist[errno] : "Unknown Error");
	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: %s: %s\n", myunixpid, s, em);
	else
		fprintf(stderr, "libpvm [t%x]: %s: %s\n", mypvmtid, s, em);
}


hdump(p, n, pad)
	char *p;	/* bytes */
	int n;		/* length */
	char *pad;
{
	int i;
	pad = pad ? pad : "";
	for (i = 0; n-- > 0; i = ++i & 15) {
		fprintf(stderr, "%s%02x%s",
			i ? "" : pad,
			0xff & *p++,
			n && i != 15 ? " " : "\n");
	}
}


/*	mxfer()
*
*	Move message frags between task and pvmd.
*	Returns when
*		outgoing message (if any) fully sent
*		(if block is true) at least one message fully received
*	Returns >=0 the number of complete messages downloaded, or
*	negative on error.
*/

int
mxfer(mid, dtid, code, block)
	int mid;		/* message */
	int dtid;		/* dest */
	int code;		/* type code */
	int block;		/* get at least one message */
{
	struct umbuf *txup;			/* tx message or null */
	struct frag *txfp = 0;		/* cur tx frag or null */
	struct timeval tout;
	struct timeval *tvp;
	fd_set rfds, wfds;
	int nfds = loclsock + 1;
	int gotem = 0;				/* count complete msgs downloaded */
	struct frag *rxfp = 0;		/* cur rx frag or null */
	struct umbuf *rxup;			/* rx message */
	struct frag *fp;
	char *cp;
	int src;
	int ff;
	int n;
	char *txcp = 0;				/* point to remainder of txfp */
	int txtogo = 0;				/* len of remainder of txfp */

	if (txup = midtobuf(mid)) {
		txfp = txup->ub_frag->fr_link;
		txfp = txfp->fr_buf ? txfp : 0;
	}

	do {

		TVCLEAR(&tout);
		tvp = (txfp || (block && !gotem) || rxfp) ? 0 : &tout;

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		FD_SET(loclsock, &rfds);
		if (txfp) {
			FD_SET(loclsock, &wfds);
		}
		if (select(nfds, &rfds, &wfds, (fd_set*)0, tvp) == -1
		&& errno != EINTR) {
			log_perror("mxfer() select");
			return PvmSysErr;
		}

	/*
	* if socket has data ready, read packets from it
	*/

		if (FD_ISSET(loclsock, &rfds)) {

			if (!rxfp)
				rxfp = fr_new(ourudpmtu);

/*
			fprintf(stderr, "mxfer() read fr_len=%d fr_dat=+%d\n",
					rxfp->fr_len, rxfp->fr_dat - rxfp->fr_buf);
*/
			n = read(loclsock, rxfp->fr_dat + rxfp->fr_len, rxfp->fr_max -
					(rxfp->fr_dat - rxfp->fr_buf) - rxfp->fr_len);
/*
			fprintf(stderr, "mxfer() read=%d\n", n);
*/

			if (n == -1 && errno != EWOULDBLOCK && errno != EINTR) {
				log_perror("mxfer() read pvmd sock");
				return PvmSysErr;
			}
			if (!n) {
				log_error("mxfer() read EOF on pvmd sock\n");
				return PvmSysErr;
			}
			if (n > 0) {
				rxfp->fr_len += n;

				do {
					fp = rxfp;

					if (fp->fr_len < TDFRAGHDR) {
/*
						fprintf(stderr, "mxfer() not a full hdr yet\n");
*/
						break;
					}
					n = TDFRAGHDR + up32(fp->fr_dat + 8);	/* header + body */
					if (fp->fr_len < n) {
/*
						fprintf(stderr, "mxfer() not %d yet\n", n);
*/
						break;
					}

					if (fp->fr_len > n) {
						rxfp = fr_new(ourudpmtu);
						bcopy(fp->fr_dat + n, rxfp->fr_dat, fp->fr_len - n);
						rxfp->fr_len = fp->fr_len - n;
						fp->fr_len = n;
/*
						fprintf(stderr, "mxfer() %d left\n", rxfp->fr_len);
*/
					} else {
						rxfp = 0;
/*
						fprintf(stderr, "mxfer() zero left\n");
*/
					}

					cp = fp->fr_dat;
/*
					dst = up32(cp);		XXX might unpack and check
*/
					src = up32(cp + 4);
					ff = up8(cp + 12);
					fp->fr_len -= TDFRAGHDR;
					fp->fr_dat += TDFRAGHDR;

/*
					fprintf(stderr, "mxfer() frag src t%x len %d ff %d\n",
							src, fp->fr_len, ff);
*/

					if (debugmask & TDMPACKET) {
						sprintf(etext, "mxfer() src t%x len %d\n",
								src, fp->fr_len);
						log_error(etext);
					}

	/*
	* if start of message, make new umbuf, add to frag pile
	*/
					if (ff & FFSOM) {
						cp += TDFRAGHDR;
						fp->fr_len -= TTMSGHDR;
						fp->fr_dat += TTMSGHDR;
						rxup = midtobuf(umbuf_new());
						rxup->ub_cod = up32(cp);
						rxup->ub_enc = up32(cp + 4);
						rxup->ub_src = src;
						LISTPUTBEFORE(rxfrag, rxup, ub_link, ub_rlink);
					}

	/* locate frag's message */

					for (rxup = rxfrag->ub_link; rxup != rxfrag;
							rxup = rxup->ub_link)
						if (rxup->ub_src == src)
							break;

					if (rxup == rxfrag) {	/* uh oh, no message for it */
						log_error("mxfer() frag with no message\n");
						fr_unref(fp);

					} else {
						LISTPUTBEFORE(rxup->ub_frag, fp, fr_link, fr_rlink);
						rxup->ub_len += fp->fr_len;
	/*
	* if end of message, move to rxlist and count it
	*/
						if (ff & FFEOM) {
							LISTDELETE(rxup, ub_link, ub_rlink);
							rxup->ub_codef = enctovec(rxup->ub_enc);
							LISTPUTBEFORE(rxlist, rxup, ub_link, ub_rlink);
							gotem++;
						}
					}
				} while (rxfp);
			}
		}

	/*
	* if socket ready to write, send packets
	*/

		if (FD_ISSET(loclsock, &wfds)) {
			if (!txtogo) {
				txcp = txfp->fr_dat;
				txtogo = txfp->fr_len;
	/*
	* if this is first frag, prepend t-t header
	*/
				ff = 0;
				if (txfp->fr_rlink == txup->ub_frag) {
					txcp -= TTMSGHDR;
					txtogo += TTMSGHDR;
					pk32(txcp, code);
					pk32(txcp + 4, txup->ub_enc);
					ff = FFSOM;
				}
				if (txfp->fr_link == txup->ub_frag)
					ff |= FFEOM;
	/*
	* prepend t-d header
	*/
				txcp -= TDFRAGHDR;
				pk32(txcp, dtid);
				pk32(txcp + 4, mypvmtid);
				pk32(txcp + 8, txtogo);
				pk8(txcp + 12, ff);
				txtogo += TDFRAGHDR;
				if (debugmask & TDMPACKET) {
					sprintf(etext, "mxfer() dst t%x len %d\n", dtid, txfp->fr_len);
					log_error(etext);
				}
			}

			n = write(loclsock, txcp, txtogo);
			if (n == -1 && errno != EWOULDBLOCK && errno != EINTR) {
				log_perror("mxfer() write pvmd sock");
				return PvmSysErr;
			}
			if (n > 0) {
				if ((txtogo -= n) > 0) {
					txcp += n;

				} else {
					txcp = 0;
					txfp = txfp->fr_link;
					if (!txfp->fr_buf)
						txfp = 0;
				}
			}
		}

	} while (txfp || (block && !gotem) || rxfp);

	return gotem;
}


/*	msendrecv()
*
*	Single op to send a system message (usually to our pvmd) and get
*	the reply.
*	Returns message handle or negative if error.
*/

int
msendrecv(other, code)
	int other;				/* dst, src tid */
	int code;				/* message code */
{
	int cc;
	struct umbuf *up;

	if (sbufmid <= 0)
		return PvmNoBuf;

	/* send code to other */
	if (debugmask & TDMMESSAGE) {
		sprintf(etext, "msendrecv() to t%x code %d\n", other, code);
		log_error(etext);
	}
	if ((cc = mxfer(sbufmid, other, code, 1)) < 0)
		return cc;

	/* recv code from other */
	for (up = rxlist->ub_link; 1; up = up->ub_link) {
		if (up == rxlist) {
			up = up->ub_rlink;
			if ((cc = mxfer(0, 0, 0, 1)) < 0)
				return cc;
			up = up->ub_link;
		}

		if (debugmask & TDMMESSAGE) {
			sprintf(etext, "msendrecv() cmp from t%x code %d\n",
					up->ub_src, up->ub_cod);
			log_error(etext);
		}
		if (up->ub_src == other && up->ub_cod == code)
			break;
	}
	LISTDELETE(up, ub_link, ub_rlink);
	if (rbufmid > 0)
		umbuf_free(rbufmid);
	rbufmid = 0;
	if (cc = pvm_setrbuf(up->ub_mid))
		return cc;
	return up->ub_mid;
}


/*	mksocs()
*
*	Make socket for talking to pvmd.  Connect to address advertized
*	in TDSOCKNAME file.
*
*	Returns 0 if okay, else error code.
*/

int
mksocs()
{
	char buf[128];
	int d;
	int n;

	if (loclsock != -1)
		return 0;

	/* get addr of pvmd, make socket to talk */

	sprintf(buf, TDSOCKNAME, useruid);
	if ((d = open(buf, O_RDONLY, 0)) == -1) {
		log_perror(buf);
		return PvmSysErr;
	}
	if ((n = read(d, buf, sizeof(buf))) > 0)
		buf[n] = 0;
	else {
		if (n)
			log_perror("mksocs() read addr file");
		else
			log_error("mksocs() read addr file: short read\n");
		return PvmSysErr;
	}
	(void)close(d);

	bzero((char*)&pvmdsadr, sizeof(pvmdsadr));
	hex_inadport(buf, &pvmdsadr);
	pvmdsadr.sin_family = AF_INET;
	pvmdslen = sizeof(pvmdsadr);

	if ((loclsock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		log_perror("mksocs() socket");
		return PvmSysErr;
	}

	if (connect(loclsock, (struct sockaddr*)&pvmdsadr, pvmdslen) == -1) {
		log_perror("mksocs() connect");
		return PvmSysErr;
	}

#ifndef NOSOCKOPT
	d = 1;
	if (setsockopt(loclsock, IPPROTO_TCP, TCP_NODELAY, (char*)&d, sizeof(int))
	== -1) {
		log_perror("mksocs() setsockopt");
		return PvmSysErr;
	}
#endif
	return 0;
}


/*	unmksocs()
*
*	Close socket to local pvmd.
*/

int
unmksocs()
{
	if (loclsock == -1)
		return 1;
	(void)close(loclsock);
	loclsock = -1;
	return 0;
}


/*	beatask()
*
*	Initialize libpvm, config process as a task.
*	This is called as the first step of each libpvm function so no
*	explicit initialization is required.
*
*	Makes socket to talk to local pvmd, connects to pvmd, trades
*	config information and authentication.
*
*	Returns 0 if okay, else error code.
*/

#ifdef	L_tmpnam
#define	LEN_OF_TMP_NAM	L_tmpnam
#else
#define	LEN_OF_TMP_NAM	64
#endif

int
beatask()
{
	int sbf = 0, rbf = 0;			/* saved rx and tx message handles */
	int prver;						/* protocol version */
	static int altpid = -1;			/* pid assigned by pvmd if diff from real */
	int cc;
	char authfn[LEN_OF_TMP_NAM];	/* auth file name */
	int authfd = -1;				/* auth fd to validate pvmd ident */

	if (mypvmtid != -1)
		return 0;

	authfn[0] = 0;

	myunixpid = getpid();

	/*
	* get expected pid from environment in case we were started by
	* the pvmd and someone forked again
	*/

	if (altpid == -1) {
		char *p;

		if (p = getenv("PVMEPID"))
			altpid = atoi(p);
		else
			altpid = 0;
	}

	if ((useruid = getuid()) == -1) {
		log_error("can't getuid()\n");
		return PvmSysErr;
	}

	if (cc = mksocs())		/* make socket to talk to pvmd */
		return cc;

	/*
	*	initialize received-message list and fragment reassembly list
	*/

	rxfrag = TALLOC(1, struct umbuf, "umb");
	bzero((char*)rxfrag, sizeof(struct umbuf));
	rxfrag->ub_link = rxfrag->ub_rlink = rxfrag;

	rxlist = TALLOC(1, struct umbuf, "umb");
	bzero((char*)rxlist, sizeof(struct umbuf));
	rxlist->ub_link = rxlist->ub_rlink = rxlist;

	/*
	*	create empty t-auth file that pvmd must write and we'll check later
	*/

	(void)tmpnam(authfn);
	if ((authfd = open(authfn, O_RDONLY|O_CREAT|O_TRUNC, 0600)) == -1) {
		log_perror("beatask() creat t-auth file");
		return PvmSysErr;
	}

	sbf = pvm_setsbuf(pvm_mkbuf(PvmDataFoo));
	rbf = pvm_setrbuf(0);

	/*
	*	send first connect message to pvmd
	*/

	prver = TDPROTOCOL;
	pvm_pkint(&prver, 1, 0);
	pvm_pkstr(authfn);
	if ((cc = msendrecv(TIDPVMD, TM_CONNECT)) <= 0)
		goto bail;
	pvm_upkint(&prver, 1, 0);
	if (prver != TDPROTOCOL) {
		sprintf(etext, "beatask() t-d protocol mismatch (%d/%d)\n",
			TDPROTOCOL, prver);
		log_error(etext);
		cc = PvmSysErr;
		goto bail;
	}
	pvm_upkint(&cc, 1, 0);
	if (!cc) {
		log_error("beatask() pvmd refuses connection\n");
		goto bail;
	}

	/*
	*	check our t-auth file; write in pvmd d-auth file
	*/

	if ((cc = read(authfd, (char*)&cc, 1)) == -1) {
		log_perror("beatask() read authfile");
		cc = PvmSysErr;
		goto bail;
	}
	if (cc != 1) {
		log_error("beatask() pvmd didn't validate itself\n");
		cc = PvmSysErr;
		goto bail;
	}
	(void)close(authfd);
	(void)unlink(authfn);

	pvm_upkstr(authfn);
	if ((authfd = open(authfn, O_WRONLY, 0)) == -1) {
		log_perror("beatask() open d-auth file");
		authfn[0] = 0;
		cc = PvmSysErr;
		goto bail;
	}
	(void)write(authfd, authfn, 1);
	(void)close(authfd);
	authfd = -1;
	authfn[0] = 0;

	/*
	*	send second connect message to pvmd
	*/

	pvm_initsend(PvmDataFoo);
	pvm_pkint(&myunixpid, 1, 0);
	pvm_pkint(&altpid, 1, 0);
	if ((cc = msendrecv(TIDPVMD, TM_CONN2)) <= 0)
		goto bail;
	pvm_upkint(&cc, 1, 0);
	if (!cc) {
		log_error("beatask() pvmd refuses connection\n");
		goto bail;
	}
	pvm_upkint(&mypvmtid, 1, 0);
	pvm_upkint(&mypvmptid, 1, 0);
	pvm_upkint(&ourudpmtu, 1, 0);
	pvm_upkint(&myndf, 1, 0);

	pvm_freebuf(pvm_setrbuf(rbf));
	pvm_freebuf(pvm_setsbuf(sbf));

	return 0;

bail:
	if (pvm_getrbuf() > 0)
		pvm_freebuf(pvm_getrbuf());
	if (pvm_getsbuf() > 0)
		pvm_freebuf(pvm_getsbuf());
	pvm_setrbuf(rbf);
	pvm_setsbuf(sbf);

	if (authfd != -1)
		(void)close(authfd);
	if (authfn[0])
		(void)unlink(authfn);
	unmksocs();

	return cc;
}


int
endtask()
{
	if (mypvmtid != -1) {
		unmksocs();
		mypvmtid = -1;
	}

	/* XXX free rxfrag and rxlist */

	return 0;
}


/************************
 **  Libpvm Functions  **
 **                    **
 ************************/

int
pvm_perror(s)
	char *s;
{
	if (mypvmtid == -1)
		fprintf(stderr, "libpvm [pid%d]: ", myunixpid);
	else
		fprintf(stderr, "libpvm [t%x]: ", mypvmtid);
	fprintf(stderr, "%s: %s\n",
		(s ? s : "(null)"),
		(pvm_errno <= 0 && pvm_errno > -pvm_nerr
				? pvm_errlist[-pvm_errno] : "Unknown Error"));
	return 0;
}


int
pvm_setdebug(mask)
	int mask;
{
	int old = debugmask;

	debugmask = mask;
	return old;
}


int
pvm_getfd()
{
	int cc;

	if (cc = beatask())
		return uliberr("pvm_getfd", cc);
	return loclsock;
}


int
pvm_start_pvmd(argc, argv)
	int argc;
	char **argv;
{
	char buf[128];
	struct stat sb;
	int cc;
	char *fn;			/* file to exec */
	char *hd;			/* homedir */

	if ((useruid = getuid()) == -1) {
		log_error("can't getuid()\n");
		return uliberr("pvm_start_pvmd", PvmSysErr);
	}

	(void)sprintf(buf, TDSOCKNAME, useruid);

	if (stat(buf, &sb) != -1)
		return uliberr("pvm_start_pvmd", PvmDupHost);

	if (!(hd = getenv("HOME"))) {
		log_error("can't getenv(HOME)\n");
		return uliberr("pvm_start_pvmd", PvmSysErr);
	}

	if (!(fn = TALLOC(strlen(hd) + 20, char, "XXX"))) {
		log_error("can't malloc\n");
		return uliberr("pvm_start_pvmd", PvmSysErr);
	}
	strcpy(fn, hd);
	strcat(fn, "/pvm3/lib/pvmd");

	argv[0] = fn;
	argv[argc] = 0;
	if (!fork()) {
		if (!fork()) {
			for (cc = getdtablesize(); cc-- > 0; )
				(void)close(cc);
			(void)open("/dev/null", O_RDONLY, 0);
			(void)open("/dev/null", O_WRONLY, 0);
			(void)dup2(1, 2);
			execvp(argv[0], argv);
		}
		_exit(0);
	}
	(void)wait(0);

	PVM_FREE(fn);

	for (cc = 8; cc > 0 && stat(buf, &sb) == -1; cc--)
		sleep(1);

	if (cc > 0)
		cc = pvm_mytid();
	else
		cc = PvmCantStart;
	return (cc < 0 ? uliberr("pvm_start_pvmd", cc) : 0);
}


#ifdef IMA_I860
int
sendtopvmd(buf, len)
    char    *buf;
    int     len;
{
    int     n, togo = len;

    while (togo) {
        n = write(loclsock, buf, togo);
        if (n == -1 && errno != EWOULDBLOCK && errno != EINTR) {
            log_perror("sendtopvmd() write pvmd sock");
            return PvmSysErr;
        }
        togo -= n;
    }
    return 0;
}
#endif
