/*
        Routines for manipulating line graphs
*/
#include "tools.h"
#include "xtools/xlg/xlg.h"

/*@
     XBLineGraphCreate - creates a XBLineGraph data structure

  Input Parameters:
.   win - the window where the graph will be made.
.   dim - the number of line cures which will be drawn

@*/
XBLineGraph *XBLineGraphCreate(win, dim)
int      dim;
XBWindow *win;
{
  int         i;
  XBLineGraph *lg = NEW(XBLineGraph); CHKPTRN(lg);

  lg->nopts = 0;
  lg->win   = win;
  lg->winw  = win->w;
  lg->winh  = win->h;
  lg->dim   = dim;
  lg->xmin  = 1.e20;
  lg->ymin  = 1.e20;
  lg->xmax  = -1.e20;
  lg->ymax  = -1.e20;
  lg->axis  = XBAInitAxis(win,XBFontFixed(win, 7, 13)); CHKERRN(1);
  lg->lines = (Lines **) MALLOC(dim*sizeof(Lines *)); 
  CHKPTRN(lg->lines);
  for ( i=0; i<dim; i++ ) {lg->lines[i] = XBLinesInit(); CHKERRN(1); }
  return lg;
}
/*@
    XBLineGraphReset - Clears line graph to allow for reuse with new data.

@*/
void XBLineGraphReset(lg)
XBLineGraph *lg;
{
  XBWindow *win = lg->win; 
  int      i;
  for ( i=0; i<lg->dim; i++ ) {XBLinesFree(lg->lines[i]);}
  for ( i=0; i<lg->dim; i++ ) {lg->lines[i] = XBLinesInit(); CHKERR(1); }
  lg->nopts = 0;
  lg->xmin  = 1.e20;
  lg->ymin  = 1.e20;
  lg->xmax  = -1.e20;
  lg->ymax  = -1.e20;
  XBClearWindow(win,win->x,win->y,win->w,win->h);  
}
/*@
    XBLineGraphDestroy - frees all space taken up by LineGraph 
                         data structure.
@*/
void XBLineGraphDestroy(lg)
XBLineGraph *lg;
{
  int i;
  for ( i=0; i<lg->dim; i++ ) {XBLinesFree(lg->lines[i]);}
  FREE(lg->lines);
  XBADestroyAxis(lg->axis);
  FREE(lg);
}
/*@
    XBLineGraphAddPoint - Adds another point to each of the 
                          line graphs. The new point must have a
                          X coordinate larger than the old points.

   Input Parameters:
.   lg - the LineGraph data structure
.   x,y - the points to two vectors containing the new x and y 
           point for each curve.

@*/
void XBLineGraphAddPoint(lg,x,y)
XBLineGraph *lg;
double      *x, *y;
{
  int i;
  for (i=0; i<lg->dim; i++) {
    if (x[i] > lg->xmax) lg->xmax = x[i]; 
    if (x[i] < lg->xmin) lg->xmin = x[i];
    if (y[i] > lg->ymax) lg->ymax = y[i]; 
    if (y[i] < lg->ymin) lg->ymin = y[i];
    XBLinesAddLines(lg->lines[i],x+i,y+i,1,lg->win->cmapping[i+1]);
    CHKERR(1);
  }
  lg->nopts++;
}
/*@
   XBLineGraphDraw - Redraws a line graph
@*/
void XBLineGraphDraw(lg)
XBLineGraph *lg;
{
  double   xmin=lg->xmin, xmax=lg->xmax, ymin=lg->ymin, ymax=lg->ymax;
  int      i, dim = lg->dim;
  XBWindow *win = lg->win;

  if (xmin >= xmax || ymin >= ymax) return;
  xmax = xmax + .1*(xmax-xmin);
  XBClearWindow(win, 0, 0, lg->winw, lg->winh);
  XSetForeground( win->disp, win->gc.set, win->foreground);
  XBASetLimits(lg->axis, xmin, xmax, ymin, ymax);
  XBADrawAxis(lg->axis);
  for ( i=0; i<dim; i++ ) {
    XBLinesSetScale(lg->lines[i],xmin,xmax,ymin,ymax);
    XBLinesRescale(win, lg->lines[i]);
    XBLinesDraw(win,lg->lines[i]);
  }
  XBFlush(lg->win);
} 
 

