/*

    MESSAGE.C

    This file contains routines that are used to produce a Virtual Front
    Panel.

*/

#include <X11/Xlib.h>
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "xtools/basex11.h"
#include "xtools/vfp/windesign.h"
#include "xtools/vfp/message.h"     /*I "xtools/vfp/message.h" I*/


#define extra  1.2  /*extra space desired between processors (1.2 -> 20% white space) */

#define SIZE 10  /*****
		       this is for the size of the message color map.  
                       probably (definitely?) should not be a defined constant
                 *****/

/*@  
    VFPInit - Initilizes the screen for the message display

    Input Parameters:
.   XBWin          - window
.   font           - font
.   num_proc       - number of processors.
.   rect_w, rect_h - width and height of rectanges for the processors 
.   x,y            - upper left coordinates for region

@*/
VFPInfo* VFPInit( XBWin, font, num_proc, rect_w, rect_h, x, y )
     XBWindow     *XBWin ;
     XBFont       *font ;
     int          num_proc,
                  rect_w, 
                  rect_h,
                  x, y ;
{
  VFPInfo  *info ;

  /**** this procedure should handle the sizing and mapping 
        of the window (???)****/

  info = NEW(VFPInfo); CHKPTRN(info);
  info->border_size = 10 ;
  info->num_processors = num_proc ;
  info->proc_w      = rect_w ;
  info->proc_h      = rect_h ;
  info->x           = x ;
  info->y           = y ;
  info->proc_coords = NULL ;  /* for now */
  info->XBWin       = XBWin;
  info->nstates     = 0;
  return (VFPInfo*)info ;
}


/*@
    VFPFree - Take care of any needed clean up of the message display

    Input Parameter:
.   info - structure containing the screen layout information

@*/ 
void VFPFree( info ) 
VFPInfo  *info ;
{
if (info->proc_coords)
    FREE( info->proc_coords );
FREE( info );
}


void VFPDrawConnection();

/*@
    VFPDrawRoute - Draw a routed message connection 
   
    Input Parameters:
.   info    - screen layout info
.   state   - state to shade in connection with
.   route   - routing array
.   size    - indicates size of message - used of shading of message 
.   flag    - 1 to draw, 0 to undraw
    notes:
      The number of nodes in the route is indicated by route[0].
      If no routing of the message is desired, then the function 
      VFPDrawConnection should be used.
@*/
void VFPDrawRoute( info, route, size, flag )
     VFPInfo      *info ;
     int          *route,
                   size,
                   flag;
{
  int i ;

  for( i=1 ; i<route[0] ; i++ ) 
    {
      VFPDrawConnection( info, route[i], route[i+1], size, flag ) ;
    }
}



/*@
    VFPDrawConnection - Draw a message connection between two given nodes
   
    Input Parameters:
.   info    - screen layout info
.   from    - processor #'s to draw connection from
.   to      - processor #'s to draw connection to
.   size    - size of the message
.   flag    - 1 to draw, 0 to undraw
@*/
void VFPDrawConnection( info, from, to, size, flag )
     VFPInfo      *info ;
     int           from,
                   to,
                   size,
                   flag;

{
  int  from_x, from_y, to_x, to_y, dx, dy, pdx, pdy ;
  XBWindow     *XBWin=info->XBWin ;
  
  /* check if given processor is in range */
  if( (from<0) || (from>info->num_processors) || 
      (to<0)   || (to>info->num_processors) ) {
      fprintf( stderr, 
	       " could not draw connection - proc. # is out of range \n" ) ;
      return ;
      }

  if( !flag )
    /* set to xor */
    XSetFunction( XBWin->disp, XBWin->gc.set, GXxor ) ;
  
  if (XBWin->numcolors == 2) 
    {
      XSetFillStyle( XBWin->disp, XBWin->gc.set, FillStippled );

/***** 
  needs to use some value to indicate the stipple that should be
  used.  May just set one stipple for b&w messages and just use the
  line thickness to indicate message size.

*****/
      XSetStipple( XBWin->disp, XBWin->gc.set, info->dpm[size] );
    }
  else 
    {

/***** 
  instead of getting the pixel value fron the xmapping, it should come
  from the info->msg_cmap field.  I had problems getting that to work
  so I just did it this way.  This should be changed.                 
*****/
      /* map size in range */
      size = size >> 8;
      if (size > 10) size = 10;

      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->cmapping[size+15] ) ;
#ifdef THICKEN_LINES
      XSetLineAttributes( XBWin->disp, XBWin->gc.set, 3 - (int)(size/(SIZE/3)), 
			 LineSolid, CapButt, JoinMiter ) ;
#endif
    }

  from_x = info->proc_coords[from].x_center + info->x ;
  from_y = info->proc_coords[from].y_center + info->y ;
  to_x =   info->proc_coords[to].x_center + info->x ;
  to_y =   info->proc_coords[to].y_center + info->y ;
  
  /* Here we should try to clip the segment.  Rather than do a perfect job,
     we simply increment the locations by the x/y half widths in both
     cases */
  pdx = info->proc_w / 2 + 3 ;
  pdy = info->proc_h / 2 + 3 ;
  dx = to_x - from_x;
  if (dx < 0)
    {
      from_x -= pdx;  to_x   += pdx ;
    }
  else if (dx > 0) 
    {
      from_x += pdx;  to_x   -= pdx ;
    }
  dy = to_y - from_y;
  if (dy < 0)
    {
      from_y -= pdy;  to_y   += pdy ;
    }
  else if (dy > 0) 
    {
      from_y += pdy;  to_y   -= pdy ;
    }
  
  XDrawLine( XBWin->disp, XBDrawable(XBWin), XBWin->gc.set, 
	    from_x, from_y, to_x, to_y ); 
#ifdef THICKENLINE
  XSetLineAttributes( XBWin->disp, XBWin->gc.set, 1, LineSolid, CapButt, JoinMiter ) ;
#endif

/*****
  users should probably have some control over whether or not an arrow is
  draw on each connection and over the size of the arrow head.
*****/

  VFPDrawArrow( info, from_x, from_y, to_x, to_y, 10 ) ;

  if (XBWin->numcolors == 2) 
    XSetFillStyle( XBWin->disp, XBWin->gc.set, FillSolid );
  else 
    XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;
  if( !flag )
    XSetFunction( XBWin->disp, XBWin->gc.set, GXcopy ) ;
  XFlush( XBWin->disp ) ;

}



/*
   VFPDrawProcRects - Draw the processor rectangles that make up the display

   Input Parameters:
.  info - screen info

   notes:
     This function uses the values given in the VFPInfo field proc_coords.
     The proc-coords field contains info on the position of each processor.
     These values can be set using on of the given functions, or set though
     a user supplied function .
*/
VFPDrawProcRects( info ) 
     VFPInfo      *info ;
{
  int i, x, y ;
  XBWindow     *XBWin=info->XBWin ;
  XBDecoration Rgn;

  for( i=0; i < info->num_processors; i++ )
    {
      x = info->proc_coords[i].x_upleft + info->x ;
      y = info->proc_coords[i].y_upleft + info->y ;
      Rgn.Box.x    = x - 4;
      Rgn.Box.y    = y - 4;
      Rgn.Box.w    = info->proc_w + 8;
      Rgn.Box.h    = info->proc_h + 8;
      Rgn.Box.xh   = Rgn.Box.x + Rgn.Box.w - 1;
      Rgn.Box.yh   = Rgn.Box.y + Rgn.Box.h - 1;
      Rgn.width    = 4;
      Rgn.HasColor = 0 ; /*XBWin->numcolors > 2;*/
      Rgn.is_in    = 0;
      Rgn.Hi       = BlackPixel( XBWin->disp, XBWin->screen ) ;
      Rgn.Lo       = WhitePixel( XBWin->disp, XBWin->screen ) ;;
      XBDrawFrame( XBWin, &Rgn );
    }
    XFlush( XBWin->disp ) ;
}


/*@
    VFPSetState - Set the state for a given processor
   
    Input Parameters:
.   info   - screen layout info
.   proc   - number of the processor
.   state  - state to fill in proc rectangle with
@*/
void VFPSetState( info, proc, state ) 
     VFPInfo   *info ;
     int        proc ;
     int        state;
{
  int  x, y,
       width,
       height ;
  XBWindow     *XBWin=info->XBWin ;

  x = info->proc_coords[proc].x_upleft + info->x ;
  y = info->proc_coords[proc].y_upleft + info->y ;
  width = info->proc_w - 1;
  height = info->proc_h - 1 ;

  /* fill proc with new state color */
  if (XBWin->numcolors == 2) {
      /* erase current state color */
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->background ) ;
      XFillRectangle( XBWin->disp, XBDrawable(XBWin), XBWin->gc.set, 
		      x, y, width, height ) ;
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;
      XSetFillStyle( XBWin->disp, XBWin->gc.set, FillStippled );
      XSetStipple( XBWin->disp, XBWin->gc.set, info->dpm[state] );
      }
  else {
      XSetForeground( XBWin->disp, XBWin->gc.set, info->dp[state] ) ;
      }
  XFillRectangle( XBWin->disp, XBDrawable(XBWin), XBWin->gc.set, 
		  x, y, width, height ) ;
  XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;
  if (XBWin->numcolors == 2) {
      XSetFillStyle( XBWin->disp, XBWin->gc.set, FillSolid );
      }
  XFlush( XBWin->disp ) ;
}


/*@
    VFPFillProcCoordGrid2d - Produce processor coordinates for a grid
        configuration using a given size.
   
    Input Parameters:
.   info   - screen layout info
.   tot_w, tot_h - total width and height of the region
.   nx, ny - the number of processors desired horizontally (nx) and
             vertically (ny).

    notes:
      This function produces a set of processor coordinates in a grid
      that is of the size indicated by the values nx, and ny.
@*/
void VFPFillProcCoordGrid2d( info, tot_w, tot_h, nx, ny )
VFPInfo  *info ;
int      tot_w,
         tot_h, nx, ny ;
{
VFPProcCoord  *coords ;
int i ;
int num_x, num_y, x_coord, y_coord, x_spacing, y_spacing, x, y ;
void VFPFillProcCoordGrid();

if (nx > 0 && ny > 0) {
    /* Use specified mesh sizes */
    coords = (VFPProcCoord*)
	MALLOC( info->num_processors * sizeof( VFPProcCoord) ) ;
    CHKPTR(coords);
    x_spacing = (tot_w - (nx * info-> proc_w)) / (nx + 1) ;
    y_spacing = (tot_h - (ny * info-> proc_h)) / (ny + 1) ;
    y_coord = y_spacing ;
    i = 0;
    for( y=0 ; y<ny ; y++ ) {
	x_coord = x_spacing ;
	for( x=0 ; (x<nx) && (i<info->num_processors) ; x++ ) {
	    coords[i].x_upleft   = x_coord ;
	    coords[i].y_upleft   = y_coord ;
	    coords[i].x_center   = x_coord + (int)(0.5*info->proc_w) ;
	    coords[i++].y_center = y_coord + (int)(0.5*info->proc_h) ;
	    x_coord += info->proc_w + x_spacing ;
	    }
	y_coord += info->proc_h + y_spacing ;
	}
    info->proc_coords = coords ;
    }
else 
    VFPFillProcCoordGrid( info, tot_w, tot_h );
}



/*****
  This procedure needs help.  Don't really know how much it is needed????
  Works great for some values but other times it will generate coordinates
  that are far from a best fit.  
*****/

/*@
    VFPFillProcCoordGrid - produce processor coordinates for a grid 
         configuration.
   
    Input Parameters:
.   info   - screen layout info
.   tot_w, tot_h - total width and height of the region
	     
    notes:
      This function produces a set of processor coordinates in a grid.
      The actual number of processors located horizontally and vertically
      it determined by a series of attempts to get a best fit for the
      number of processors and the size of the given region.
      Also note that if the given processors will not fit the given region
      the value of the processor rectanges given in the initilization may
      be changed.
@*/
void VFPFillProcCoordGrid( info, tot_w, tot_h ) 
     VFPInfo  *info ;
     int      tot_w,
              tot_h ;
{
  VFPProcCoord  *coords ;
  int i ;
 

  int num_x, num_y, x_coord, y_coord, x_spacing, y_spacing, x, y ;
  coords = (VFPProcCoord*)MALLOC( info->num_processors * sizeof( VFPProcCoord) ) ;
  
  /* determine if enough area to fit processor rectangles in given region */
  while( !((info->proc_w * info->proc_h * info->num_processors * extra * extra) < 
	   (tot_w * tot_h)) )
    /* reduce the size of the processor rectangles to fit given region */
    (info->proc_w > info->proc_h) ? info->proc_w-- : info->proc_h-- ;
  
  /* first try to make a relatively square rectangle of processes */
  num_x = (int)sqrt( (double)info->num_processors ) ;
  num_y = (int)(info->num_processors / num_x) + 
               (info->num_processors%num_x ? 1: 0 );
  while( (num_x >= (int)(info->num_processors*0.3)) && 
	 (info->num_processors % num_x) )
    num_x -- ;
  if( (num_x > 0) && (info->num_processors % num_x) )
    {
      num_y = (int)(info->num_processors / num_x) + (info->num_processors%num_x ? 1: 0 );
      if( (num_x*info->proc_w*extra <= tot_w) || (num_y*info->proc_h*extra <= tot_h) )
	goto got_fit ;  /*  <--- GOT A FIT!! */
 
       /* swap values */
       num_x += num_y ; num_y = num_x - num_y ; num_x -= num_y ;
       if( (num_x*info->proc_w*extra <= tot_w) || (num_y*info->proc_h* extra < tot_h) )
	 goto got_fit ;  /*  <--- GOT A FIT!! */
    }

  /* couldn't make a filled in rectangle of processors, now just try to make a decent one */
  num_x = (int)(0.3*info->num_processors) ;
  while( num_x< info->num_processors )
    {
      num_y = (int)(info->num_processors / num_x) + (info->num_processors%num_x ? 1: 0 );
      if( (num_x * info->proc_w * extra <= tot_w) || (num_y * info->proc_h * extra <= tot_h) )
	goto got_fit ;  /*  <--- GOT A FIT!! */
      num_x ++ ;
    }

  /* in a final case to get a grid, we start with a relativly square grid and reduce the
     size of the processor rectanges till we MAKE them fit                               */

  num_x = (int)sqrt( (double)info->num_processors ) ;
  num_y = (int)(info->num_processors / num_x) + (info->num_processors%num_x ? 1: 0 );
  while( ! (info->proc_w * num_x * extra < tot_w ) )
    info->proc_w-- ;
  while( ! (info->proc_h * num_y * extra < tot_h ) )
    info->proc_h-- ;
  /* NOW the squares should fit */

got_fit:
  /* fill the array with coornate values */
  /*printf( " final : proc_w = %d, proc_h = %d\n", info->proc_w, info->proc_h );*/
  i = 0 ;
  x_spacing = (tot_w - (num_x * info-> proc_w)) / (num_x + 1) ;
  y_spacing = (tot_h - (num_y * info-> proc_h)) / (num_y + 1) ;
  y_coord = y_spacing ;
  for( y=0 ; y<num_y ; y++ ) 
    {
      x_coord = x_spacing ;
      for( x=0 ; (x<num_x) && (i<info->num_processors) ; x++ ) 
	{
	  coords[i].x_upleft   = x_coord ;
	  coords[i].y_upleft   = y_coord ;
	  coords[i].x_center   = x_coord + (int)(0.5*info->proc_w) ;
	  coords[i++].y_center = y_coord + (int)(0.5*info->proc_h) ;
	  x_coord += info->proc_w + x_spacing ;
	}
      y_coord += info->proc_h + y_spacing ;
    }
  info->proc_coords = coords ;
 }


/*@
    VFPFillProcCoordTree - produce processor coordintes for a binary tree
         configuration.

    Input Parameters:
.   info   - screen layout info
.   tot_w, tot_h - total width and height of the region

@*/	     
void VFPFillProcCoordTree( info, tot_w, tot_h ) 
     VFPInfo  *info ;
     int      tot_w,
              tot_h ;
{
  VFPProcCoord  *coords ;
  int i ;
  int x_coord, y_coord, y_spacing, curr_level, indent,
      power, margin, height, temp3 ;
  double mult, x_spacing ;

  /* determine the height of the tree */
  for( height=1, power=1 ; info->num_processors >= power ; height++, power*=2 ) ;
  if( height > 1)
    height -- ;

  coords = (VFPProcCoord*)MALLOC( info->num_processors * sizeof( VFPProcCoord) ) ;
  
  /* determine max size of proc rectangles */
  while( !(  (info->proc_w * (int)pow( 2.0, (double)height-1) * extra)  < tot_w ) )
    info->proc_w-- ;
  while( ! (info->proc_h * height * extra < tot_h ) )
    info->proc_h-- ;

  indent = 0 ;
  y_spacing = (int)( (tot_h - (height * info->proc_h)) / (height + 1) ) ;
  for( curr_level=height ; curr_level>0 ; curr_level-- ) 
    {
      power = (int)pow( (double)2, (double)curr_level-1 ) ;
      if( power > 1)
	if( curr_level == height )
	  {
	    x_spacing = (tot_w - (power * info->proc_w)) / (power + 1.0) ; 
	    temp3 = (info->proc_w + (int)x_spacing)  ;
	    mult = 0.5 ;
	    margin = (int)x_spacing ;
	  }
	else
	  x_spacing = (tot_w - (power * info->proc_w) - 2.0*(indent+margin) ) / (power - 1.0) ; 
      else
	x_spacing = 0.0 ;
      y_coord = (curr_level - 1) * (info->proc_h + y_spacing) + y_spacing ;
      for( i=power ; (i<2*power) && (i<=info->num_processors) ; i++ )
	{
	  x_coord = ((i - power) * info->proc_w) + ((i - power) * x_spacing) 
	    + indent + margin ;
	  coords[i-1].x_upleft = x_coord ;
	  coords[i-1].y_upleft = y_coord ;
	  coords[i-1].x_center = x_coord + (int)(0.5*info->proc_w) ;
	  coords[i-1].y_center = y_coord + (int)(0.5*info->proc_h) ;
	}
      indent = (int)( indent +  temp3 * mult )   ;
      mult *= 2.0 ;
    }
  info->proc_coords = coords ;
 }




/*@
     VFPDefineStateColor - Define the color for a given state.
   
    Input Parameters:
.   info   - screen layout info
.   idx    - state number
.   name   - name of desired color

@*/
void VFPDefineStateColor( info, idx, name )
VFPInfo *info;
int     idx;
char    *name;
{
PixVal pixval;

XBFindColor( info->XBWin, name, &pixval );
info->dp[idx] = pixval;
}


/*@
     VFPDefineStateBW - Define the stipple for a given state.
   
    Input Parameters:
.   info   - screen layout info
.   idx    - state number
.   pixmap - the pixmap for the desired stipple

@*/
void VFPDefineStateBW( info, idx, pixmap )
VFPInfo *info;
int     idx;
Pixmap  pixmap;
{
info->dpm[idx] = pixmap;
}



/*@
    VFPMakeRoute - Produce an array of proc numbers to route a message through
   
    Input Parameters:
.   info   - screen layout info
.   from   - number of the processor the message originated from
.   to     - number of the processor that receives the message

    notes:
    Current logic for this procedure will ONLY produce a routing for a 2D grid.
    The routing produced follows the pattern; move horizontally as much as
    needed, then move vertically.

@*/
int* VFPMakeRoute( info, from, to ) 
     VFPInfo  *info ;
     int       from,
               to ;
{
  XBWindow     *XBWin=info->XBWin ;
  int          *route,
                index = 1,
                curr_proc ;
  static int    num_per_row = 0 ;

  /* allocate "array" that is size of number of processors */

/**** do we want to allocate each array?  if we don't need to hold onto a
   route then just one allocated array would do.   ****/
  route = (int*)MALLOC( (info->num_processors+1) * sizeof(int) ) ;
  CHKPTR(route);
  
  route[index] = from ;
  curr_proc = from ;

  /* compute number of processors per row - login ONLY valid for a grid */
  if( !num_per_row ) 
    {
      while( (info->proc_coords[0].y_center == info->proc_coords[num_per_row].y_center) 
	    && (num_per_row < info->num_processors) )
	num_per_row++ ;
    }
  /* move to the right if necessary */
  while( info->proc_coords[curr_proc].x_center < info->proc_coords[to].x_center )
    {
      /* must check if is procesor to the right - may not be in last row */
      if( curr_proc >= info->num_processors - 1 )
	{
	  route[index++] = curr_proc ;
	  curr_proc -= num_per_row ;
	}
      route[index++] = curr_proc ;
      curr_proc ++ ;
    }
  /* move to the left if necessary */
  while( info->proc_coords[curr_proc].x_center > info->proc_coords[to].x_center )
    {
      route[index++] = curr_proc ;
      curr_proc -- ;
    }
  /* move downward if necessary */
  while( info->proc_coords[curr_proc].y_center < info->proc_coords[to].y_center )
    {
      route[index++] = curr_proc ;
      curr_proc += num_per_row ;
    }
  /* move upward if necessary */
  while( info->proc_coords[curr_proc].y_center > info->proc_coords[to].y_center )
    {
      route[index++] = curr_proc ;
      curr_proc -= num_per_row ;
    }

  if( curr_proc != to )
    fprintf( stderr, " **** ERROR - in routing algorithm  \n" ) ;

  route[index] = to ;
  route[0] = index ;



  return( route ) ;

}



/*@
    VFPDrawArrow - Draw an arrow on the given line segment.

 Parameters:
.   info   - screen layout info
.   xstart, ystart, xend, yend - the x and y coordinates for the starting
        and ending points of the line.
.   size   - a value indicating the size of the message

    notes:
      The value of size is simply a relative value to indicate the amount
      of data in a messge.  It is a value that is between 0 and the 
      number of different message colors allocated.

@*/
void VFPDrawArrow( info, xstart, ystart, xend, yend, size ) 
     VFPInfo  *info ;
     int       xstart,
               ystart,
               xend,
               yend,
               size ;
{
  XBWindow *XBWin=info->XBWin ;
  int       adj_xend, adj_yend;
  double    slope, perp_slope, dx, dy, x1, x2, x3, y1, y2, y3 ;
  XPoint    points[4] ;


  /* Check for horizontal and vertical lines.  This is not necessary but since
     they are easier to draw this should be quicker */
  if( xstart == xend ) 
    {
      /* is vertical */
      points[1].x = xend + .5*size ;
      points[1].y = yend + ( (yend < ystart) ? (size) : (0-size) ) ;
      points[2].x = xend - .5*size ;
      points[2].y = yend + ( (yend < ystart) ? (size) : (0-size) ) ;
    }
  else if( ystart == yend )
    {
      /* is horizontal */
      points[1].x = xend + ( (xend < xstart) ? (size) : (0-size) ) ;
      points[1].y = yend - .5*size ;
      points[2].x = xend + ( (xend < xstart) ? (size) : (0-size) ) ;
      points[2].y = yend + .5*size ;
    }
  else 
    {    
      /* not horizontal or vertical */
      /* adjust values as if at origin */
      adj_xend = xend - xstart ;
      adj_yend = (-1) * ( yend - ystart ) ;
      perp_slope = (-1.0) * (double)(adj_xend / adj_yend ) ;
      x1 = (double)(adj_xend - (size * adj_xend) / 
		    sqrt( (double)( adj_xend * adj_xend + adj_yend * adj_yend) )) ;
      y1 = (double)(adj_yend - (size * adj_yend) /
		    sqrt( (double)( adj_xend * adj_xend + adj_yend * adj_yend) )) ;
      dx = sqrt( (double)(.25*(size * size)) / ((perp_slope * perp_slope) +  1) ) ;
      dy = perp_slope * dx ;
      x2 = x1 + dx ;  y2 = y1 + dy ;
      x3 = x1 - dx ;  y3 = y1 - dy ;
      points[1].x = (int)x2 + xstart ;
      points[1].y = (-1) * (int)y2 + ystart;
      points[2].x = (int)x3 + xstart ;
      points[2].y = (-1) * (int)y3 + ystart;
    }      
  points[3].x = points[0].x = xend  ;
  points[3].y = points[0].y = yend  ; 
  XDrawLines( XBWin->disp, XBDrawable(XBWin), XBWin->gc.set, points, 4, CoordModeOrigin ) ;
  XFillPolygon( XBWin->disp, XBDrawable(XBWin), XBWin->gc.set, points, 4, Convex, 
	       CoordModeOrigin ) ;
  XBFlush( XBWin );
}






















