#ifndef lint
static char SCCSid[] = "@(#) ./sparse/subset.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"
#include "inline/copy.h"

#define DO_SKIPS
/*@
  SpSetSubmatrix - Assign a part of a matrix the value contained in 
                   another matrix, Mat(rmap,cmap) = submat.

  Input Parameters:
. mat    - matrix to change
. submat - matrix of values to use
. rmap,cmap - set mat(rmap,cmap) to the values in submat.
              rmap has the same number of elements as there are rows in submat,
              cmap has the same number of elements as there are columns in 
              submat

  Notes:
  The implementation sets a row at a time.  
  Q: do we assume that cmap is sorted?
 @*/
void SpSetSubmatrix(mat,submat,rmap,cmap)
SpMat *mat, *submat;
int   *rmap, *cmap;
{
int      i,j,nz,*ix,Mrow,N,M;
double   *xv;

SPLITTOMAT(mat);
SPLITTOMAT(submat);
N  = mat->rows;
M  = mat->cols;
for ( i=0; i<submat->rows; i++ ) {
    SpScatterFromRow( submat, i, &nz, &ix, &xv );
    Mrow = rmap[i];
    if ( Mrow < N && Mrow > -1 ) {
	for ( j=0; j<nz; j++ ) {
	    if ( cmap[ix[j]] < M && cmap[ix[j]] > -1) {
		 SpSetValue(mat,xv[j],Mrow,cmap[ix[j]]); CHKERR(1);
		}
	    }
	}
    }
}

/*@
  SpSubset - Generate a subset of a given matrix and return that (sub)matrix.

  Input Parameters:
.  mat          - matrix to subset
.  nrows, ncols - size of subset matrix
.  rowrange, colrange - rows and columns to use from mat

  Returns:
  subset matrix

  Notes:
  The resulting matrix has rows and columns in consecutive locations.

  This routine needs to be able to allocate a vector with mat->cols 
  elements.  This is not a good general-purpose choice.
 @*/
SpMat *SpSubset( mat, nrows, ncols, rowrange, colrange )
SpMat *mat;
int   nrows, ncols, *rowrange, *colrange;
{
SpMat  *new;
double *vexpand, *v, *xv, *v1;
int    i, j, nc, nz, *cr, *xi, *i1, mcols;

SPLITTOMAT(mat);

mcols = mat->cols;
if (mcols < ncols) {
    SETERRC(1,"Invalid submatrix size in SpSubset" );
    return 0;
    }
new = SpCreate( nrows, ncols, 0 ); CHKERRV(1,0);
vexpand = (double *)MALLOC( mcols * sizeof(double) ); CHKPTRV(vexpand,0);
v1      = (double *)MALLOC( ncols * sizeof(double) ); CHKPTRV(v1,0);
i1      = (int *) MALLOC( ncols * sizeof(int) );      CHKPTRV(i1,0);

for (i=0; i<nrows; i++) {
    SpScatterFromRow( mat, rowrange[i], &nz, &xi, &xv );
    nc     = ncols;
    cr     = colrange;
    v      = vexpand;
    /* Expand the matrix row */
    SCATTERVAL(v,cr,nc,0.0);
    /* Assume column indices are sorted; check that we are in range */
    if (nz > 0 && xi[nz-1] >= mcols) {
	SETERRC(1,"Columns out-of-range in SpSubset" );
	return 0;
	}
    SCATTER(xv,xi,v,nz);
    /* Collect the (non-zero) entries in colrange */
    nc     = ncols;
    nz     = 0;
    cr     = colrange;
    v      = vexpand;
    j      = 0;
    while (nc--) {
	if (v[*cr] != 0.0) {
	    v1[nz]   = v[*cr];
	    i1[nz++] = j;
	    }
	cr++;
	j++;
	}

    /* Allocate and copy this to the new matrix */
    SpSetInRow( new, i, nz, v1, i1 ); CHKERRV(1,0);
    }

FREE(vexpand);
FREE(v1);
FREE(i1);
return new;
}

#ifndef OLD_SUBSET
/*@
  SpSubsetSorted - Generate a subset of a given matrix and return that 
                   (sub)matrix.

  Input Parameters:
.  mat          - matrix to subset
.  nrows,ncols - size of subset matrix
.  rowrange,colrange - rows and columns to use from mat.  columns is SORTED
   in increasing order.

  Returns:
  subset matrix

  Notes:
  The resulting matrix has rows and columns in consecutive locations.

  This routine needs a workspace proportional to the colrange[ncols-1]-
  colrange[0].
 @*/
SpMat *SpSubsetSorted( mat, nrows, ncols, rowrange, colrange )
SpMat *mat;
int   nrows, ncols, *rowrange, *colrange;
{
SpMat           *new;
double          *xv;
int             ii, i, nc, nz, *cr, *xi, *i1, nnc;
register double *v1, *Xv;
register int    Nz, j, clast, xival, cfirst, nnz;
register int    *cwork, *Xi;

TRPUSH(SPTRID+5);
SPLITTOMAT(mat);

cr    = colrange;
nc    = ncols;
clast = cr[nc-1];
cfirst= cr[0];
new   = SpCreate( nrows, ncols, 0 );                CHKPTRN(new);
v1    = (double *)MALLOC( ncols * sizeof(double) ); CHKPTRN(v1);
i1    = (int *)   MALLOC( ncols * sizeof(int) );    CHKPTRN(i1);
nnc   = clast - cfirst + 1;
cwork = (int *) MALLOC( nnc * sizeof(int) );        CHKPTRN(cwork);
for (ii=0; ii<nnc; ii++) cwork[ii]            = -1;
for (ii=0; ii<nc; ii++)  cwork[cr[ii]-cfirst] = ii;
for (ii=0; ii<nrows; ii++) {
    SpScatterFromRow( mat, rowrange[ii], &nz, &xi, &xv );
    nnz    = 0;
    i      = 0;
    Xi     = xi;
    Xv     = xv;
    Nz     = nz;
    /* skip xv values until we reach the first cr value.  This assumes
       that the matrix is relatively sparse */
    while (i < Nz && Xi[i] < cfirst) i++;
    while (i < Nz ) {
	xival = Xi[i];
	if (clast < xival) break;
	j = cwork[xival - cfirst];
	if (j >= 0) {
	    v1[nnz]   = Xv[i];
	    i1[nnz++] = j;  
	    }
	i++;
	}

    /* Allocate and copy this to the new matrix */
    SpSetInRow( new, ii, nnz, v1, i1 ); CHKERRN(1);
    }

FREE(v1);
FREE(i1);
FREE(cwork);
TRPOP;
return new;
}
#else
SpMat *SpSubsetSorted( mat, nrows, ncols, rowrange, colrange )
SpMat *mat;
int   nrows, ncols, *rowrange, *colrange;
{
SpMat  *new;
double *vexpand, *v, *xv, *v1;
int    ii, i, j, nc, nz, nnz, *cr, *xi, *i1;
int    clast, xival, cfirst;
int    b, t;                 /* used for search */

TRPUSH(SPTRID+5);
SPLITTOMAT(mat);

new = SpCreate( nrows, ncols, 0 ); CHKERRV(1,0);
v1  = (double *)MALLOC( ncols * sizeof(double) ); CHKPTRV(v1,0);
i1  = (int *)   MALLOC( ncols * sizeof(int) );    CHKPTRV(i1,0);

cr    = colrange;
nc    = ncols;
clast = cr[nc-1];
cfirst= cr[0];
for (ii=0; ii<nrows; ii++) {
    SpScatterFromRow( mat, rowrange[ii], &nz, &xi, &xv );
    /* Expand the matrix row */
    nnz    = 0;
    j      = 0;
    b      = nc/2;
    /* skip xv values until we reach the first cr value.  This assumes
       that the matrix is relatively sparse */
    i = 0;
    while (i < nz && xi[i] < cfirst) i++;
    while (i < nz ) {
	xival = xi[i];
	if (clast < xival) break;
	/* A bisection search is done here to find the next column in cr.
	   We should check for nz*log(nc) << nc.  We may also want to 
	   special case cr[j] == xival; more appropriately, perhaps we should
	   try the last value of b before resetting it to nc. */
#ifdef DO_SKIPS
	/* Speed this up a little by comparing cr[j] to xival
	   (good for adjacent values, such as are common in 
	   blocked matrices) */
	if (cr[j] == xival) {
	    v1[nnz]   = xv[i];
	    i1[nnz++] = j++;        /* done with this j */
	    i++;
	    continue;
	    }
	/* b     = nc; */
	/* I'm not sure that this test is worthwhile */
	if (b < nc && cr[b] < xival) {
	    if (j < b) j = b;
	    b = nc;
	    }
	while (b - j > 4) {
	    t     = (b+j) / 2;
	    if (cr[t] > xival) b = t;
	    else               j = t;
	    }
#endif    
	while (j < nc && cr[j] < xival) j++;
	if (j == nc) break;
	if (cr[j] == xival) {
	    v1[nnz]   = xv[i];
	    i1[nnz++] = j++;        /* done with this j */
	    }
	/* else cr[j] > xival, so skip that xival */
	i++;
	}

    /* Allocate and copy this to the new matrix */
    SpSetInRow( new, ii, nnz, v1, i1 ); CHKERRV(1,0);
    }

FREE(v1);
FREE(i1);
TRPOP;
return new;
}
#endif

#ifdef FOO
/* @
  SpSubsetInplace - Generate a subset of a given matrix and return 
                    that (sub)matrix. Does no copy the matrix data.

  Input Parameters:
.  mat          - matrix to subset
.  nrows, ncols - size of subset matrix
.  rowrange, colrange - rows and columns to use from mat

  Returns:
.  subset matrix

  Note:  Does NOT allocate space for the entries.  Rather, the entries in mat
  are rearanged to allow the subset matrix to reference the storage in the
  original matrix.
 @ */
SpMat *SpSubsetInplace( mat, nrows, ncols, rowrange, colrange )
SpMat *mat;
int   nrows, ncols, *rowrange, *colrange;
{
SpMat  *new;
SpVec  **rs, *oldrow, **newrows;
double *vexpand, *v, *xv, *v1;
int    i, nc, nz, *cr, *xi, *i1;

SPLITTOMAT(mat);
rs  = GETROWMAT(mat)->rs;

new = SpCreate( nrows, ncols, 0 ); CHKERRV(1,0);
newrows = GETROWMAT(new)->rs;

vexpand = (double *)MALLOC( mat->cols * sizeof(double) ); CHKPTRV(vexpand,0);
v1      = (double *)MALLOC( ncols * sizeof(double) ); CHKPTRV(v1,0);
i1      = (int *) MALLOC( ncols * sizeof(int) ); CHKPTRV(i1,0);

for (i=0; i<nrows; i++) {
    oldrow = rs[rowrange[i]];
    nc     = ncols;
    cr     = colrange;
    v      = vexpand;
    /* Expand the matrix row */
    SCATTERVAL(v,cr,nc,0.0);
    xi     = oldrow->i;
    xv     = oldrow->v;
    nz     = oldrow->nz;
    SCATTER(xv,xi,v,nz);
    /* Collect the (non-zero) entries in colrange */
    nc     = ncols;
    nz     = 0;
    cr     = colrange;
    v      = vexpand;
    while (nc--) {
	if (v[*cr] != 0.0) {
	    v1[nz]   = v[*cr];
	    i1[nz++] = *cr;
	    v[*cr] = 0.0;
	    }
	cr++;
	}
    newrows[i]->nz = nz;
    newrows[i]->v  = oldrow->v;
    newrows[i]->i  = oldrow->i;

    /* append the unused entries */
    nc = oldrow->nz;
    xi = oldrow->i;
    while (nc--) {
	if (v[*xi] != 0.0) {
	    v1[nz] = v[*xi];
	    i1[nz++] = *xi;
	    }
	xi++;
	}
    /* Copy these back over the storage */
    nz = oldrow->nz;
    xi = oldrow->i;
    cr = i1;
    COPY(xi,cr,nz);
    nz = oldrow->nz;
    xv = oldrow->v;
    v  = v1;
    COPY(xv,v,nz);
    }
FREE(vexpand);
FREE(v1);
FREE(i1);
return new;
}
#else
/*@
  SpSubMatrixInPlace - Generate a submatrix of a given matrix and return 
                       that (sub)matrix. Does not copy the matrix data.

  Input Parameters:
.  mat          - matrix to subset
.  nrows,ncols - size of subset matrix
.  rowrange,colrange - rows and columns to use from mat

  Returns:
  subset matrix

  Note:  
  Does NOT allocate space for the entries.  Rather, the entries in mat
  are rearanged to allow the subset matrix to reference the storage in the
  original matrix.
 @*/
SpMat *SpSubMatrixInPlace( mat, rows, m, cols, n )
SpMat *mat;
int   *cols,*rows,m,n;
{
SpVec     *row,**rs,**newrs;
double    *xv,tmpv;
SpRowMat  *R = (SpRowMat *)mat->data;
int       i,j,*ix,tail,M = mat->rows,N = mat->cols,tmpi;
int       nz,*smap,head,*newmap;
SpMat     *newmat;

if (rows || m != 0) {SETERRC(1,"SpSubMatrixInPlace not supported with rows");
                     return 0;} 
/* NERROR(1,"cannot yet SpSubMatrixInPlace with rows "); */

/* smap is logical array of columns to be removed */
smap = (int *) MALLOC(N*sizeof(int)); CHKPTRV(smap,0);
MEMSET(smap,0,N*sizeof(int));
for ( i=0; i<n; i++ ) smap[cols[i]] = 1;

/* create new matrix, it gets same map as old matrix */
newmat = SpCreate(M-m,N-n,0); CHKERRV(1,0);
rs    = R->rs;
newrs = ((SpRowMat *)newmat->data)->rs;
/* reorder the columns for each row */
for ( j=0; j<M; j++ ) {
    row  = rs[j];
    nz   = row->nz;
    ix   = row->i;
    xv   = row->v;
    tail = nz - 1;
    for ( i=0; i<tail+1; i++ ) {
        while (smap[ix[tail]] && (tail > i)) tail--;
        if (tail == i && smap[ix[i]]) {tail--; break;}
        else if (tail == i) break;
        else if (smap[ix[i]]) {
	    tmpv     = xv[tail];  tmpi   = ix[tail];
	    xv[tail] = xv[i]; ix[tail--] = ix[i];
	    xv[i]    = tmpv;  ix[i]      = tmpi;
	    }
	}
    /* sort the first set of items */
    SpSort(ix,xv,tail+1); CHKERRV(1,0); 
    newrs[j]->nz = tail+1;
    newrs[j]->i = ix;
    newrs[j]->v = xv;
    }

/* the column indices of the first tail+1 elements in each row */
/* should be between 0 and N-n; we therefore reorder them so that */
/* this will be true. smap will contain the new ordering */
tail = 0; head = N-n;
for ( i=0; i<N; i++ ) {
    if (smap[i]) smap[i] = head++;
    else         smap[i] = tail++;
    }
/* fix the column indices */
for ( j=0; j<M; j++ ) {
    row  = rs[j];
    nz   = row->nz;
    ix   = row->i;
    xv   = row->v;
    for ( i=0; i<nz; i++ ) {
        ix[i] = smap[ix[i]];
	}
    }
/* fix the column map */
newmap = (int *) MALLOC(N*sizeof(int)); CHKPTRV(newmap,0);
if (mat->map->colmap) {
    for ( i=0; i<N; i++ ) {
	newmap[smap[i]] = mat->map->colmap[i];
	}
    }
else
    for (i=0; i<N; i++) 
	newmap[smap[i]] = i;
if (!mat->map->q_user_col) FREE(mat->map->colmap);
mat->map->colmap     = newmap;
mat->map->q_user_col = 0;
if (mat->map) 
    SpCopyMappings( newmat, 
 		    mat->map->rowmap, mat->map->colmap, mat->map->icolmap );
FREE(smap);
newmat->nz = -1;   /* For now, just use don't know */
return newmat;
}
#endif

/*@
  SpRowSubMatrixInPlace - Generate a submatrix consisting of all the columns 
                          and the specified rows of a given matrix and return 
                          that (sub)matrix. Does not copy the matrix data.

  Input Parameters:
.  mat      - matrix to subset
.  nrows    - size of subset matrix
.  rowrange - rows and columns to use from mat

  Returns:
  subset matrix

  Note:  
  Does NOT allocate space for the entries.  
 @*/
SpMat *SpRowSubMatrixInPlace( mat, rowrange, nrows )
SpMat *mat;
int   *rowrange,nrows;
{
SpVec     **rs,**newrs;
SpRowMat  *R = (SpRowMat *)mat->data;
int       j,N = mat->cols;
SpMat     *newmat;

/* create new matrix, it gets same map as old matrix */
newmat = SpCreate(nrows,N,0); CHKERRV(1,0);
rs     = R->rs;
newrs  = ((SpRowMat *)newmat->data)->rs;
/* Extract the rows */
for (j=0; j<nrows; j++) {
    newrs[j] = rs[rowrange[j]];
    }

newmat->nz = -1;   /* For now, just use don't know */
return newmat;
}
