#ifndef lint
static char SCCSid[] = "@(#) ./sparse/spsup.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "set/iset.h"
#include "inline/copy.h"

/*
    This file contains routines to find the "support" of a matrix.
    This is defined as all of the columns that are adjacent to
    a given set of rows; this corresponds to the support of the
    approximation basis if the matrix comes from a Finite Element
    discretization.

    Also note that the support is a property of the non-zero structure,
    and is the same for matrices with the same structure even if the
    values of the elements are different.

 */

/*
    This is because the iset code copies the sets each time.  We could
    improve this in several ways

        Use a bit-oriented union code for the intermediate steps

	Use a log-tree combination (but requiring significantly more storage
	for the intermediate isets) (log base could be larger than 2 to
	reduce storage requirements).  This doesn't actually reduce the
	NUMBER of unions, but does reduce the cost of each one since the
	bottom level unions (half the unions) all have few elements 
	(just the nonzeros on the two rows).

     There is also a linear algorithm that is

        set a support vector to have all -1's
	
	View the support vector as a list.  Each element indexes the next
	element.  When inserting an element, we

	    look to see if the element is present

	    if so, skip

	    else, insert.

	We can take advantage of the sorted column values to speed this
	up.

     For large matrices, this can be done in blocks, by recalling where 
     the last column was encountered in each row, and just looping through
     them.  
 */
/*@
    SpSupport - find the support for a sub-matrix

    Input Parameters:
.    mat     - matrix
.    nrows   - number of rows to use
.    rows    - rows to use
.    maxcols - maximum number of columns to return

    Output Parameters:
.    ncols  - number of columns in support (not counting rows)
.    cols   - columns in support

    Note:
    cols does include rows; that is, cols is the support.

    This version is very simple; it just forms the union of the column
    indices for the given rows.

    As written, this algorithm can take time proportional to the nrows*ncols.
    See SpSupport2 for an alternative routine.
@*/
void SpSupport( mat, nrows, rows, maxcols, ncols, cols )
SpMat *mat;
int   nrows, *rows, maxcols, *ncols, *cols;
{
ISet     *is, *icol, *icol2, *swap;
int      i, nz, *xi;
int      wascached = 0;
void SpiSupportNoWork();

SPLITTOMAT(mat);

#define USE_LOG_SUPPORT_NOALLOC
#ifdef USE_LOG_SUPPORT
i = nrows / 2;
if (i == 0) {
    SpScatterFromRow( mat, rows[0], ncols, &xi, (double **)0 );
    nz = *ncols;
    ICOPY(cols,xi,nz);
    return;
    }
else {
    /* Special case for support is the entire set of rows */
    /* if (*ncols == maxcols) return; */

    wascached = ISEnableCache( maxcols );
    /* Recursively determine support */
    icol  = ISAlloc( maxcols );                          CHKPTR(icol);
    icol2 = ISAlloc( maxcols );                          CHKPTR(icol2);
    is    = ISAlloc( maxcols );                          CHKPTR(is);
    SpSupport( mat, i,       rows,   maxcols, &icol->n,  icol->idx );
    SpSupport( mat, nrows-i, rows+i, maxcols, &icol2->n, icol2->idx );
    ISUnion( icol, icol2, is );
    *ncols = ISSize( is );
    if (*ncols < maxcols) maxcols = *ncols;
    xi = ISIdx( is );
    ICOPY(cols,xi,maxcols);
    
    if (!wascached) ISDisableCache();
    ISDestroy( is );
    ISDestroy( icol );
    ISDestroy( icol2 );
    }
#elif defined(USE_LOG_SUPPORT_NOALLOC)
icol  = ISAlloc( maxcols );                          CHKPTR(icol);
icol2 = ISAlloc( maxcols );                          CHKPTR(icol2);
is    = ISAlloc( maxcols );                          CHKPTR(is);

wascached = ISEnableCache( maxcols );
SpiSupportNoWork( mat, nrows, rows, is, icol, icol2 );
if (!wascached) ISDisableCache();

*ncols = ISSize( is );
if (*ncols < maxcols) maxcols = *ncols;
xi = ISIdx( is );
ICOPY(cols,xi,maxcols);

ISDestroy( is );
ISDestroy( icol );
ISDestroy( icol2 );

#else
/* Perhaps we should use maxcols here instead ??? */
is    = ISCreateFromData( 0, (int *)0 );               CHKPTR(is);
icol  = ISAlloc( mat->cols );                          CHKPTR(icol);
icol2 = ISAlloc( mat->cols );                          CHKPTR(icol2);
for (i=0; i<nrows; i++) {
    /* This is rather heavy: three routine calls for each row. */
    SpScatterFromRow( mat, rows[i], &nz, &xi, (double **)0 );
    ISSetFromData( is, nz, xi );
    ISUnion( is, icol, icol2 );
    swap = icol2; icol2 = icol; icol  = swap;
    }
*ncols = ISSize( icol );
if (*ncols < maxcols) maxcols = *ncols;
xi = ISIdx( icol );
ICOPY(cols,xi,maxcols);

ISSetFromData( is, 0, (int *)0 );
ISDestroy( is );
ISDestroy( icol );
ISDestroy( icol2 );
#endif
}

/* This is a helper routine for the recursive version of SpSupport
   that does as little as possible allocation of storage.
   The support is returned in rs. */
void SpiSupportNoWork( mat, nrows, rows, rs, c1, c2 )
SpMat *mat;
int   nrows, *rows;
ISet  *rs, *c1, *c2;
{
int          i, nnz, *xii;
ISet         *c3;
register int *p, *xi, nz;

SPLITTOMAT(mat);

i = nrows / 2;
if (i == 0) {
    SpScatterFromRow( mat, rows[0], &nnz, &xii, (double **)0 );
    if (nnz > rs->na) nnz = rs->na;
    rs->n = nnz;
    p     = rs->idx;
    nz    = nnz;
    xi    = xii;
    ICOPY(p,xi,nz);
    return;
    }
else if (i == 1) {
    /* Handle pairs of rows as well */
    SpScatterFromRow( mat, rows[0], &nnz, &xii, (double **)0 );
    if (nnz > c1->na) nnz = c1->na;
    c1->n = nnz;
    p     = c1->idx;
    nz    = nnz;
    xi    = xii;
    ICOPY(p,xi,nz);
    SpScatterFromRow( mat, rows[1], &nnz, &xii, (double **)0 );
    if (nnz > c2->na) nnz = c2->na;
    c2->n = nnz;
    p     = c2->idx;
    nz    = nnz;
    xi    = xii;
    ICOPY(p,xi,nz);

    ISUnion( c1, c2, rs );
    return;
    }
else {
    /* Recursively determine support: 
       Return support in c1 of the first i rows */
    SpiSupportNoWork( mat, i,       rows,   c1, rs, c2 );

    if (nrows - i > 0) 
	c3 = ISAlloc( c1->na );                          CHKPTR(c3);
    /* Return support in c2 for the last nrows-i rows */
    SpiSupportNoWork( mat, nrows-i, rows+i, c2, rs, c3 );
    if (nrows - i > 0) 
	ISDestroy( c3 );
    ISUnion( c1, c2, rs );
    }
}

/*
    I probably need both SpSupport and SpSupportMR (minus rows)

    Also, if I had a light-weight submatrix that just had its
    own rowmap, then this routine would just take mat, not
    mat, nrows, rows.  This would be particularly handy for nrows=mat->n,
    rows = [0:nrows-1].
 */    

/*@
    SpSupport2 - find the support for a sub-matrix

    Input Parameters:
.    mat     - matrix
.    nrows   - number of rows to use
.    rows    - rows to use
.    maxcols - maximum number of columns to return

    Output Parameters:
.    ncols  - number of columns in support (not counting rows)
.    cols   - columns in support

    Note:
    cols does include rows; that is, cols is the support.

    This version is very simple.

    As written, this algorithm can take time proportional to 2ncols + nz.

    For large matrices, this can be done in blocks, by recalling where 
    the last column was encountered in each row, and just looping through
    them.  An optimization counts the number of entries made.
@*/
void SpSupport2( mat, nrows, rows, maxcols, ncols, cols )
SpMat *mat;
int   nrows, *rows, maxcols, *ncols, *cols;
{
int          *xi;
register int i, j, nz, *is, Ncols = mat->cols;
SPLITTOMAT(mat);

is = (int *)MALLOC( Ncols * sizeof(int) );   CHKPTR( is );
for (i=0; i<Ncols; i++) is[i] = 0;
for (i=0; i<nrows; i++) {
    SpScatterFromRow( mat, rows[i], ncols, &xi, (double **)0 );
    nz = *ncols;
    for (j=0; j<nz; j++) 
	is[xi[j]] = 1;
    }
j = 0;
for (i=0; i < Ncols; i++) {
    if (is[i] != 0) {
	cols[j++] = i;
	if (j >= maxcols) break;
	}
    }
*ncols = j;
FREE( is );
}
