#ifndef lint
static char SCCSid[] = "@(#) ./sparse/row/solve.c 07/23/93";
#endif

/*
    This file contains a simple sparse solve routine.
    These are intended for problems where the matrix is re-ordered
    to improve performance (e.g., numerical stability, reduction of fill)
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

void SpiSolvePerm(), SpiSolveBase(), SpiSolvePermBase();

/*
  SpRSolve - Solve a system of equations, the matrix must have been
            factored, by, for instance, SpComputeFactor().

  Input Parameters:
.  BB - Split matrix already processed by Factor
.  b  - Right-hand-side
.  x  - solution
 */
void SpRSolve( BB, b, x )
SpMatSplit *BB;
double     *b, *x;
{
int err;
if (BB->factor->map) 
    SpiSolvePerm( BB, x, b );
else
    SpiSolveBase( BB, x, b );
}

/*
        [(R A C) Cinv] x = R b
 */
void SpiSolvePerm( BB, x, b )
SpMatSplit *BB;
double      *x, *b;
{
double *tmp;
if (BB->factor->rows == 0) return;
tmp    = (double *)SPAllocTemp(BB->factor->rows * sizeof(double)); CHKPTR(tmp);
SpiSolvePermBase( BB, x, b, tmp );
SPFreeTemp( tmp );
}

void SpiSolvePermBase( BB, x, b, Tmp )
SpMatSplit *BB;
double      *x, *b, *Tmp;
{
int          i, n, *nzs;
double   *vv;
register double       *v, sum, *tmp = Tmp;
register int          *vi, nz;
SpMat    *B = BB->factor;
SpVec    *row, **rs;
int      *r = B->map->rowmap, *c = B->map->colmap;
SpRowMat *R = (SpRowMat *)B->data;

n      = B->rows;
/* forward solve the lower triangular */
nzs    = BB->nzl + 1;
rs     = R->rs + 1;
tmp[0] = b[*r++];
for (i=1; i<n; i++) {
    row  = *rs++;
    v    = row->v;
    vi   = row->i;
    nz   = *nzs++;
    sum  = b[*r++];
    SPARSEDENSEMDOT(sum,tmp,v,vi,nz);
    tmp[i] = sum;
    }

/* backward solve the upper triangular */
nzs--; rs--;
c = c + (n-1);
for (i=n-1; i>=0; i--) {
    nz   = *nzs-- + 1;
    row  = *rs--;
    vv = v = row->v + nz;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    sum  = tmp[i];
    SPARSEDENSEMDOT(sum,tmp,v,vi,nz);
    x[*c--] = tmp[i] = sum * vv[-1];
    }
}

/*
   This is a solve that does not involve any mappings 
 */
void SpiSolveBase( BB, x, b )
SpMatSplit  *BB;
double      *x, *b;
{
int          i, n, *nzs;
double   *vv;
register double       *v, sum;
register int          *vi, nz;
SpMat    *B = BB->factor;
SpVec    *row, **rs;
SpRowMat *R = (SpRowMat *)B->data;

n      = B->rows;
/* forward solve the lower triangular */
nzs    = BB->nzl + 1;
rs     = R->rs + 1;
x[0]   = *b++;
for (i=1; i<n; i++) {
    row  = *rs++;
    v    = row->v;
    vi   = row->i;
    nz   = *nzs++;
    sum  = *b++;
    SPARSEDENSEMDOT(sum,x,v,vi,nz);
    x[i] = sum;
    }

/* backward solve the upper triangular */
nzs--; rs--;
for (i=n-1; i>=0; i--) {
    nz   = *nzs-- + 1;
    row  = *rs--;
    vv = v = row->v + nz;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    sum  = x[i];
    SPARSEDENSEMDOT(sum,x,v,vi,nz);
    x[i] = sum * vv[-1];
    }
}


