#ifndef lint
static char SCCSid[] = "@(#) ./sparse/row/factor.c 07/23/93";
#endif

/*
    This file contains a simple sparse factor and solve routine.
    This is appropriate if no re-orderings are applied to the matrix.
    For (row,column) permutations, the routines in factorperm.c should
    be used.
    Diagonal elements are stored as inverse (LDU storage).
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

/*
   SpRComputeFill - Find storage for factor of matrix B in BB 
                      (descriptors allocated but not set).

   Note: 
   Because fill propagates down, a single row effectively adds 
   all of the fill for the rows above.  This is crucial in reducing the
   cost of the fill algorithm.  The rule is

$   Let IM = # of non-zeros to include.
$   1. If, at row "prow", we add a row (row) that contains column prow, then
      set IM[row] = nz;
$   2. IM[prow] = nz of row.

 */
int SpRComputeFill( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          *fill, n, *im, err;

if (BB->factor->map)
    err = SpiComputeFillPerm( B, BB );
else {
    n    = B->rows;
    im   = (int *) SPAllocTemp( 2*(n+1)*sizeof(int) ); CHKPTRV(im,1);
    fill = im + n + 1;
    if (!im) return -(n+1);
    err = SpComputeFillBase( B, BB, fill, im );
    SPFreeTemp(im);
    }
return err;
}

int SpComputeFillBase( B, BB, fill, im )
SpMat       *B;
SpMatSplit *BB;
register int       *fill, *im;
{
int          prow, row, *xi, nz, nzi, n, fm, nnz, idx, m, nzf, 
             *nzl, err = 0;
SpMat *BBf = BB->factor;
SpVec *pivot_row, *elim_row, **rs, **frs;
SpRowMat     *R = (SpRowMat *)B->data, *RR = (SpRowMat *)BB->factor->data;

/* Nothing to do for the first row */
n          = B->rows;
BB->nzl[0] = 0;
rs         = R->rs;
frs        = RR->rs;
nzl        = BB->nzl;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  */
    elim_row  = rs[prow];
    xi        = elim_row->i;
    row       = *xi++;
    /* Add the initial row (fill,rtmp) */
    nz        = elim_row->nz;
    nzf       = nz--;
    fm        = row;
    while (nz--) {
	idx       = *xi++;
	fill[fm]  = idx;
	fm        = idx;
	}
    fill[fm] = n;

    nzi = 0;
    /* Eliminate into (fill,rtmp) */
    while (row < prow) {
	pivot_row  = frs[row];
	nz         = nzl[row];
	/* Get the pivot row */
	xi         = pivot_row->i + nz;
	if (*xi++ != row) 
	    {SETERRC(-(row+1),"Zero pivot encountered");
             return -(row+1);}     /* zero (missing) pivot */
	nz++;
	nnz        = im[row] - nz;
	fm         = row;
	while (nnz-- > 0) {
	    idx       = *xi++;
	    nz++;
	    if (idx == prow) im[row] = nz;
	    /* find fm such that fill[m] <= idx fill[fill[fm]] */
	    do {
		m  = fm;
		fm = fill[m];
		} while (fm < idx);
	    if (fm != idx) {
		/* insert */
		fill[m]   = idx;
		fill[idx] = fm;
		fm        = idx;
		nzf++;
		}
	    }
	row   = fill[row];
	nzi++;
	};

    /* Allocate elim_row */
    row      = elim_row->i[0];
    elim_row = frs[prow];
    SPiMallocNVt(BBf,nzf,&elim_row->v,&elim_row->i,err); CHKERRV(1,1);
    if (err) return -(prow+1);
    elim_row->maxn = nzf;
    elim_row->nz   = nzf;

    /* Save the number of non-zeros in the lower triangle */
    nzl[prow] = nzi;
    
    /* Store filled row */
    nnz = nzf;
    xi  = elim_row->i;
    fm  = row;
    while (nnz--) {
	*xi++ = fm;
	fm    = fill[fm];
	}
    im[prow] = nzf;
    }

return ERR_NONE;
}

/*
   SpRComputeFactor - Factor a matrix.

   Description:
   Given a matrix B and a computed fill area BB, 
   find the numerical factor. BB should have be
   obtained previously by SpComputeFill() or
   SpComputeILUFill().   

   Input Parameters:
.  B    - matrix to factor
.  BB   - matrix to hold factor   
*/
int SpRComputeFactor( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          *fill, n, err;
double       *rtmp;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
/* Check for mapping */
if (BB->factor->map) {
    if (((SpRowMat *)(BB->factor->data))->blks)
	err = SpRComputeFactorPermBlock( B, BB );
    else
	err = SpComputeFactorPerm( B, BB );
    }
else {
    n    = B->rows;
    rtmp = (double *)SPAllocTemp( (n+1)*(sizeof(int) + sizeof(double)) );
    CHKPTRV(rtmp,-(n+1));
    fill = (int *)(rtmp + n + 1);
    err = SpComputeFactorBase( B, BB, rtmp, fill );
    SPFreeTemp(rtmp);
    }
return err;
}

int SpComputeFactorBase( B, BB, rtmp, fill )
SpMat       *B;
SpMatSplit *BB;
register double    *rtmp;
register int       *fill;
{
int          prow, row, *xi, nz, n, nnz, *yi;
double       multiplier, *pc, *xv;
SpMat *BBf = BB->factor;
SpVec *pivot_row, *elim_row, **rs;
SpRowMat     *R = (SpRowMat *)B->data, *RR = (SpRowMat *)BBf->data;
int          nf = 0;    /* Used to hold the number of floating point 
			   operations */

n          = B->rows;
/* Nothing to do for the first row */
BB->nzl[0] = 0;
rs         = RR->rs;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  We simultaneously compute the fill and
       update the row. */
    elim_row  = rs[prow];
    yi = xi   = elim_row->i;
    row       = *yi++;
    nz        = elim_row->nz;
    /* Set rtmp to 0 */
    while (nz--) rtmp[*xi++] = 0.0;
    /* Load in the initial row from B */
    pivot_row = R->rs[prow];
    xi        = pivot_row->i;
    xv        = pivot_row->v;
    nz        = pivot_row->nz;
    while (nz--) rtmp[*xi++] = *xv++;
    
    /* Eliminate into (fill,rtmp) */
    while (row < prow) {
	pivot_row = rs[row];
	pc        = rtmp+row;
	if (*pc != 0.0) {
	    nz         = BB->nzl[row];
	    /* Get the pivot row */
	    xv         = pivot_row->v + nz;
	    xi         = pivot_row->i + nz;
	    if (*xi++ != row) 
		{SETERRC(-(row+1),"Zero pivot encountered"); 
                 return -(row+1);} /* zero (missing) pivot */
	    multiplier = *pc * *xv++;
	    *pc        = multiplier;
	    nnz        = pivot_row->nz - nz - 1;
	    nf         += 2*nnz;  /* (really (2*nnz - 1) + 1 */
	    SPARSEDENSESMAXPY(rtmp,multiplier,xv,xi,nnz);
	    }
	row   = *yi++;
	};
    /* copy row into v */
    xv  = elim_row->v;
    xi  = elim_row->i;
    nz  = elim_row->nz;
    /* Replace diagonal entry with inverse (pivot multiplier) */
    if (rtmp[prow] == 0.0) return -(prow+1);
    rtmp[prow] = 1.0 / rtmp[prow];
    GATHER(xv,xi,rtmp,nz);
    }

BB->nf = nf + n;     /* The extra n is for the divides */
return ERR_NONE;
}

