#ifndef lint
static char SCCSid[] = "@(#) ./sparse/ops.c 07/23/93";
#endif

#include "tools.h"
/* NO_GENERIC_SPARSE disables the inlining of things like SpMult. 
   The inlining is useful in Sparse routine that use the generic
   operations to perform their work. */
#define NO_GENERIC_SPARSE
#include "sparse/spmat.h"
#include "sparse/sppriv.h"

/* Routines for the basic operations */

/*@
   SpMult - Computes a matrix-vector product.

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply

   Output parameters:
.  vout - result vector
 @*/
void SpMult( m, vin, vout )
SpMat  *m;
double *vin, *vout;
{
(*m->ops->mv)( m, vin, vout );
}

/*@
   SpMultAdd - Computes a matrix-vector product with addition. 

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply
.  v1  - vector to add to result

   Output parameters:
.  vout - result vector = mat*v + v1
 @*/
void SpMultAdd( m, v, v1, vout )
SpMat  *m;
double *v, *v1, *vout;
{
(*m->ops->mvpv)( m, v, v1, vout );
}

/*@
   SpMultTrans - Computes a matrix-vector product, using the TRANSPOSE
                 of the matrix.

   Input parameters:
.  mat - matrix to multiply with
.  v   - vector to multiply

   Output parameters:
.  vout - result vector = mat*v
@*/
void SpMultTrans( mat, v, vout )
SpMat  *mat;
double *v, *vout;
{
(*mat->ops->mtv)( mat, v, vout );
}

/*@ 
   SpComputeFill - Finds storage for factor of matrix B in BB 
                      (descriptors allocated but not set).

   Note: 
   Because fill propagates down, a single row effectively adds 
   all of the fill for the rows above.  This is crucial in reducing the
   cost of the fill algorithm.  The rule is

$   Let IM = # of non-zeros to include.
$   1. If, at row "prow", we add a row (row) that contains column prow, then
      set IM[row] = nz;
$   2. IM[prow] = nz of row.

 @*/
int SpComputeFill( m, mb )
SpMat      *m;
SpMatSplit *mb;
{
return (*m->ops->fill)( m, mb );
}

/*@
   SpComputeFactor - Factors a matrix.

   Description:
   Given a matrix B and a computed fill area BB, 
   find the numerical factor. BB should have be
   obtained previously by SpComputeFill() or
   SpComputeILUFill().   

   Input Parameters:
.  B    - matrix to factor
.  BB   - matrix to hold factor   
@*/
int SpComputeFactor( m, mb )
SpMat      *m;
SpMatSplit *mb;
{
return (*m->ops->factor)( m, mb );
}

/*@ 
  SpSolve - Solves a system of equations. The matrix must have been
            factored, by, for instance, SpComputeFactor().

  Input Parameters:
.  BB - Split matrix already processed by Factor
.  b  - Right-hand-side
.  x  - solution
 @*/
void SpSolve( m, b, x )
SpMatSplit *m;
double *b, *x;
{
(*m->factor->ops->solve)( m, b, x );
}
/*@ 
  SpSolveTrans - Solves a system of equations using the TRANSPOSE of
                 a factored matrix.

  Input Parameters:
.  BB - Split matrix already processed by Factor
.  b  - Right-hand-side
.  x  - solution
 @*/
void SpSolveTrans( BB, b, x )
SpMatSplit *BB;
double     *b, *x;
{
(*BB->factor->ops->solveTranspose)( BB, b, x );
}

#ifndef SpRDestroy
/*@
  SpDestroy - Frees a sparse matrix (any format).

  Input Parameter:
. m - matrix to destroy
 @*/
void SpDestroy( m )
SpMat *m;
{
(*m->ops->destroy)( m );
}
#endif
/* There really isn't any room for create */

/*@
  SpGatherToRow - Given i[*], v[i[*]], insert into the matrix on the
                  given row

  Input parameters:
.  mat - matrix to set values in
.  row - row to set
.  nz  - number of element
.  i   - indices of columns
.  v   - vector of values

  Note:
  v[i[k]] for k=0 to nz-1 is the value for column i[k]
 @*/
void SpGatherToRow( mat, row, nz, i, v )
SpMat  *mat;
int    nz, *i, row;
double *v;
{
(*mat->ops->gathertorow)( mat, row, nz, i, v );
}

/*@
  SpGatherAddToRow - Given i[*], v[i[*]], add into the matrix on the
                     given row

  Input parameters:
.  mat - matrix to set values in
.  row - row to set
.  nz  - number of element
.  i   - indices of columns
.  v   - vector of values

  Note:
  v[i[k]] for k=0 to nz-1 is the added to the value for column i[k]
 @*/
void SpGatherAddToRow( mat, row, nz, i, v )
SpMat  *mat;
int    nz, *i, row;
double *v;
{
(*mat->ops->gatheraddtorow)( mat, row, nz, i, v );
}

/*@
    SpScatterFromRow - Returns pointers to a compressed row of a matrix

    Input parameters:
.    mat - matrix to set values in
.    row - row to set

    Output parameters:
.    nz  - number of element
.    i   - indices of columns
.    v   - vector of values

    Note:
    v[k] for k=0 to nz-1 is the value for column i[k] .  Note that pointers
    to the values are returned rather than the actual values.  These should
    be considered volatile (don't call any other sparse routines that
    may change the structure of the same matrix).

    If any of the pointers is nil, that value will not be provided.
    For example, 
$      SpScatterFromRow( mat, row, &nz, (int **)0, (double**)0)
    returns the number of non-zeros in the row.
@*/
void SpScatterFromRow( mat, row, nz, i, v )
SpMat  *mat;
int    *nz, **i, row;
double **v;
{
(*mat->ops->scatterfromrow)( mat, row, nz, i, v );
}

/*@
   SpCreateSplit - Allocates an n x m sparse matrix factor, using the same
   format as the given matrix.

   Input Parameters:
.   mat - model matrix (provides number of rows and columns)
.   mmax - estimated number of elements in each row (this many elements
          will be pre-allocated; this value may be set to zero, in which
	  case space will be allocated as required.)  Some versions
	  will ignore this; a reasonable default is zero.
 @*/
SpMatSplit *SpCreateSplit( mat, mmax )
SpMat *mat;
int   mmax;
{
return (*mat->ops->createsplit)( mat, mmax );
}
