#ifndef lint
static char SCCSid[] = "@(#) ./sparse/fblock/fbnftr.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "sparse/fblock/spfbpriv.h"
#include "inline/blas2.h"
#include "inline/blas1.h"

/*
    This file contains a routine to compute a block factorization,
    given a computed fill.  Thus this routine gives incomplete
    factorizations if the symbolic fill was computed using an
    incomplete symbolic factorization.

    There are a number of choices in forming the blocks, particularly the
    diagonal of the matrix.  One is to put the factors of the diagonal
    block there, another is to compute the inverse.

    For this version, the inverse is computed.  The reasons are mostly
    that we'd like to use a pivoting routine for the computation
    with blocks, and it is easier to just compute the inverse than to
    arrange to carry the permutation vector along with the diagonal.
    However, if we decide that this is a problem, then we can add the
    necessary data to handle the factorization, with pivots, of the
    diagonal blocks.

    Given this, the algorithm is distressingly simple.  We just do the
    regular (non-"blocked") version, put the operations are replaced
    with matrix-matrix versions (mostly DAXPY, but also the inverse
    operation).
 */    
/* Given a matrix B and a computed fill area BB, find the numerical 
   factor.
   When eliminating rows, if the rows are:
   A   C
   B   D
   (A, B are single blocks), then the resulting matrix is
   A^-1   C
   BA^-1 (D - BA^-1 C)
 */
int SpFBComputeFactorPerm( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          prow, row, *xi, nz, n, nnz, *yi;
double       multiplier, *pc, *xv, *rtmp, *mmplier;
SpMat        *BBf = BB->factor;
SpFBVec       *pivot_row, *elim_row, **rs;
int          *r = BBf->map->rowmap, *ic = BBf->map->icolmap;
SpFBRowMat    *R = (SpFBRowMat *)B->data, *RR = (SpFBRowMat *)BBf->data;
register int i;
int          bsize, bsize2, oi, ii, j;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
bsize = R->bsize;
bsize2 = bsize * bsize;
n    = B->rows;
rtmp = (double *)MALLOC( (n + 1) * bsize2 * sizeof(double) );
/* mmplier is storage to hold the product of the block matrices */
mmplier = rtmp + n*bsize2;
CHKPTRV(rtmp,ERR_NO_MEM);

/* Nothing to do for the first row */
/* Question: given that we are using blocks, can we spend the time
   to find the correct blocks, rather than using this "fill a vector"
   approach? */
BB->nzl[0] = 0;
rs         = RR->rs;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row. */
    elim_row  = rs[prow];
    yi = xi   = elim_row->i;
    row       = *yi++;
    nz        = elim_row->nz;
    /* Set rtmp to 0 */ 
    for (i=0; i<nz; i++) {
	oi = xi[i] * bsize2;
        for (ii=0; ii<bsize2; ii++) rtmp[oi+ii] = 0.0;
        }
        
    /* Load in the initial row from B */
    pivot_row = R->rs[r[prow]];
    xi        = pivot_row->i;
    xv        = pivot_row->v;
    nz        = pivot_row->nz;
    for (i=0; i<nz; i++) {
        oi = ic[xi[i*bsize]];
        for (ii=0; ii<bsize2; ii++) rtmp[oi+ii] = *xv++;
        }
        
    /* Eliminate into (rtmp) */
    while (row < prow) {
	pivot_row = rs[row];
	pc        = rtmp+row;
	if (*pc != 0) {
	    nz         = BB->nzl[row];
	    /* Get the pivot row */
	    xv         = pivot_row->v + nz;
	    xi         = pivot_row->i + nz;
	    if (*xi++ != row) 
		{SETERRC(-(row+1),"Zero pivot encountered"); 
                 return -(row+1);}   /* zero (missing) pivot */
	    DMV(pc,xv,bsize,bsize,mmplier); xv += bsize2;
	    DCOPY(pc,mmplier,bsize2);
	    nnz        = pivot_row->nz - nz - 1;
	    for (i=0; i<nnz; i++) {
	    	/* we need to add the -1 times the matrix product */
                DVPMV(rtmp+xi[i*bsize],xv+i*bsize2,bsize,bsize,mmplier);
                }
#ifdef FOO                
	    multiplier = *pc * *xv++;
	    *pc        = multiplier;
	    nnz        = pivot_row->nz - nz - 1;
	    SPARSEDENSESMAXPY(rtmp,multiplier,xv,xi,nnz);
#endif	    
	    /* To here */
	    }
	row   = *yi++;
	};
    /* copy row into v */
    xv  = elim_row->v;
    xi  = elim_row->i;
    nz  = elim_row->nz;
    /* Replace diagonal entry with inverse (pivot multiplier) */
    /* Replace this with the inverse of the matrix */
    /* DMINVERT(rtmp+prow*bsize2,bsize); */
    /*     rtmp[prow] = 1.0 / rtmp[prow]; */
    for (i=0; i<nz; i++)
        for (j=0; j<bsize2; j++) *xv++ = rtmp[xi[i*bsize]+j];
    /*     GATHER(xv,xi,rtmp,nz); */
    }

/* Recover temp used to compute fill */
FREE( rtmp );
return ERR_NONE;
}

#ifdef FOO  
int SpFBComputeFactorBase( B, BB, rtmp, fill )
SpMat       *B;
SpMatSplit *BB;
register double    *rtmp;
register int       *fill;
{
int          prow, row, *xi, nz, n, nnz, *yi;
double       multiplier, *pc, *xv;
SpMat        *BBf = BB->factor;
SpFBVec       *pivot_row, *elim_row, **rs;
SpRowMat     *R = (SpRowMat *)B->data, *RR = (SpRowMat *)BBf->data;

n          = B->rows;
/* Nothing to do for the first row */
BB->nzl[0] = 0;
rs         = RR->rs;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  We simultaneously compute the fill and
       update the row. */
    elim_row  = rs[prow];
    yi = xi   = elim_row->i;
    row       = *yi++;
    nz        = elim_row->nz;
    /* Set rtmp to 0 */
    while (nz--) rtmp[*xi++] = 0.0;
    /* Load in the initial row from B */
    pivot_row = R->rs[prow];
    xi        = pivot_row->i;
    xv        = pivot_row->v;
    nz        = pivot_row->nz;
    while (nz--) rtmp[*xi++] = *xv++;
    
    /* Eliminate into (fill,rtmp) */
    while (row < prow) {
	pivot_row = rs[row];
	pc        = rtmp+row;
	if (*pc != 0.0) {
	    nz         = BB->nzl[row];
	    /* Get the pivot row */
	    xv         = pivot_row->v + nz;
	    xi         = pivot_row->i + nz;
	    if (*xi++ != row) 
		{SETERRC(-(row+1),"Zero pivot encountered"); 
                 return -(row+1);} /* zero (missing) pivot */
	    multiplier = *pc * *xv++;
	    *pc        = multiplier;
	    nnz        = pivot_row->nz - nz - 1;
	    SPARSEDENSESMAXPY(rtmp,multiplier,xv,xi,nnz);
	    }
	row   = *yi++;
	};
    /* copy row into v */
    xv  = elim_row->v;
    xi  = elim_row->i;
    nz  = elim_row->nz;
    /* Replace diagonal entry with inverse (pivot multiplier) */
    if (rtmp[prow] == 0.0) return -(prow+1);
    rtmp[prow] = 1.0 / rtmp[prow];
    GATHER(xv,xi,rtmp,nz);
    }

return ERR_NONE;
}

#endif

/*
   SpFBComputeFactor - Factor a matrix.

   Description:
   Given a matrix B and a computed fill area BB, 
   find the numerical factor. BB should have be
   obtained previously by SpComputeFill() or
   SpComputeILUFill().   

   Input Parameters:
.  B    - matrix to factor
.  BB   - matrix to hold factor   
*/
int SpFBComputeFactor( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          *fill, n, err;
double       *rtmp;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
/* Check for mapping */
if (BB->factor->map) {
    err = SpFBComputeFactorPerm( B, BB );
    }
else {
#ifdef FOO
    n    = B->rows;
    rtmp = (double *)SPAllocTemp( (n+1)*(sizeof(int) + sizeof(double)) );
    CHKPTRV(rtmp,-(n+1));
    fill = (int *)(rtmp + n + 1);
    err = SpFBComputeFactorBase( B, BB, rtmp, fill );
    SPFreeTemp(rtmp);
#else
    /* Not yet available */
    err = 1;
#endif
    }
return err;
}
