/*
   A program to test sparse vector ops
 */

#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "sparse/spmat.h"

/* Forward references */
void CompareSolution();
SpMat *BuildLaplacian();

/* If YSMP is defined, compare with YSMP solver */
/* #define YSMP */

#define max(a,b) ( ((a)>(b))?a:b )
#define min(a,b) ( ((a)<(b))?a:b )

double MegaFlopsFactor( BB, time )
SpMatSplit *BB;
double     time;
{
return (SpCostFactor( BB ) / time) * 1.0e-6;
}
double MegaFlopsSolve( BB, time )
SpMatSplit *BB;
double     time;
{
return (SpCostSolve( BB ) / time) * 1.0e-6;
}

main( argc, argv )
int  argc;
char **argv;
{
int        i,n, m, *perm, *iperm, *perm2, *iperm2, err = 0, ordering;
SpMat      *mat, *bmat;
SpMatSplit *factor;
double     *v, t1, t2, SYGetCPUTime(), *vsol, tf;

if (SYArgHasName( &argc, argv, 1, "-help" )) {
    fprintf( stderr, 
	    "%s - test sparse direct solvers, using 2-d Poisson problem\n",
	    argv[0] );
    fprintf( stderr, "-n n for size of problem (n x n mesh)\n" );
    exit(0);
    }
m = 16;
SYArgGetInt( &argc, argv, 1, "-n", &m );
n   = m * m;
v   = (double *)MALLOC( n * sizeof(double) );
vsol= (double *)MALLOC( n * sizeof(double) );

/* Form a permutation */
perm  = (int *)MALLOC( 2*n*sizeof(int) );
iperm = perm + n;
for ( i=0; i<n; i++ ) {iperm[i] = perm[i] = i;}

/* Try symbolic/numeric separately */
TRID(1);
mat = BuildLaplacian( m, v );
TRID(11);
factor = SpCreateSplit( mat, 0 );
TRID(12);
SpSetMappingPtrs( factor, iperm, iperm, perm );
t1 = SYGetCPUTime();
TRID(13);
SpSetMappingPtrs( mat, perm, perm, iperm );
TRID(14);
SpComputeFill( mat, factor );
TRID(15);
SpComputeFactor( mat, factor );
tf = SYGetCPUTime() - t1;
printf( "Time to factor is %f (with reordering) [%f Mf]\n", 
         tf, MegaFlopsFactor( factor, tf) );
printf( "Size of factor is %d\n", SpNz( factor ) );


t1 = SYGetCPUTime();
TRID(16);
SpSolve( factor, v, vsol );
tf = SYGetCPUTime() - t1;
printf( "Time to solve is %f (with reordering) [%f Mf]\n", 
        tf, MegaFlopsSolve( factor, tf ) );
CompareSolution( vsol, n );
SpDestroySplit( factor );

TRID(2);
/* Test code to permute in place */
SpDestroy( mat );
mat = BuildLaplacian( m, v );
TRID(21);
factor = SpCreateSplit( mat, 0 );
TRID(22);
SpSetMappingPtrs( factor, iperm, iperm, perm );

t1 = SYGetCPUTime();
TRID(23);
SpComputeFill( mat, factor );
TRID(24);
SpComputeFactor( mat, factor );
/* SpPrintMatrix( stdout, factor->factor ); */
printf( "Time to factor is %f (inplace with reordering)\n", SYGetCPUTime() - t1 );
printf( "Size of factor is %d\n", SpNz( factor ) );

t1 = SYGetCPUTime();
#ifdef DISPLAY_ORDERING
SpGraphNz( factor->factor, 0, 400 );
#endif

TRID(25);
SpSolve( factor, v, vsol ); 
printf( "Time to solve is %f (inplace with reordering)\n", SYGetCPUTime() - t1 );
CompareSolution( vsol, n );
SpDestroySplit( factor );
SpDestroy( mat );

/* Test various alternate orderings */
mat = BuildLaplacian( m, v );
for (ordering=0; ordering < 4; ordering++) {
    factor = SpCreateSplit( mat, 0 );
    TRID(12+ordering*100);
    t1 = SYGetCPUTime();
    SpOrder( mat, ordering, iperm, perm );
    t1 = SYGetCPUTime() - t1;
    printf( "Time to order is %f\n", t1 );
    SpSetMappingPtrs( factor, iperm, iperm, perm );
    t1 = SYGetCPUTime();
    TRID(13+ordering*100);
    SpSetMappingPtrs( mat, perm, perm, iperm );
    TRID(14+ordering*100);
    SpComputeFill( mat, factor );
/*
    printf( "Run structure for this ordering %s is:\n", 
	     SpOrderName(ordering) );
 */
    t2 = SYGetCPUTime();
    printf( "Time to compute symbolic factor is %f\n", t2 - t1 );
    t2 = SYGetCPUTime();
    SpFindBlocks( factor->factor, 8 );
    t2 = SYGetCPUTime() - t2;
    printf( "Time to find runs is %f\n", t2 );
    TRID(15+ordering*100);
    t2  = SYGetCPUTime();
    SpComputeFactor/*PermBlock*/( mat, factor );
    /* SpComputeFactor( mat, factor );   */
    tf  = SYGetCPUTime() - t2;
    printf( "Time to factor is %f [%f Mf] (with reordering %s)\n",
             tf, MegaFlopsFactor( factor, tf ), SpOrderName(ordering) );
/* trSummary( stdout ); */
#ifdef DISPLAY_ORDERING
SpGraphNz( factor->factor, 0, 400 );
#endif
    printf( "Size of factor is %d\n", SpNz( factor ) );

    t1 = SYGetCPUTime();
    TRID(16+ordering*100);
    SpSolve( factor, v, vsol );
    tf = SYGetCPUTime() - t1;
    printf( "Time to solve is %f [%f Mf] (with reordering)\n", 
	     tf, MegaFlopsSolve( factor, tf ) );
    CompareSolution( vsol, n );
    SpDestroySplit( factor );
    }
SpDestroy( mat );

#ifdef YSMP
TRID(3);
/* Compare with ysmp */
{ int *ia, *ja, nsp, esp, flag, path, maxj;
double *a, *rsp;
void ShiftPerm(), InverseOrderingOne();

mat = BuildLaplacian( m, vsol );
ia  = (int *)MALLOC( (n+1)*sizeof(int) );
maxj= 2*n*m;
ja  = (int *)MALLOC( maxj*sizeof(int) );
a   = (double*)MALLOC( maxj*sizeof(double) );
SpToAIJ( mat, ia, ja, a, maxj );
ShiftPerm( perm, n );
InverseOrderingOne( perm, iperm, n );

t1 = SYGetCPUTime();
flag = 0;
path = 1;
nsp  = 10*maxj;
rsp  = (double *)MALLOC( nsp*sizeof(double) );
ndrv_( &n, iperm, iperm, perm, ia, ja, a, vsol, v, &nsp, rsp, rsp, &esp, 
       &path, &flag );
if (flag != 0) {
    printf( "Error in ndrv call %d\n", flag );
    ndrv_error( stdout, flag, n );
    }
printf( "Time to factor is %f\n", SYGetCPUTime() - t1 );
/* printf( "Size of factor is %d\n", SpNz( mat ) ); */
  }
CompareSolution( v, n );

FREE(ia);
FREE(ja);
FREE(a);
FREE(rsp);
#endif

/* Here is some code to test Copy, Subsetting, and Mappings */
{ 
int *rows, *cols, i;
SpMat *mat2, *mat3;

TRID(100);
mat  = BuildLaplacian( m, v );
TRID(101);
SpSetMappingPtrs( mat, perm, perm, iperm );
TRID(102);
mat2 = SpCopy( mat );
TRID(103);
rows = (int *)MALLOC( m/2 * sizeof(int) );
cols = (int *)MALLOC( m/2 * sizeof(int) );
for (i=0; i<m/2; i++) {
    rows[i] = i;
    cols[i] = i;
    }
TRID(104);
mat3 = SpSubMatrixInPlace(mat,0,0,cols,m/2);

TRID(105);
SpDestroy( mat );
TRID(106);
SpDestroy( mat2 );
TRID(107);
SpDestroy( mat3 );
TRID(108);
FREE(rows); FREE(cols);

TestInverse();
}

/* At this point, mat2 and mat3 share storage for the matrix entries 
   but NOT for the mappings */

/* Here is some code to print the memory usage */
FREE(v);
FREE(vsol);
FREE(perm);
/*SpChunckFlush(); */
SpChunckClear();
{int s, f;
TRSPACE(&s,&f);
printf( "Memory allocated = %d in %d blocks\n", s, f );
printf( "List of blocks is\n" );
TRDUMP(stdout);
printf( "Dump of memory management chuncks\n" );
SpChunckPrintUsage( stdout );
 }
return 0;
}

SpMat *BuildLaplacian( m, vrhs )
int    m;
double *vrhs;
{
SpMat       *f;
double sum;
int    n, i, err=0;

n   = m * m;
/* This would be faster if the last parameter was 5, but using 1 gives
   us more testing (on-the-fly allocation of space) */
f   = SpCreate( n, n, 1 );

for (i=0; i<n; i++) {
    sum = 0.0;
    if (i - m >= 0) {
	SpAddValue( f, -1.0, i, i - m );
	sum += -1.0;
	}
    if (i - 1 >= 0) {
	SpAddValue( f, -1.0, i, i - 1 );
	sum += -1.0;
	}
    SpAddValue( f, 4.0, i, i );
    sum += 4.0;
    if (i + 1 < n) {
	SpAddValue( f, -1.0, i, i + 1 );
	sum += -1.0;
	} 
    if (i + m < n) {
	SpAddValue( f, -1.0, i, i + m );
	sum += -1.0;
	}
    vrhs[i] = sum;
    }
return f;
}

void CompareSolution( v, n )
register double *v;
register n;
{
register double sum = 0.0;
while (n--) {
    sum += (*v - 1.0) * (*v - 1.0);
    v++;
    }
printf( "Difference in 2-norm is %f\n", sum );
}

/* Add 1 to perm (change from zero origin to one origin */
void ShiftPerm( perm, n )
register int    *perm, n;
{
while (n--) 
    *perm++ += 1;
}

void InverseOrderingOne( perm, iperm, n )
int *perm, *iperm, n;
{
int i;

for (i=1; i<=n; i++) {
    iperm[*perm++ - 1] = i;
    }
}

TestInverse()
{
SpMat      *mat;
SpMatSplit *factor;
int        i, j, err;
double     *x, *y, *xt, *b;

mat = SpCreate( 3, 3, 3 );

for (i=0; i<3; i++)
    for (j=0; j<3; j++) 
	SpAddValue( mat, (double)(i+j), i, j );
SpAddValue( mat, 1.0, 0, 0 );
factor = SpCreateSplit( mat, 3 );
SpComputeFill( mat, factor );
SpComputeFactor( mat, factor );
b      = (double *)MALLOC( 3 * sizeof(double) );
x      = (double *)MALLOC( 3 * 3 * sizeof(double) );
xt     = (double *)MALLOC( 3 * 3 * sizeof(double) );
/* We compute inverses by computing the COLUMNS of the inverse.
   For the transpose case, this gives us the ROWS of the inverse */
/* Compute the inverse matrix of the transpose */
for (i=0; i<3; i++) {
    for (j=0; j<3; j++) b[j] = 0.0;
    b[i] = 1.0;
    SpSolveTrans( factor, b, xt + (3*i) );
    }
/* Compute the inverse matrix */    
for (i=0; i<3; i++) {
    for (j=0; j<3; j++) b[j] = 0.0;
    b[i] = 1.0;
    SpSolve( factor, b, x + (3*i) );
    }

/* Now that we have the inverse, print it */
printf( "Inverse matrix\n" );
DisplayMat( x, 3, 3 );
printf( "Inverse of transpose\n" );
DisplayMat( xt, 3, 3 );

/* Check that the two versions match up */
for (i=0; i<3; i++)
    for (j=0; j<3; j++) {
    	if (fabs(x[i+j*3] - xt[j+i*3]) > 1.0e-8) {
    	    printf( "a^-1(%d,%d)[%e] != (a^-T)^T[%e]\n", i, j,
    	    	    x[i+j*3], xt[j+i*3] );
    	    }
        }

SpDestroy( mat );
SpDestroySplit( factor );
FREE(b); FREE(x); FREE(xt);
}

DisplayMat( x, nr, nc )
double *x;
int    nr, nc;
{
int    i, j;
double v;  /* Used to clamp value */

for (i=0; i<nr; i++) {
    for (j=0; j<nc; j++) {
	v = x[i+j*nr];
	if (fabs(v) < 1.e-300) v = 0.0;
    	printf( "%13e ", v );
        }
    printf( "\n" );
    } 
}
