#ifndef lint
static char SCCSid[] = "@(#) ./sparse/dense/dncreate.c 07/23/93";
#endif

/*
   This file contains routines for representing and manipulating matrices
   in dense format.  
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "sparse/dense/dense.h"
#include "inline/blas1.h"
#include "inline/blas2.h"

#define HAS_LAPACK

void   SpDnMult(), SpDnMultTrans();
void   SpDnDestroy();
void   SpiDnForwardsolve(), SpiDnBacksolve();
SpMat  *SpDnCreate();
SpMatSplit *SpDnCreateSplit();
int    SpDnFill(), SpDnFactor();
void   SpDnSolve();
void   SpDnScatterFromRow(), SpDnGatherToRow(), SpDnGatherAddToRow();
static SpOps _SpDenseOps = { SpDnMult, 0, SpDnMultTrans, 
			     SpDnSolve, 0,
			     SpiDnForwardsolve,   SpiDnBacksolve,
			     SpDnDestroy, 
			     (void *(*)())SpDnCreate, 
			     (void *(*)())SpDnCreateSplit,
			     0, SpDnFill, SpDnFactor, 
			     0, 0, 
			     SpDnGatherToRow, SpDnGatherAddToRow,
			     SpDnScatterFromRow, 0,
			     0,
			     0, 0, 0 };

/*
   SpDnMult - Form the matrix-vector product vout = mat * vin.

   Input Parameters:
.  nmat  - matrix (in dense format)
.  vin   - vector to multiply

   Output Parameters:
.  vout - result vector
*/
void SpDnMult( nmat, vin, vout )
SpMat  *nmat;
double *vin, *vout;
{
SpMatDense    *mat = (SpMatDense *)nmat->data;
double        *p = mat->p;
int _One=1;double _DOne=1.0, _DZero=0.0;

LAgemv_( "N", &(nmat->rows), &(nmat->cols), &_DOne, p, &(mat->decl_rows), 
	 vin, &_One, &_DZero, vout, &_One, 1 );
/* DMV(vout,p,nmat->rows,nmat->cols,vin); */
}

/*
   SpDnMultTrans - Form the matrix-vector product vout = mat' * vin.

   Input Parameters:
.  nmat  - matrix (in dense format)
.  vin   - vector to multiply

   Output Parameters:
.  vout - result vector
*/
void SpDnMultTrans( nmat, vin, vout )
SpMat  *nmat;
double *vin, *vout;
{
SpMatDense    *mat = (SpMatDense *)nmat->data;
double        *p = mat->p;
int _One=1;double _DOne=1.0, _DZero=0.0;

LAgemv_( "T", &(nmat->rows), &(nmat->cols), &_DOne, p, &(mat->decl_rows), vin, 
       &_One, &_DZero, vout, &_One, 1 );
/* DMV(vout,p,nmat->rows,nmat->cols,vin); */
}

/*@
   SpDnCreateFromData - Creates a dense matrix descriptor, given an existing
   dense matrix.

   Input Parameters:
.  nr - number of rows
.  nrd - declared number of rows
.  nc - number of columns
.  p   - pointer to matrix
 @*/
SpMat *SpDnCreateFromData( nr, nrd, nc, p )
int    nr, nrd, nc;
double *p;
{
SpMatDense *mat;
SpMat      *nmat;

mat            = NEW(SpMatDense);      CHKPTRV(mat,0);
mat->p         = p;
mat->quser     = 1;
mat->decl_rows = nrd;
mat->pivots    = 0;

/* Allocate and initialize the SpMat structure (there should be a separate
   routine for this) */
nmat                 = NEW(SpMat);         CHKPTRV(nmat,0);
nmat->type           = MATDENSE;
nmat->ops            = &_SpDenseOps;
nmat->rows           = nr;
nmat->cols           = nc;
nmat->is_sorted      = 0;
nmat->element_length = 8;
nmat->nz             = 0;
nmat->pool.pool.ptr  = 0;
nmat->pool.pool.n    = 0;
nmat->pool.pool.next = 0;
nmat->pool.alloc     = 0;
nmat->pool.free      = 0;
nmat->alloc_together = 0;
nmat->alloc_incr     = 0;
nmat->map            = 0;
nmat->data           = (void *)mat;

return nmat;
}

/*@
   SpDnCreate - Creates a dense matrix descriptor.

   Input Parameters:
.  nr - number of rows
.  nc - number of columns
 @*/
SpMat *SpDnCreate( nr, nc )
int   nr, nc;
{
double *p;
SpMat      *nmat;

p    = (double *)MALLOC( nr * nc * sizeof(double) );  CHKPTRV(p,0);
nmat = SpDnCreateFromData( nr, nr, nc, p );           CHKERRV(1,0);
((SpMatDense *)(nmat->data))->quser = 0;
return nmat;
}

/*
  SpDnDestroy - Frees storage for a dense matrix. 

  Input Parameter:
. nmat - dense format matrix to free
 */
void SpDnDestroy( nmat )
SpMat     *nmat;
{
SpMatDense *mat = (SpMatDense *)nmat->data;

/* Free the storage if not user defined */
if (!mat->quser) FREE( mat->p );

if (mat->pivots) FREE( mat->pivots );
FREE( mat );
FREE( nmat );
}

/* These are factor and solve routines; they should use LAPACK 
   (with pivoting)*/
void SpDnSolve( m, b, x )
SpMatSplit *m;
double *b, *x;
{
SpMat      *mat = m->factor;
SpMatDense *dmat = (SpMatDense *)mat->data;
int        i, one = 1, info, n = mat->rows;

for (i=0; i<n; i++) x[i] = b[i];
#ifdef HAS_LAPACK
LAgetrs_( "N", &n, &one, dmat->p, &dmat->decl_rows, dmat->pivots, 
         x, &n, &info, 1 );
#endif
}

int SpDnFill( m, mb )
SpMat      *m;
SpMatSplit *mb;
{
SpMat      *mat = mb->factor;
SpMatDense *dmat = (SpMatDense *)mat->data;

/* If dmat not allocated, create it here */

return 0;
}

int SpDnFactor( m, mb )
SpMat      *m;
SpMatSplit *mb;
{
SpMat      *mat = mb->factor;
SpMatDense *dmat = (SpMatDense *)mat->data;
int        one = 1, info, n = mat->rows;

/* Allocate the pivot vector if needed */
if (!dmat->pivots) {
    dmat->pivots = (int *)MALLOC( n * sizeof(int) );  
    CHKPTRV(dmat->pivots,-1);
    }
#ifdef HAS_LAPACK
/* Since the factor is inplace, we need to copy mat into dmat */
SpCopyOver( m, mat );
LAgetrf_( &n, &mat->cols, dmat->p, &dmat->decl_rows, dmat->pivots, &info );
if (info) 
    return (info > 0) ? (-(info + 1)) : info - 1;
#endif
return 0;
}

/* These are used to hold scattered values */
static double *dnrow;
static int    dnsize = 0, *dncols;

void SpDnScatterFromRow( mat, row, nz, i, v )
SpMat  *mat;
int    *nz, **i, row;
double **v;
{
SpMatDense *dmat = (SpMatDense *)mat->data;
double     *p    = dmat->p;
int        nc    = mat->cols;
int        j, k;
int        nr    = dmat->decl_rows;

if (nz) *nz = nc;
if ((i || v) && nc > dnsize) {
    if (dnsize > 0) {
	FREE( dnrow );
	FREE( dncols );
	}
    dnrow  = (double *)MALLOC( nc * sizeof(double) );
    dncols = (int *)MALLOC( nc * sizeof(int) );
    dnsize = nc;
    for (j=0; j<nc; j++) 
	dncols[j] = j;
    }
if (i) *i = dncols;
if (v) {
    *v = dnrow;
    k  = row;
    for (j=0; j<nc; j++) {
	dnrow[j] = p[k];
	k        += nr;
	}
    }
}

void SpDnGatherToRow( mat, row, nz, i, v )
SpMat  *mat;
int    nz, *i, row;
double *v;
{
SpMatDense *dmat = (SpMatDense *)mat->data;
double     *p    = dmat->p;
int        j;
int        nr    = dmat->decl_rows;

p += row;
for (j=0; j<nz; j++) 
    p[i[j]*nr] = v[j];
}

void SpDnGatherAddToRow( mat, row, nz, i, v )
SpMat  *mat;
int    nz, *i, row;
double *v;
{
SpMatDense *dmat = (SpMatDense *)mat->data;
double     *p    = dmat->p;
int        j;
int        nr    = dmat->decl_rows;

p += row;
for (j=0; j<nz; j++) 
    p[i[j]*nr] += v[j];
}

/*+
   SpDnCreateSplit - Allocates an n x m dense matrix factor.

   Input Parameters:
.   model - model matrix   
.   mmax - estimated number of elements in each row (this many elements
          will be pre-allocated; this value may be set to zero, in which
	  case space will be allocated as required.)
 +*/
SpMatSplit *SpDnCreateSplit( model, mmax )
SpMat      *model;
int        mmax;
{
int  n = model->rows, m = model->cols;
SpMatSplit *mat;

TRPUSH(SPTRID + TRIDCREATE + 1);
mat         = NEW(SpMatSplit);  CHKPTRV(mat,0);
mat->type   = MATSPLIT;
mat->factor = SpDnCreate( n, m ); CHKERRV(mat->factor,0);
mat->nzl    = (int *) MALLOC( n*sizeof(int) ); CHKPTRV(mat->nzl,0);
TRPOP;
return mat;
}

/* 
   Forward and back solves with multiplier.  If omega == 1, we can use
   a Level-2 BLAS; otherwise we have to do this directly 
 */
/*+
   SpiDnForwardsolve - For use inside the SSOR preconditioner. Performs 
   a forwardsolve with a relaxation factor omega.

   Input Parameters:
.  BB - Split matrix
.  omega - SSOR parameter
.  b     - right-hand side vector

   Output Parameter:
.  x     - solution vector
 +*/
void SpiDnForwardsolve( BB, omega, b ,x)
SpMatSplit  *BB;
double      *x, *b, omega;
{
  int             i, j, kj, nrd, n;
  register double *d, sum;
  SpMat           *B = BB->factor;
  SpMatDense      *mat = (SpMatDense *)B->data;
  
  n      = B->rows;
  nrd    = mat->decl_rows;
  d      = mat->p;
  for (i=0; i<n; i++) {
      sum = b[i];
      kj  = i;
      for (j=0; j<i; j++) {
	  sum -= x[j] * d[kj];
	  kj += nrd;
	  }
    x[i] = omega * sum/d[kj];
  }
}
/*------------------------------------------------------------------*/
/*+
   SpiDnBacksolve - For use inside the SSOR preconditioner. Performs 
   a backsolve with a relaxation factor omega.

   Input Parameters:
.  BB - Split matrix
.  omega - SSOR parameter
.  b     - right-hand side vector

   Output Parameter:
.  x     - solution vector
 +*/
void SpiDnBacksolve( BB, omega, b, x)
SpMatSplit  *BB;
double      *x, *b, omega;
{
  int             i, j, kj, nrd, n;
  register double *d, sum, diag;
  SpMat           *B = BB->factor;
  SpMatDense      *mat = (SpMatDense *)B->data;
  
  n      = B->rows;
  nrd    = mat->decl_rows;
  d      = mat->p;
  for (i=n-1; i>=0; i--) {
    sum  = b[i];
    kj   = i * (nrd + 1);
    diag = d[kj];
    kj   += nrd;
    for (j=i+1; j<n; j++) {
	sum -= d[kj] * x[j];
	kj  += nrd;
	}
    x[i] = omega * sum / diag;
  }
}

