/*
       These routines provide access to some of the methods in the IBM
       ESSL library.  This file is of interest to those who want to add
       new "black-box" solvers to SLES, as well as those using ESSL.

       Things to handle here
       The code requires preallocation (!) of sparse in the INPUT matrix
       for the factorization in the direct method.
*/
#include <math.h>
#include "tools.h"
#include "solvers/svctx.h"
#include "solvers/svpriv.h"

#if defined(rs6000) && defined(rs6000essl)

/* Forward definitions */
void SViSetupESSLLU(), SViDestroyESSLLU();
int  SViSolveESSLLU();

typedef struct {
  double       *aux, rparm[5], oparm[5];
  int          lna, naux, iparm[5];
  int          *ia, *ja;
  double       *a;
  int          nr;
} SVESSLLUctx;

/*ARGSUSED*/
void SViCreateESSLLU(ctx,mat)
SVctx *ctx;
SpMat *mat;
{
  SVESSLLUctx *lctx;

  lctx = NEW(SVESSLLUctx); CHKPTR(lctx);
  ctx->private = (void *) lctx;

  ctx->setup   = SViSetupESSLLU;
  ctx->solve   = SViSolveESSLLU;
  ctx->destroy = SViDestroyESSLLU;
}
/*------------------------------------------------------------------*/
void SViSetupESSLLU(ctx)
SVctx *ctx;
{
  SVESSLLUctx *lctx = (SVESSLLUctx *) ctx->private;
  double   t1;
  int      one = 1, nz, lna, nr;

  t1          = SYGetCPUTime();
  nr          = ctx->mat->rows;
  nz          = SpNz( ctx->mat );
  lna         = nz * sqrt( (double)nz );
  lctx->naux  = 10 * nr + 100;            /* From ESSL manual, page 611 */

  /* Make an AIJ version of the matrix with extra space. */
  lctx->lna   = lna;
  /* Note that ia must be of size lna, not nr+1! (page 611) */
  lctx->ia    = (int *)MALLOC( lna * sizeof(int) );        CHKPTR(lctx->ia);
  lctx->a     = (double *)MALLOC( lna * sizeof(double) );  CHKPTR(lctx->a);
  lctx->ja    = (int *)MALLOC( lna * sizeof(int) );        CHKPTR(lctx->ja);
  lctx->aux   = (double *)MALLOC( lctx->naux * sizeof(double) );
                CHKPTR(lctx->aux);
  SpToAIJ( ctx->mat, lctx->ia, lctx->ja, lctx->a, lna );
  lctx->rparm[0] = 0.0;
  lctx->rparm[1] = 0.1;        /* See page 501 */
  lctx->iparm[0] = 0;          /* Use defaults; see page 610 */
  dgsf( &one, &nr, &nz, lctx->a, lctx->ia, lctx->ja, &lna, lctx->iparm,
        lctx->rparm, lctx->oparm, lctx->aux, &lctx->naux );
  /* oparm[0] is amount of fill */
  ctx->setupcalled = 1;
  ctx->t_setup += SYGetCPUTime() - t1;
}
/*------------------------------------------------------------------*/
int SViSolveESSLLU(ctx,b,x)
SVctx   *ctx;
double  *b,*x;
{
  SVESSLLUctx *lctx = (SVESSLLUctx *) ctx->private;
  double  t1;
  int     nr = ctx->mat->rows, zero= 0;
  
  t1          = SYGetCPUTime();
  MEMCPY( x, b, nr*sizeof(double) );
  dgss( &zero, &nr, lctx->a, lctx->ia, lctx->ja, &lctx->lna, x, 
        lctx->aux, &lctx->naux );
  ctx->t_solve += SYGetCPUTime() - t1;
  return 0;
}
/*------------------------------------------------------------------*/
void SViDestroyESSLLU(ctx)
SVctx *ctx;
{
  SVESSLLUctx *lctx = (SVESSLLUctx *) ctx->private;

  FREE( lctx->aux );
  FREE( lctx->ia );
  FREE( lctx->ja );
  FREE( lctx->a );
  FREE(ctx->private); FREE(ctx);
}

/* 
   These routines provide for vendor-independent specification of the ESSL
   routines (by being no-ops when ESSL is not available) 
 */
void SVAddESSL()
{
  /*  SVRegisterAll(); */
  SVRegister( -1,    "essllu",     SViCreateESSLLU);
}
#else
void SVAddESSL()
{
}
#endif
