/*
     Test of the nonlinear solvers code. This uses several problems
 from the minpack2 test suite. The Jacobians are treated as dense in
 the minpack2 test suite.  This code allows the user to pick between a
 dense representation and a sparse one, and to pick the method.

 This program includes everything with the possible exception of the
 kitchen sink.  This is NOT usually the best way to proceed, but 
 we do it here to show how the various packages can co-exist.

     Thanks to Brett Averick and Jorge More' for the test problems.

*/
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "tools.h" 
#include "solvers/svctx.h"
#include "nonlin/nlall.h"

extern SpMat *SpDnCreateFromData(), *SpDnClampToSparse();

#if defined(FORTRANCAPS)
#define dficfj_   DFICFJ
#define dsfdfj_   DSFDFJ
#define dhhdfj_   DHHDFJ
#define dcprfj_   DCPRFJ
#define dcpffj_   DCPFFJ
#define dsfifj_   DSFIFJ
#define dfdcfj_   DFDCFJ
#define dgetf2_   DGETF2
#define dgetrs_   DGETRS
#define dgemv_    DGEMV

#elif !defined(FORTRANUNDERSCORE)
#define dficfj_   dficfj
#define dsfdfj_   dsfdfj
#define dhhdfj_   dhhdfj
#define dcprfj_   dcprfj
#define dcpffj_   dcpffj
#define dsfifj_   dsfifj
#define dfdcfj_   dfdcfj
#define dgetf2_   dgetf2
#define dgetrs_   dgetrs
#define dgemv_    dgemv
#endif

typedef struct {
  int     n;                  
  double  *jacobian;
  double  *jacobianFtr;
  SVctx   *svctx;
  SpMat   *fjac;
  int     *pivot;
  int     Problem;
} DenseCntx;

DenseCntx    *CreateDense();
void         DestroyDense();
void         Function();
void         InitialGuess();
void         SetUpLinearSolve();
void         LinearSolve();
void         LinearDestroy();
void         DestroyLinearSolve();

/* For the dog-leg routine */
void GetJ(), JTv(), Jv(), JSolve(), ScaleJ();

/* Problem parameters */
static double Reynolds = 1.0;
static double Viscosity = 10.0;
static int    HHp = 1;
static double Lambda = 2.0;
static int    Nx = 10;
static int    Ny = 10;
static int    UseSparse = 0;
static int    SVmethod = SVLU;
static FILE   *SumFile = 0;

#define ERR {printf("Error %d %s \n",__LINE__,__FILE__); exit(0);}

/* These are the problem sizes.  For those that can must be a multiple
   of something, the multiple is given as a negative value */
static int    Ns[7] = { -8, -14, 8, 5, 11, 0, 0 };

main(argc,argv)
int  argc;
char *argv[];
{
  NLCntx    *neP;
  DenseCntx *neD;
  int       n, its, i, Problem = 1, nfunc,nsteps,nvector;
  int       stepkind = 0, dodis = 0;
  double    *x, *f, norm,xnorm;
  NLMETHOD  nlmethod = NLNEWTONLS;
  char      fname[100];

  SYSetFPTraps();
  if (SYArgHasName( &argc, argv, 1, "-help")) {
   fprintf( stderr, 
      "%s: [-t <trust>] [-l <linesearch>] [-t2 <trustminpack2>]\n", argv[0] );
   fprintf( stderr, "    [-d <doglegstep> ] [-p n <problem>] [-log]\n" );
   fprintf( stderr, "[-Re <reynolds#>] [-L <lambda>] [-V <viscosity>]\n" );
   fprintf( stderr, "[-nx <nn>] [-ny <nn>] [-n <nn>] [-hp <heart-problem>]\n");
   fprintf( stderr, "[-sparse]\n" );
   fprintf( stderr, "-p 1 - driven cavity (dficfj)\n" );
   fprintf( stderr, "-p 2 - Swirling Flow between Disks (dsfdfj)\n" );
   fprintf( stderr, "-p 3 - Human Heart Dipole (dhhdfj)\n" );
   fprintf( stderr, 
            "-p 4 - Combustion of Propane - Reduced Formulation (dcprfj)\n" );
   fprintf( stderr, 
            "-p 5 - Combustion of Propane- Full Formulation (dcpffj)\n" );
   fprintf( stderr, "-p 6 - Solid Fuel Ignition (dsfifj)\n" );
   fprintf( stderr, "-p 7 - Flow in a Driven Cavity (dfdcfj)\n" );
   exit(0);
   }
  if (SYArgHasName( &argc, argv, 1, "-t" )) nlmethod  = NLNEWTONTR;
  if (SYArgHasName( &argc, argv, 1, "-l" )) nlmethod  = NLNEWTONLS;
  if (SYArgHasName( &argc, argv, 1, "-t2" )) nlmethod = NLNEWTONTR2;
  if (SYArgHasName( &argc, argv, 1, "-dis")) dodis = 1;
  if (SYArgHasName( &argc, argv, 1, "-sparse")) UseSparse = 1;
  if (SYArgHasName( &argc, argv, 1, "-d" )) {
      stepkind  = 1;
      if (nlmethod == NLNEWTONLS)
	  nlmethod  = NLNEWTONTR;
      }
  if (SYArgGetInt( &argc, argv, 1, "-p", &Problem )) {
      if (Problem > 7) ERR;
      }
  SYArgGetDouble( &argc, argv, 1, "-Re", &Reynolds );
  SYArgGetDouble( &argc, argv, 1, "-L",  &Lambda );
  SYArgGetDouble( &argc, argv, 1, "-V",  &Viscosity );
  SYArgGetInt( &argc, argv, 1, "-hp", &HHp );
  SYArgGetInt( &argc, argv, 1, "-nx", &Nx );
  SYArgGetInt( &argc, argv, 1, "-ny", &Ny );
  n = 0;
  SYArgGetInt( &argc, argv, 1, "-n", &n );
  if (SYArgGetString( &argc, argv, 1, "-summary", fname, 100 )) {
      SumFile = fopen( fname, "a" );
      }

  /* Adjust/check n */
  if (Ns[Problem-1] > 0) 
      n = Ns[Problem-1];
  else if (Ns[Problem-1] < 0) {
      if (n == 0) n = -2 * Ns[Problem-1];
      else if (n % (-Ns[Problem-1]) != 0) {
	  fprintf( stderr, "Invalid problem size, must be multiple of %d\n",
		   -Ns[Problem-1] );
	  }
      }
  else if (n == 0) {
      n = Nx * Ny;
      }
	      
  neD         = CreateDense(Problem,n); if (!neD) ERR;
  /* n           = neD->n; */

  printf("Running Problem number %d Size %d \n",Problem, n);

  neP         = NLCreate(nlmethod);       if (!neP) ERR;
  NLSetVectorOps(neP,VECreate());
  DVSetDefaultFunctions(neP->vc);
  NLSetFunction(neP,Function);
  NLSetInitialGuess(neP,InitialGuess);
  if (nlmethod == NLNEWTONTR2)
      NLSetJv(neP,Jv,neD);
  if (stepkind) {
      NLSetStepSetUp(neP,NLDogLegSetup);
      NLSetStepCompute(neP,NLDogLeg);
      NLSetStepDestroy(neP,NLDogLegDestroy);
      NLDogLegCreate( neP, neD, neD, GetJ, (void (*)())0, JTv, Jv, 
		      JSolve, neD );
#ifdef USEX
      if (dodis)
	  NLSetDogLegMonitor( neP, NLDogLegMonitor );
#endif
      }
  else {
      NLSetStepSetUp(neP,SetUpLinearSolve);
      NLSetStepCompute(neP,LinearSolve);
      NLSetStepDestroy( neP, LinearDestroy );
      }
  neP->vec_sol = (void *) MALLOC(n*sizeof(double)); if (!neP->vec_sol) ERR;
  neP->residual_history = (double *) MALLOC(neP->max_it*sizeof(double));
  if (!neP->residual_history) ERR;
  if (SYArgHasName( &argc, argv, 1, "-log" )) neP->fp = stderr;

#ifdef USEX
  if (Problem == 6 || Problem == 7) {
      void MeshMonitor();
      XBQContourMappingType( 1 );
      NLSetMonitor( neP, MeshMonitor );
      }
#endif
  NLSetUp(neP,neD);

  its = NLSolve(neP,neD); printf("Number of Newton iterations %d \n",its);

  for ( i=0; i<its+1; i++ ) printf("%d %g \n",i,neP->residual_history[i]);

  NLGetNumberFunctionApplicationsUsed(neP,nfunc);
  NLGetNumberStepComputationsUsed(neP,nsteps);
  NLGetNumberVectorOpsUsed(neP,nvector);

  printf("Function evaluations %d \n",nfunc);
  printf("Inverse Jacobian Applications %d \n",nsteps);
  printf("Number of vector operations %d \n",nvector);

  if (SumFile) {
      char method[10], ProbName[4];
      double param, fnorm0, fnorme;
      if (nlmethod == NLNEWTONLS) strcpy( method, "LS" );
      else {
	  if (nlmethod == NLNEWTONTR)
	      strcpy( method, "TR" );
	  else 
	      strcpy( method, "TR2" );
	  if (stepkind)
	      strcat( method, "_D" );
	  }
      param = 0.0;
      switch (Problem) {
	  case 1: param = Reynolds;  strcpy( ProbName, "FIC" ); break;
	  case 2: param = Viscosity; strcpy( ProbName, "SFD" ); break;
	  case 3: param = HHp;       strcpy( ProbName, "HHD" ); break;
	  case 4:                    strcpy( ProbName, "CPR" ); break;
	  case 5:                    strcpy( ProbName, "CPF" ); break;
	  case 6: param = Lambda;    strcpy( ProbName, "SFI" ); break; 
	  case 7: param = Reynolds;  strcpy( ProbName, "FDC" ); break;
	  }
      fnorm0 = neP->residual_history[0];
      i = its;
      while (neP->residual_history[i] <= 0.0) i--;
      fnorme = neP->residual_history[i];
      fprintf( SumFile, "%d %3s %10.2lf %10s %4d %4d %4d %12.4le %12.4le\n", 
	       Problem, ProbName, 
	       param, method, n, nfunc, nsteps, fnorm0, fnorme );
      }
#ifdef USEX
  if (Problem == 6 || Problem == 7) {
      XBQContourYdirection( 1 );
      XBQContour( neP->vec_sol, (double *)0, (double *)0, Nx, Ny, 32 );
      getc( stdin );
      }
#endif
  NLDestroy(neP,neD);
  DestroyDense(neP,neD);
}
/*-------------------------------------------------------------*/
void Function(nlP,neD,x,f)
NLCntx    *nlP;
DenseCntx *neD;
double    *x,*f;
{
  int    n = neD->n,info = 1,prob = neD->Problem;
  int    nint;
  char   HHProb[6];

switch (prob) {
    case 1:
        nint = n / 8;
	dficfj_(&n,x,f,NULL,&n,"F",&Reynolds,&nint,1);
	break;
    case 2:
	nint = n / 14;
	dsfdfj_(&n,x,f,NULL,&n,"F",&Viscosity,&nint,1);
	break;

    case 3:
	sprintf( HHProb, "DHHD%d", HHp );
	dhhdfj_(&n,x,f,NULL,&n,"F",HHProb,1,5 );
	break;

    case 4:
	dcprfj_(&n,x,f,NULL,&n,"F",1);
	break;

    case 5:
	dcpffj_(&n,x,f,NULL,&n,"F",1);
	break;

    case 6:
	dsfifj_(&Nx,&Nx,x,f,NULL,&n,"F",&Lambda,1);
	break;

    case 7:
	dfdcfj_(&Nx,&Ny,x,f,NULL,&n,"F",&Reynolds,1);
	break;
    }
}
/*-------------------------------------------------------------*/
void GetJacobian(nlP,neD,x,fjac)
NLCntx    *nlP;
DenseCntx *neD;
double    *x,*fjac;
{
  int    n = neD->n,info = 1,prob = neD->Problem;
  int    nint;
  char   HHProb[6];

switch (prob) {
    case 1:
        nint = n / 8;
	dficfj_(&n,x,NULL,fjac,&n,"J",&Reynolds,&nint,1);
	break;
    case 2:
	nint = n / 14;
	dsfdfj_(&n,x,NULL,fjac,&n,"J",&Viscosity,&nint,1);
	break;

    case 3:
	sprintf( HHProb, "DHHD%d", HHp );
	dhhdfj_(&n,x,NULL,fjac,&n,"J",HHProb,1,5 );
	break;

    case 4:
	dcprfj_(&n,x,NULL,fjac,&n,"J",1);
	break;

    case 5:
	dcpffj_(&n,x,NULL,fjac,&n,"J",1);
	break;

    case 6:
	dsfifj_(&Nx,&Nx,x,NULL,fjac,&n,"J",&Lambda,1);
	break;

    case 7:
	dfdcfj_(&Nx,&Ny,x,NULL,fjac,&n,"J",&Reynolds,1);
	break;
    }
}
/*-------------------------------------------------------------*/
/*
      Forms and factors Jacobian at the point x
*/
void SetUpLinearSolve(nlP,neD,x)
NLCntx     *nlP;
DenseCntx  *neD;
double     *x;
{
  double *A = neD->jacobianFtr;
  double *AA= neD->jacobian;
  int    n = neD->n,info = 2,prob = neD->Problem, one = 1,i,*piv = neD->pivot;
  int    n2 = n * n;
  FILE   *fp = (FILE *) nlP->fp;

  /* get Jacobian */
  GetJacobian(nlP,neD,x,AA);
  if (UseSparse) {
      SpMat *dmat;
      dmat       = SpDnCreateFromData( n, n, n, AA );
      neD->fjac  = SpDnClampToSparse( dmat, 0.0, n );
      SpDestroy( dmat );
      neD->svctx = SVCreate( neD->fjac, SVmethod );
      /* Use the defaults */
      SVSetUp( neD->svctx );
      }
  else {
      for ( i=0; i<n2; i++ )
	  A[i] = AA[i];
      
      /* factor Jacobian */
      dgetf2_(&n,&n,A,&n,piv,&info); if (info < 0) ERR;
      
      /* if factor is singular we use a tacky fix */
      if (info > 0) { 
	  if (fp) fprintf(fp,"Zero Pivot in LU %d\n",info);
	  for ( i=0; i<n; i++ ) {
	      if (A[n*i + i] == 0.0) A[n*i + i] = 1.0;
	      }
	  }
      }
}
/*-------------------------------------------------------------*/
/*             y = -J(x)\f  || y || <= delta
*/
void LinearSolve(nlP,neD,x,f,y,fnorm,delta,rtol,gpnorm,ynorm)
NLCntx    *nlP;
DenseCntx *neD;
double    *f,*x,*y,fnorm,*gpnorm,delta,*ynorm,rtol;
{
  double *A = neD->jacobianFtr,norm;
  int    n = neD->n,info,prob = neD->Problem, one = 1, *piv = neD->pivot;
  FILE   *fp = (FILE *) nlP->fp;

  /* determine y by solving linear system */
  if (UseSparse) {
      SVSolve( neD->svctx, f, y );
      }
  else {
      VCOPY(nlP->vc,neD,f,y);
      dgetrs_("N",&n,&one,A,&n,piv,y,&n,&info);
      }
  /* scale step so its length is less then delta */
  VNORM(nlP->vc,neD,y,&norm);
  if (norm > delta) {
    norm = delta/norm;
    *gpnorm = (1.0 - norm)*fnorm;
    VSCALE(nlP->vc,neD,-norm,y);
    if (fp) fprintf(fp,"Scaling direction by %g \n",norm);
    *ynorm = delta;
  }
  else {
    *gpnorm = 0.0;
    VSCALE(nlP->vc,neD,-1.0,y);
    if (fp) fprintf(fp,"Direction is in Trust Region \n");
    *ynorm = norm;
  }
}
/*-------------------------------------------------------------*/
void LinearDestroy(nlP,neD)
NLCntx     *nlP;
DenseCntx  *neD;
{
if (UseSparse) {
    SVDestroy( neD->svctx );
    SpDestroy( neD->fjac );
    }
}

/*-------------------------------------------------------------*/
void InitialGuess(nlP,neD,x)
NLCntx    *nlP;
DenseCntx *neD;
double    *x;
{
  int    n = neD->n,info = -1,prob = neD->Problem, i;
  int    nint;
  char   HHProb[6];

switch (prob) {
    case 1:
        nint = n / 8;
	dficfj_(&n,x,NULL,NULL,&n,"XS",&Reynolds,&nint,2);
	break;
    case 2:
	nint = n / 14;
	dsfdfj_(&n,x,NULL,NULL,&n,"XS",&Viscosity,&nint,2);
	break;

    case 3:
	sprintf( HHProb, "DHHD%d", HHp );
	dhhdfj_(&n,x,NULL,NULL,&n,"XS",HHProb,2,5 );
	break;

    case 4:
	dcprfj_(&n,x,NULL,NULL,&n,"XS",2);
	break;

    case 5:
	dcpffj_(&n,x,NULL,NULL,&n,"XS",2);
	break;

    case 6:
	dsfifj_(&Nx,&Nx,x,NULL,NULL,&n,"XS",&Lambda,2);
	break;

    case 7:
	dfdcfj_(&Nx,&Ny,x,NULL,NULL,&n,"XS",&Reynolds,2);
	break;
    }
}
/*-------------------------------------------------------------*/
int PrintVector(n,x)
int    *n;
double *x;
{
  int i;
  for ( i=0; i<*n; i++ ) fprintf(stderr,"%d %g \n",i,x[i]);
}
/*-------------------------------------------------------------*/
static int N[5] = {8,11,5,80,140};
DenseCntx *CreateDense(problem,n)
int       problem;
int       n;
{
  DenseCntx *nedense;
  /* n                 = N[problem-1]; */
  nedense           = NEW(DenseCntx);       if (!nedense) ERR;
  nedense->n        = n;
  nedense->jacobian = (double *) MALLOC(n*n*sizeof(double)); 
  if (!nedense->jacobian) ERR;
  if (!UseSparse) {
      nedense->jacobianFtr = (double *) MALLOC(n*n*sizeof(double)); 
      if (!nedense->jacobianFtr) ERR;
      nedense->pivot    = (int *) MALLOC(n*sizeof(int)); 
      if (!nedense->pivot) ERR;
      }
  nedense->Problem  = problem;
  return nedense;
}
/*-------------------------------------------------------------*/
void DestroyDense(nlP,nedense)
NLCntx    *nlP;
DenseCntx *nedense;
{
  FREE(nedense->jacobian);
  FREE(nedense->jacobianFtr);
  FREE(nedense->pivot);
  FREE(nedense);
}

/* For the dogleg routine */
void GetJ( nlP, neD, x )
NLCntx    *nlP;
DenseCntx *neD;
double    *x;
{
SetUpLinearSolve(nlP,neD,x);
}

/* If this routine is not provided, the identity will be used */
void ScaleJ( nlP, neD, d )
NLCntx    *nlP;
DenseCntx *neD;
double    *d;
{
double *A = neD->jacobian, *a, sum;
int     n = neD->n, one = 1, i, j;

a = A;
for (i=0; i<n; i++) {
    sum = 0.0;
    for (j=0; j<n; j++) 
	sum += a[j]*a[j];
    a += n;
    d[i] = sqrt(sum);
    }
}

/* g <- J' f */
void JTv( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobian;
int     n = neD->n, one = 1;
double  done = 1.0, dzero = 0.0; 

if (UseSparse) {
    SpRMultTrans( neD->fjac, f, g );
    }
else {
    dgemv_( "T", &n, &n, &done, A, &n, f, &one, &dzero, g, &one, 1 );
    }
}

/* g <- J f */
void Jv( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobian;
int     n = neD->n, one = 1;
double  done = 1.0, dzero = 0.0; 

if (UseSparse) {
    SpMult( neD->fjac, f, g );
    }
else {
    dgemv_( "N", &n, &n, &done, A, &n, f, &one, &dzero, g, &one, 1 );
    }
}

/* g <- J \ f */
void JSolve( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobianFtr;
int     n = neD->n;
int     i, one = 1, info, *piv = neD->pivot;

if (UseSparse) {
    SVSolve( neD->svctx, f, g );
    }
else {
    for (i=0; i<n; i++) 
	g[i] = f[i];
    
    dgetrs_("N", &n, &one, A, &n, piv, g, &n, &info, 1);
    }
}

#ifdef USEX
void MeshMonitor( nlP, neD, x, f, fnorm )
NLCntx    *nlP;
DenseCntx *neD;
double    *x, *f, fnorm;
{
XBQContourYdirection( 1 );
XBQContour( x, (double *)0, (double *)0, Nx, Ny, 32 );
}
#endif
