#ifndef lint
static char SCCSid[] = "@(#) ./iter/tfqmr/tfqmr.c 07/23/93";
#endif

/*                       
       This implements TFQMR
*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/itpriv.h"

/*+
ITTFQMRCreate - Creates an ITCntx variable for the TFQMR method.
The prefered calling sequence is ITCreate(ITBTFQMR);
+*/
ITCntx *ITTFQMRCreate()
{
ITCntx    *itP;

itP = NEW(ITCntx);  CHKPTRV(itP,0);
ITSetDefaults( itP );
itP->MethodPrivate = (void *) 0;

itP->method               = ITTFQMR;
itP->max_it               = 50;
itP->right_inv            = 0;
itP->calc_res             = 1;
itP->use_pres             = 0;
itP->guess_zero           = 0;
itP->rtol                 = 1.e-5;
itP->atol                 = 1.e-50;
itP->usr_monitor          = ITDefaultMonitor;
itP->residual_history     = NULL;

itP->setup                = ITTFQMRSetUp;
itP->solver               = ITTFQMRSolve;
itP->adjustwork           = ITDefaultAdjustWork;
itP->closedown            = ITDefaultDestroy;
return(itP);
}

/*+
ITTFQMRSetUp - Called after a call to ITTFQMRCreate() or ITCREATE(ITTFQMR)
allocates space needed in the conjugate gradient solution. Preferred
calling sequence is ITSetUp(itP,usrP).

Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
+*/
void ITTFQMRSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
if (itP->method != ITTFQMR) {
   SETERRC(1,"Attempt to use TFQMR Setup on wrong context"); return;}

/* check user parameters and functions */
if (ITCheckDef( itP )) return;

/* get work vectors from user code */
ITDefaultGetWork( itP, usrP, 10 );
}

/*+
ITTFQMRSolve - Called after a call to ITTFQMRCreate() or ITCreate(ITTFQMR)
and the call to ITTFQMRSetUp() or ITSetUp().
Actually solves the linear system using the TFQMR
method. Prefered calling sequence ITSolve(itP,usrP).

Input Parameters: 
.   itP  - the iterative context
.   usrP - the user context

Returns:
the number of iterations required or -1 on error.
+*/
int  ITTFQMRSolve(itP, usrP)
void   *usrP;
ITCntx *itP;
{
int       i = 0, maxit, res, pres, m, conv, hist_len, cerr;
double    rho, rhoold, a, s, b, *history, dp, dpold, dpest, cm, eta, tau,
          etaold, psiold, w, psi, cf;
void      *X,*B,*V,*P,*R,*RP,*T,*T1,*Q,*U, *D, *BINVF, *AUQ;

res     = itP->calc_res;
pres    = itP->use_pres;
maxit   = itP->max_it;
history = itP->residual_history;
hist_len= itP->res_hist_size;
X       = itP->vec_sol;
B       = itP->vec_rhs;
R       = itP->work[0];
RP      = itP->work[1];
V       = itP->work[2];
T       = itP->work[3];
Q       = itP->work[4];
P       = itP->work[5];
BINVF   = itP->work[6];
U       = itP->work[7];
D       = itP->work[8];
T1      = itP->work[9];
AUQ     = V;

/* Compute initial preconditioned residual */
ITResidual(usrP,itP,X,V,T, R, BINVF, B );

/* Test for nothing to do */
NORM(R,&dp);
if (CONVERGED(dp,0)) return RCONV(0);
MONITOR(dp,0);

/* Make the initial Rp == R */
COPY(R,RP);

/* Set the initial conditions */
etaold = 0.0;
psiold = 0.0;
tau    = dp;
dpold  = dp;

DOT(RP,R,&rhoold);
COPY(R,U);
COPY(R,P);
MATOP(P,V,T);
SET(0.0,D);

for (i=0; i<maxit; i++) {
    DOT(RP,V,&s);                          /* s <- rp' v          */
    a = rhoold / s;                        /* a <- rho / s        */
    DWAXPY(-a,V,U,Q);                      /* q <- u - a v        */
    DWAXPY(1.0,U,Q,T);                     /* t <- u + q          */
    MATOP(T,AUQ,T1);
    DAXPY(-a,AUQ,R);                       /* r <- r - a K (u + q) */
    NORM(R,&dp);
    for (m=0; m<2; m++) {
	if (m == 0)
	    w = sqrt(dp*dpold);
	else 
	    w = dp;
	psi = w / tau;
	cm  = 1.0 / sqrt( 1.0 + psi * psi );
	tau = tau * psi * cm;
	eta = cm * cm * a;
	cf  = psiold * psiold * etaold / a;
	if (m == 0) {
	    DAYPX(cf,U,D);
	    }
	else {
	    DAYPX(cf,Q,D);
	    }
	DAXPY(eta,D,X);

	dpest = sqrt(m + 1.0) * tau;
	if (history && hist_len > i + 1) history[i+1] = dpest;
	MONITOR(dpest,i+1);
	if (conv = CONVERGED(dpest,i+1)) break;

	etaold = eta;
	psiold = psi;
	}
    if (conv) break;

    DOT(RP,R,&rho);                        /* newrho <- rp' r       */
    b = rho / rhoold;                      /* b <- rho / rhoold     */
    DWAXPY(b,Q,R,U);                       /* u <- r + b q          */
    DAXPY(b,P,Q);                          
    DWAXPY(b,Q,U,P);                       /* p <- u + b(q + b p)   */
    MATOP(P,V,Q);                          /* v <- K p              */

    rhoold = rho;
    dpold  = dp;
    }
if (i == maxit) i--;
if (history) itP->res_act_size = (hist_len < i + 1) ? hist_len : i + 1;

/* Update computational work */
itP->nmatop   += 2*(i+1);
itP->nvectors += (i+1)*26;

ITUnwindPre( usrP, itP, X, T );
return RCONV(i+1);
}
