#ifndef lint
static char SCCSid[] = "@(#) ./iter/cgs/cgs.c 07/23/93";
#endif

/*                       
       This implements CGS
*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/itpriv.h"

/*+
ITCGSCreate - Creates an ITCntx variable for the CGS method.
The prefered calling sequence is ITCreate(ITCGS);
+*/
ITCntx *ITCGSCreate()
{
ITCntx    *itP;

itP = NEW(ITCntx);  CHKPTRV(itP,0);
ITSetDefaults( itP );
itP->MethodPrivate = (void *) 0;

itP->method               = ITCGS;
itP->max_it               = 50;
itP->right_inv            = 0;
itP->calc_res             = 1;
itP->use_pres             = 0;
itP->guess_zero           = 0;
itP->rtol                 = 1.e-5;
itP->atol                 = 1.e-50;
itP->usr_monitor          = ITDefaultMonitor;
itP->residual_history     = NULL;

itP->setup                = ITCGSSetUp;
itP->solver               = ITCGSSolve;
itP->adjustwork           = ITDefaultAdjustWork;
itP->closedown            = ITDefaultDestroy;
return(itP);
}

/*+
ITCGSSetUp - Called after a call to ITCGSCreate() or ITCREATE(ITCGS)
allocates space needed in the conjugate gradient solution. Preferred
calling sequence is ITSetUp(itP,usrP).

Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
+*/
void ITCGSSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
if (itP->method != ITCGS) {
   SETERRC(1,"Attempt to use CGS Setup on wrong context"); return;}

/* check user parameters and functions */
if (ITCheckDef( itP )) return;

/* get work vectors from user code */
ITDefaultGetWork( itP, usrP, 8 );
}

/*+
ITCGSSolve - Called after a call to ITCGSCreate() or ITCreate(ITCGS)
and the call to ITCGSSetUp() or ITSetUp().
Actually solves the linear system using the CGS
method. Prefered calling sequence ITSolve(itP,usrP).

Input Parameters: 
.   itP  - the iterative context
.   usrP - the user context

Returns:
the number of iterations required or -1 on error.
+*/
int  ITCGSSolve(itP, usrP)
void   *usrP;
ITCntx *itP;
{
int       i = 0, maxit, res, pres, hist_len, cerr;
double    rho, rhoold, a, s, b, *history, dp; 
void      *X,*B,*V,*P,*R,*RP,*T,*Q,*U, *BINVF, *AUQ;

res     = itP->calc_res;
pres    = itP->use_pres;
maxit   = itP->max_it;
history = itP->residual_history;
hist_len= itP->res_hist_size;
X       = itP->vec_sol;
B       = itP->vec_rhs;
R       = itP->work[0];
RP      = itP->work[1];
V       = itP->work[2];
T       = itP->work[3];
Q       = itP->work[4];
P       = itP->work[5];
BINVF   = itP->work[6];
U       = itP->work[7];
AUQ     = V;

/* Compute initial preconditioned residual */
ITResidual(usrP,itP,X,V,T, R, BINVF, B );

/* Test for nothing to do */
NORM(R,&dp);
if (CONVERGED(dp,0)) return RCONV(0);
MONITOR(dp,0);
if (history) history[0] = dp;

/* Make the initial Rp == R */
COPY(R,RP);

/* Set the initial conditions */
DOT(RP,R,&rhoold);
COPY(R,U);
COPY(R,P);
MATOP(P,V,T);

for (i=0; i<maxit; i++) {
    DOT(RP,V,&s);                          /* s <- rp' v          */
    a = rhoold / s;                        /* a <- rho / s        */
    DWAXPY(-a,V,U,Q);                      /* q <- u - a v        */
    DWAXPY(1.0,U,Q,T);                     /* t <- u + q          */
    DAXPY(a,T,X);                          /* x <- x + a (u + q)  */
    MATOP(T,AUQ,U);
    DAXPY(-a,AUQ,R);                       /* r <- r - a K (u + q) */
    NORM(R,&dp);

    if (history && hist_len > i + 1) history[i+1] = dp;
    MONITOR(dp,i+1);
    if (CONVERGED(dp,i+1)) break;

    DOT(RP,R,&rho);                        /* newrho <- rp' r       */
    b = rho / rhoold;                      /* b <- rho / rhoold     */
    DWAXPY(b,Q,R,U);                       /* u <- r + b q          */
    DAXPY(b,P,Q);
    DWAXPY(b,Q,U,P);                       /* p <- u + b(q + b p)   */
    MATOP(P,V,Q);                          /* v <- K p              */
    rhoold = rho;
    }
if (i == maxit) i--;
if (history) itP->res_act_size = (hist_len < i + 1) ? hist_len : i + 1;

ITUnwindPre( usrP, itP, X, T );
return RCONV(i+1);
}
