#ifndef LINT
static char SCCSid[] = "@(#) ./blkcm/compile/valid.c 07/23/93";
#endif

/*
   Helper routines for using the mesh communication package.
 */
#include "tools.h"
#include "blkcm/bcp.h"
#include "blkcm/bc.h"
#include <stdio.h>

#ifndef DBUG
#define DBUG(a)
#endif

/*
   There should be a routine to attempt to determine if a program is
   valid (types, message sizes line up).  Returns 1 if true, 0 if false.
   If pflag is true, diagnostics will be printed.

   Here are some possible tests:
   In a vector of length p, each processor adds in the number of messages
   that it is sending to the j'th processor.  Then,
   Each processor sends a list of types, phases, and sizes to each destination.
   The processors know when they are done when they have received enough
   messages.  Each processor then checks this list of sends against its own
   list of receives for type match and size match.  In addition, they may
   use the phase information to look for deadlock.  If there are leftover
   messages, that should be reported.

   For local entries, there is nothing to check.

   As part of this check, the routines could ascertain the owner that goes
   with a particular id, based on the RECV holding the id.  Since this
   service requires 90% of the code for full validation checking, the
   novalid option in this case just turns off small amounts of the code.

   Needs to check that id's are unique within each phase.
 */

#ifdef DISTRIBUTED_MEMORY
/* return 1 if we find the matching message, 0 otherwise */
typedef struct {
    int mtype, phase, id, processor, type;
    } msginfo;

int BCget_nrecv();

static int BCmatch( pgm, n, msg )
BCentry *pgm;
int        n;
msginfo    *msg;
{
while (n--) {
    if (pgm->phase == msg->phase &&
        pgm->id    == msg->id && pgm->processor == msg->processor &&
	pgm->mtype == msg->mtype &&
	(pgm->type & BLOCK_COMM_BUFFER) && IS_DEST(pgm)) {
	/* make sure that pairsync is set the same on each */
	if (pgm->type & BLOCK_COMM_SYNC_NBR) {
	    if (msg->type & BLOCK_COMM_SYNC_NBR) return 1;
	    else {
		fprintf( stderr, "Pair-sync types don't match on\n" );
		return 0;
		}
	    }
	else return 1;
	}
    NEXTLINE(pgm);
    }
return 0;
}

int BCvalid_pgm( Program, pflag )
BCPGM *Program;
int   pflag;
{
int        err = 0;

BCentry    *pgm = Program->pgm, *tpgm;
int        n    = Program->n;
int        nproc= PInumtids;
int        *nin, *nout, i, j, *work, nintot, *rc, dummy;
int        myid = PImytid;
msginfo    msg, *buf;

TRPUSH(BCTRID+7);

/* Should check that the program contains valid values */
err = BCCheckTypes( Program );
GISUM( &err, 1, &dummy, 0 );
if (err) { TRPOP; return err; }

/* First, find the number of messages being send to each processor */
nin    = (int *)MALLOC( nproc * sizeof(int) );
nout   = (int *)MALLOC( nproc * sizeof(int) );
work   = (int *)MALLOC( nproc * sizeof(int) );
if (!nin || !nout || !work) { TRPOP; return 1; }

for (i=0; i<nproc; i++) {
    nin[i] = 0;
    nout[i]= 0;
    }
nintot = 0;
for (i=0; i<n; i++) {
    if (pgm[i].type & BLOCK_COMM_BUFFER) {
	if (IS_SRC(pgm+i))
	    nout[pgm[i].processor]++;
	else if (IS_DEST(pgm+i)) {
	    nin[pgm[i].processor]++;
	    nintot++;
	    }
        }
    }
/* Get the total number of messages being sent to all processors */
GISUM( nout, nproc, work, 0 );
DBUG( "[valid] Got number of messages being sent\n" );

/* Check that the number that we are receiving is the number being sent */
err = 0;
if (nout[myid] != nintot) 
    err++;

/* See if everyone is happy so far */
GISUM( &err, 1, work, 0 );
DBUG( "[valid] Checked error value\n" );

if (err) {
    fprintf( stderr, 
	    "[%d] Number of errors in program is %d\n", myid, err );
    if (pflag) {
	for (j=0; j<=nproc; j++) { 
	    if (GTOKEN(0,j)) {
		fprintf( stderr, "[%d] receiving %d, expecting %d\n",
			myid, nout[myid], nintot );
		for (i=0; i<nproc; i++) {
		    if (nin[i] > 0) 
			fprintf( stderr, 
				 "[%d] expecting %d messages from processor %d\n", 
				 myid, nin[i], i );
		    }
		/* Also print out the data we'd like to send */
		n = Program->n;
		for (i=0; i<n; i++) {
		    if (pgm[i].type & BLOCK_COMM_BUFFER) {
			if (IS_SRC(pgm+i))
			    fprintf( stderr, 
			     "[%d] Sending id %d to %d\n", myid, pgm[i].id ,pgm[i].processor);
			else if (IS_DEST(pgm+i)) 
			    fprintf( stderr, 
			     "[%d] Receiving id %d from %d \n", myid, pgm[i].id,
pgm[i].processor );
			}
		    }
		}
	    }
	}
    }

FREE( nin );
FREE( nout );
if (err != 0) {
    TRPOP;
    return 1;
    }

/* 
   Send data to each processor about the type/phase/id that it will be sending.
   This communication will not deadlock since each processor knows how many
   messages that it is expecting.  By using irecv and local space,
   we know that the sends won't block on intel machines
 */
DBUG( "[valid] about to check type/phase/id's\n" );
/* Some mallocs will return null for 0 bytes requested */
if (nintot > 0) {
    rc = (int *)MALLOC( nintot * sizeof(int) );
    buf= (msginfo *)MALLOC( nintot * sizeof(msginfo) );
    if (!rc || !buf) { TRPOP; return 1; }
    }
else {
    rc  = 0;
    buf = 0;
    }
j  = 0;
n  = Program->n;
pgm= Program->pgm;
#ifndef PI_NO_NRECV
while (n--) {
    if (IS_DEST(pgm) && (pgm->type & BLOCK_COMM_BUFFER)) {
	if (j >= nintot) {
	    if (pflag) {
		fprintf( stderr, 
			"[%d] Mismatch of number of expected messages\n", 
			myid );
		}
	    break;
	    }
	RECVASYNCNOMEM(VALID_TYPE,buf+j,sizeof(msginfo),MSG_INT,rc[j]);
	j++;
        }
    NEXTLINE(pgm);
    }
if (j != nintot) {
    if (pflag) {
	fprintf( stderr,
	"[%d] Mismatch of number of expected messages %d found, %d expected\n",
		myid, j, nintot );
	}
    err++;
    }
#endif
n  = Program->n;
pgm= Program->pgm;
while (n--) {
    if (IS_SRC(pgm) && (pgm->type & BLOCK_COMM_BUFFER)) {
        msg.mtype      = pgm->mtype;
        msg.phase      = pgm->phase;
        msg.id         = pgm->id;
        msg.processor  = myid;
	msg.type       = pgm->type;
        PIbsend( VALID_TYPE, &msg, sizeof(msginfo), 
		       pgm->processor, MSG_INT );
        }
    NEXTLINE(pgm);
    }
DBUG( "[valid] Sent the types/phases/ids\n" );

/* Receive messages and check them */
n    = Program->n;
pgm  = Program->pgm;
tpgm = pgm;
for (i=0; i<nintot; i++) {
#ifdef PI_NO_NRECV
    /* Skip to the next program line that is a receive */
    while (tpgm && (!IS_DEST(tpgm) || !(tpgm->type & BLOCK_COMM_BUFFER))) {
	NEXTLINE(tpgm);
	}
    if (tpgm) {
	PIbrecv(VALID_TYPE,buf+i,sizeof(msginfo),MSG_INT);
#else
    if (rc[i] > 0) {
	PIwrecv(VALID_TYPE,buf+i,sizeof(msginfo),MSG_INT,rc[i]);
#endif
	/* Check this message to make sure we expect it */
	if (!BCmatch( pgm, n, buf+i )) {
	    if (pflag) {
		fprintf( stderr,
	"[%d] Unmatched messages: id %d (phase %d) from proc %d unexpected\n",
			myid, buf[i].id, buf[i].phase, buf[i].processor );
		}
	    err++;
	    }
	}
    }
DBUG( "[valid] received the types/phases/ids\n" );

GISUM( &err, 1, work, 0 );
if (err != 0)
    fprintf( stderr, "[%d] Number of errors in program is %d\n", myid, err );

/* Free the temporary storage */
FREE( work );
if (rc != 0) {
    FREE( rc );
    FREE( buf );
    }

/* Check on local part of program */
err += BCvalid_local_pgmPart( Program, pflag, 0 );

TRPOP;
return err;
}

/* 
   Return the number of msgs that this processor can expect.  Return
   -1 on an error.

   If bflag is true, use only lines with the BUFFER bit.

   Algorithm:
   Each processor determines how many messages are going to EACH 
   processor.  That value is then globally summed; a processor then
   looks in its own entry for the value.
 */
int BCget_nrecv( Program, bflag )
BCPGM *Program;
{
int        *nout, *work, n, nproc = PInumtids, i, mypid = PImytid;
BCentry *pgm;

TRPUSH(BCTRID+8);
/* First, find the number of messages being send to each processor */
nout   = (int *)MALLOC( 2*nproc * sizeof(int) );
if (!nout) return -1;
work   = nout + nproc;
for (i=0; i<nproc; i++) {
    nout[i]= 0;
    }
n   = Program->n;
pgm = Program->pgm;
while (n--) {
    if (IS_SRC(pgm) && pgm->processor != mypid && pgm->processor >= 0 &&
	(!bflag || pgm->type & BLOCK_COMM_BUFFER)) {
	nout[pgm->processor]++;
        }
    NEXTLINE(pgm);
    }

/* Get the total number of messages being sent to all processors */
GISUM( nout, nproc, work, 0 );
n    = nout[PImytid];
FREE( nout );
TRPOP;
return n;
}

#endif

int BCvalid_local_pgm( Program, pflag )
BCPGM *Program;
int   pflag;
{
return BCvalid_local_pgmPart( Program, pflag, 1 );
}

/* 
   This routine checks for valid local programs (all dests have a source,
   sizes match up).
 */
int BCvalid_local_pgmPart( Program, pflag, all_local )
BCPGM *Program;
int   pflag, all_local;
{
int        err = 0;

BCentry *pgm = Program->pgm, *pgmd;
int        n    = Program->n;

while (n--) {
    if (GET_MAJOR_MODE(pgm) == BLOCK_COMM_LOCAL_SRC) {
	pgmd = pgm + pgm->processor;
	if (GET_MAJOR_MODE(pgmd) != BLOCK_COMM_LOCAL_DEST) {
	    err++;
	    if (pflag)
		fprintf( stderr, "Local source does not have destination\n" );
	    }
	if (pgm->src.n1 != pgmd->src.n1 ||
	    pgm->src.n2 != pgmd->src.n2 ||
	    pgm->src.n3 != pgmd->src.n3 ||
	    pgm->src.n4 != pgmd->src.n4 ||
	    pgm->src.n5 != pgmd->src.n5) {
	    err++;
	    if (pflag) {
		fprintf( stderr, "Length of source and dest do not match\n" );
		fprintf( stderr, "<src> :%d %d %d %d %d\n", pgm->src.n1, 
		      pgm->src.n2, pgm->src.n3, pgm->src.n4, pgm->src.n5 );
		fprintf( stderr, "<dest>:%d %d %d %d %d\n", pgmd->src.n1, 
		      pgmd->src.n2, pgmd->src.n3, pgmd->src.n4, pgmd->src.n5 );
		}
	    }
	}
    if (all_local && (!IS_LOCAL_SRC(pgm) && !IS_LOCAL_DEST(pgm))) {
	fprintf( stderr, "Program contains unmatched local references\n" );
	err++;
	}
    NEXTLINE(pgm);
    }
return err;
}

#ifdef PARALLEL_VERSION
/*
   This routine checks that the RECV types are unique within a program.
   It does this by taking all of the remote receives, remembering their
   types, an sorting on them.  Any duplicates may then be found and
   error messages issued.
 */
#ifdef OLD_SORT
static int SortInt( a, b )
int *a, *b;
{
return (*a-*b);
}
#endif
int BCCheckTypes( Program )
BCPGM *Program;
{
BCentry *pgm;
int     *types, n, i, nrecv, err;

pgm   = Program->pgm;
n     = Program->n;
types = (int *)MALLOC( n * sizeof(int) );
nrecv = 0;
err   = 0;
for (i=0; i<n; i++) {
    if (GET_MAJOR_MODE(pgm) == BLOCK_COMM_DEST && 
	(pgm->type & BLOCK_COMM_BUFFER)) 
	types[nrecv++] = pgm->mtype;
    NEXTLINE(pgm);
    }
if (nrecv == 0) return 0;

#ifdef OLD_SORT
qsort( (char *)types, nrecv, sizeof(int), SortInt );
#else
SYIsort( nrecv, types );
#endif

for (i=1; i<nrecv; i++) 
    if (types[i-1] == types[i]) {
	fprintf( stderr, "Duplicate message types (%d) for processor %d\n",
		 types[i], PImytid );
	err++;
	}
FREE( types );
return err;
}
#endif
