/*--------------------------------------------------------------------------
 Stack operators for RPN calculator which manipulates complex vectors 
 of short integers.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include <memory.h>

#include "sig.h"
#include "stack.h"

static char sccsid[] = "@(#)stack.c	1.2 7/18/91";

void
stack_show(this)
    stack_t *this;
{
    char buf[100];
    int i;
    printf("%d elements in stack.\n", this->n);
    for (i=0; i<this->n; i++) {
	hdr_kind2a(this->sigs[i]->kind, buf);
	printf("stack(%d) (nrec=%d, reclen=%d, kind=%s): ", 
	    i, this->sigs[i]->nrec, this->sigs[i]->reclen, buf);
	sig_fprint(this->sigs[i], 1, 5, stdout);
    }
}

/*--------------------------------------------------------------------------
 Push the given signal onto the stack.

 To push a copy of a signal, use sig_copy.
 To push an empty signal, use sig_new.
--------------------------------------------------------------------------*/
void
stack_push(this, sig)
    stack_t *this;
    sig_t *sig;
{
    this->n++;
    if (this->n >= STACK_MAX) {
	fprintf(stderr, "stack_push: stack overflow\n");
	exit(1);
    }
    /* Push stack down.  Overlapping, CANNOT use memcpy() */
    bcopy(this->sigs, this->sigs+1, (this->n-1) * sizeof(sig_t *));

    /* Place new element at top */
    this->sigs[0] = sig;
}

/*--------------------------------------------------------------------------
 Pop and return stack element i (0=TOS) from the stack.
--------------------------------------------------------------------------*/
sig_t *
stack_pop(this, i)
    stack_t *this;
    int i;
{
    sig_t *sig;

    this->n--;
    if (this->n < i) {
	fprintf(stderr, "stack_pop: stack underflow\n");
	exit(1);
    }
    sig = this->sigs[i];

    /* Pop stack up.  Overlapping, can NOT use memcpy() */
    bcopy(this->sigs+i+1, this->sigs+i, (this->n-i) * sizeof(sig_t *));

    return sig;
}

/*--------------------------------------------------------------------------
 Return pointer to stack element i (0=TOS) from the stack.
 Element remains on stack.
--------------------------------------------------------------------------*/
sig_t *
stack_peek(this, i)
    stack_t *this;
    int i;
{
    if (this->n <= i) {
	fprintf(stderr, "stack_peek: stack underflow\n");
	exit(1);
    }
    return this->sigs[i];
}

/*--------------------------------------------------------------------------
 Exchange element i with the top of stack, where 0 is top of stack.
--------------------------------------------------------------------------*/
void
stack_xchg(this, i)
    stack_t *this;
    int i;
{
    sig_t *tmp;

    if (i >= this->n) {
	fprintf(stderr, "stack_xchg: stack underflow\n");
	exit(1);
    }

    tmp = this->sigs[0];
    this->sigs[0] = this->sigs[i];
    this->sigs[i] = tmp;
}

/*--------------------------------------------------------------------------
 Join the N signals at the top of the stack.
 All signals must have same type and reclen.
--------------------------------------------------------------------------*/
void
stack_cat_y(this, n)
    stack_t *this;
    int n;
{
    sig_t *tos;
    sig_t *sig;
    int i;
    int tmp_nrec;
    int recbytes;

    tos = stack_peek(this, 0);
    recbytes = tos->reclen * sig_ELBYTES(tos);

    /* Make sure all arguments have same type and reclen as TOS,
     * total up the size of the final signal.
     */
    tmp_nrec = tos->nrec;
    for (i=1; i<n; i++) {
	sig = stack_peek(this, i);
	if (tos->kind != sig->kind) {
	    char tos_akind[100];
	    char sig_akind[100];

	    hdr_kind2a(tos->kind, tos_akind);
	    hdr_kind2a(sig->kind, sig_akind);
	    fprintf(stderr, "stack_cat_y: Stack[%d].kind(%s) != stack[0].kind(%s)\n",
		i, sig_akind, tos_akind);
	    exit(1);
	}
	if (tos->reclen != sig->reclen) {
	    fprintf(stderr, "stack_cat_y: Stack[%d].reclen(%d) != stack[0].reclen(%d)\n",
		i, sig->reclen, tos->reclen);
	    exit(1);
	}
	tmp_nrec += sig->nrec;
    }

    /* Resize tos. */
    tos->re = (short *)realloc(tos->re, tmp_nrec * recbytes);
    if (tos->kind & hdr_KIND_COMPLEX)
	tos->im = (short *)realloc(tos->im, tmp_nrec * recbytes);
    if (tos->kind & hdr_KIND_BFP)
	tos->exp = (short *)realloc(tos->exp, tmp_nrec * sizeof(short));

    /* Pop off stack contents, copy them in, destroy them. */
    tmp_nrec = tos->nrec;
    for (i=1; i<n; i++) {
	sig = stack_pop(this, 1);

	memcpy(((char *)tos->re) + tmp_nrec * recbytes,
		sig->re,
		sig->nrec * recbytes);
	if (tos->kind & hdr_KIND_COMPLEX)
	    memcpy(((char *)tos->im) + tmp_nrec * recbytes, 
		    sig->im,
		    sig->nrec * recbytes);
	if (tos->kind & hdr_KIND_BFP)
	    memcpy(((char *)tos->exp) + tmp_nrec * sizeof(short),
		    sig->exp, 
		    sig->nrec * sizeof(short));

	tmp_nrec += sig->nrec;
	sig_destroy(sig);
    }
    tos->nrec = tmp_nrec;
}

/*--------------------------------------------------------------------------
 Split TOS into N separate signals with equal nrec (and possible extra
 runt at end)
--------------------------------------------------------------------------*/
void
stack_split_y(this, nsig)
    stack_t *this;
    int nsig;
{
    sig_t *tos;
    sig_t *sig;
    int i;
    int sig_nrec;
    int recbytes;
    int nrec;

    assert(nsig > 0);

    tos = stack_pop(this, 0);
    recbytes = tos->reclen * sig_ELBYTES(tos);

    /* Get number of records in each signal */
    sig_nrec = tos->nrec / nsig;

    if (tos->nrec % nsig) nsig++;

    for (i=nsig, nrec=tos->nrec; --i>=0; nrec-=sig_nrec) {
	char buf[256];
	int nr;

	nr = nrec;
	if (nr > sig_nrec) nr = sig_nrec;

	sprintf(buf, "%s part %d", tos->fname, i);
	sig = sig_new(nr, tos->reclen, tos->kind, buf);
	memcpy((char *)sig->re, ((char *)tos->re) + i*recbytes,
	    nr*recbytes);
	if (tos->kind & hdr_KIND_COMPLEX)
	    memcpy((char *)sig->im, ((char *)tos->im) + i*recbytes,
		nr*recbytes);
	if (tos->kind & hdr_KIND_BFP)
	    memcpy((char *)sig->exp, ((char *)tos->exp) + i*sizeof(short),
		nr*sizeof(short));
	stack_push(this, sig);
    }
    sig_destroy(tos);
}
