#include <stdio.h>
#include <malloc.h>
#include <memory.h>

#include "bfp.h"
#include "sig.h"
#include "fft.h"
#include "arith.h"
#include "bitrev.h"

static char sccsid[] = "@(#)sigfft.c	1.4 7/23/91";

/*--------------------------------------------------------------------------
 Find log of given number base 2.
--------------------------------------------------------------------------*/
int
ilog2(x)
    int x;
{
    int log_n;

    for (log_n=14; log_n>0; log_n--)
	if ((1 << log_n) == x)
	    break;
    if (log_n<0) {
	printf("ilog2: error: len %d is not power of two\n", x);
	exit(1);
    }
    return log_n;
}

/*--------------------------------------------------------------------------
 Run array thru cfft_dit.
 Input and output are in normal order.
--------------------------------------------------------------------------*/
void
sig_cfft_dit(this)
    sig_t *this;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int log_n;
    int offset;
    int n;
    char *tmp = malloc(this->reclen * sig_ELBYTES(this));

    bitrev_setup(reclen);

    log_n = ilog2(this->reclen);

    switch (this->kind) {
    case hdr_CINT16:
	bfp_do = 0;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    bitrev_int16(reclen, this->re+offset, tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));

	    cfft_dit(log_n, this->re+offset, this->im+offset);
	}
	break;
    case hdr_CBFP16:
	bfp_do = 1;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    bfp_scale_ct = this->exp[n];
	    bitrev_int16(reclen, this->re+offset, (short *)tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, (short *)tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));

	    cfft_dit(log_n, this->re+offset, this->im+offset);
	    this->exp[n] = bfp_scale_ct;
	}
	bfp_do = 0;
	break;
    default:
	badkind("sig_cfft_dit", this->kind);
    }

    free(tmp);
}

static void
rsmult_cfloat32(n, sr, br, bi)
    int n;
    float sr;
    float *br, *bi;
{
    while (n--) {
	*br++ *= sr;
	*bi++ *= sr;
    }
}

/*--------------------------------------------------------------------------
 Run array thru cfft_dif, but divide by 2 at all i such that
 that.re[i] == 1.
 that may be NULL, in which case the signal is not divided by 2 at any stage.
 Inputs and outputs are in normal order.
--------------------------------------------------------------------------*/
void
sig_cfft_dif(this, that)
    sig_t *this;
    sig_t *that;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int log_n;
    int offset;
    int n;
    short *divide_stages;
    char *tmp = malloc(this->reclen * sig_ELBYTES(this));

    bitrev_setup(reclen);

    log_n = ilog2(this->reclen);

    if (that != NULL) {
	if (that->kind != hdr_INT16)
	    badkind("sig_cfft_dif: expected 2nd arg to be real int16",
		that->kind);
	if (that->nrec != 1 || that->reclen != log_n) {
	    fprintf(stderr, "sig_cfft_dif: 2nd arg has reclen %d, should be log2(reclen of 1st argument=%d\n",
		that->reclen, log_n);
	    exit(1);
	}
	divide_stages = that->re;
    } else
	divide_stages = (short *)NULL;
	

    switch (this->kind) {
    case hdr_CINT16:
	bfp_do = 0;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    cfft_dif(log_n, this->re+offset, this->im+offset, divide_stages);

	    bitrev_int16(reclen, this->re+offset, tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));
	}
	break;
    case hdr_CBFP16:
	bfp_do = 1;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    bfp_scale_ct = this->exp[n];
	    cfft_dif(log_n, this->re+offset, this->im+offset, divide_stages);

	    bitrev_int16(reclen, this->re+offset, tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));

	    this->exp[n] = bfp_scale_ct;
	}
	bfp_do = 0;
	break;
    case hdr_CFLOAT32: {
	float factor;
	/* Simulate per-stage division */
	if (divide_stages != NULL)
	    for (n=0, factor=1.0; n<log_n; n++)
		if (divide_stages[n]) factor /= 2.0;

	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    cfft_dif_float32(log_n, 
		((float *)this->re)+offset, 
		((float *)this->im)+offset);

	    bitrev_float32(reclen, ((float *)this->re)+offset, tmp);
	    memcpy(((float *)this->re)+offset, tmp, reclen*sizeof(float));
	    bitrev_float32(reclen, ((float *)this->im)+offset, tmp);
	    memcpy(((float *)this->im)+offset, tmp, reclen*sizeof(float));

	    if (divide_stages != NULL)
		rsmult_cfloat32(reclen, factor, 
		    ((float*)this->re)+offset, ((float*)this->im)+offset);
	}
	} break;
    default:
	badkind("sig_cfft_dif", this->kind);
    }
    free(tmp);
}

/*--------------------------------------------------------------------------
 Run array thru cifft_dif, but divide by 2 at all i such that
 that.re[i] == 1.
 that may be NULL, in which case the signal is divided by 2 at every stage.
 Inputs and outputs are in normal order.
--------------------------------------------------------------------------*/
void
sig_cifft_dif(this, that)
    sig_t *this;
    sig_t *that;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int log_n;
    int offset;
    int n;
    short *divide_stages;
    char *tmp = malloc(this->reclen * sig_ELBYTES(this));

    bitrev_setup(reclen);

    log_n = ilog2(this->reclen);

    if (that != NULL) {
	if (that->kind != hdr_INT16)
	    badkind("sig_cifft_dif: expected 2nd arg to be real int16",
		that->kind);
	if (that->nrec != 1 || that->reclen != log_n) {
	    fprintf(stderr, "sig_cifft_dif: 2nd arg has reclen %d, should be log2(reclen of 1st argument=%d\n",
		that->reclen, log_n);
	    exit(1);
	}
	divide_stages = that->re;
    } else
	divide_stages = (short *)NULL;
	

    switch (this->kind) {
    case hdr_CINT16:
	bfp_do = 0;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    bitrev_int16(reclen, this->re+offset, tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));

	    cifft_dif(log_n, this->re+offset, this->im+offset, divide_stages);
	}
	break;
    case hdr_CBFP16:
	bfp_do = 1;
	for (n=offset=0; n<nrec; n++, offset += reclen) {
	    bfp_scale_ct = this->exp[n];

	    bitrev_int16(reclen, this->re+offset, tmp);
	    memcpy(this->re+offset, tmp, reclen*sizeof(short));
	    bitrev_int16(reclen, this->im+offset, tmp);
	    memcpy(this->im+offset, tmp, reclen*sizeof(short));

	    cifft_dif(log_n, this->re+offset, this->im+offset, divide_stages);
	    this->exp[n] = bfp_scale_ct;
	}
	bfp_do = 0;
	break;
    case hdr_CFLOAT32: {
	float factor;
	/* Simulate per-stage division */
	if (divide_stages != NULL)
	    for (n=0, factor=1.0; n<log_n; n++)
		if (!divide_stages[n]) factor *= 2.0;

	for (n=offset=0; n<nrec; n++, offset += reclen) {

	    bitrev_float32(reclen, ((float *)this->re)+offset, tmp);
	    memcpy(((float *)this->re)+offset, tmp, reclen*sizeof(float));
	    bitrev_float32(reclen, ((float *)this->im)+offset, tmp);
	    memcpy(((float *)this->im)+offset, tmp, reclen*sizeof(float));

	    cifft_dif_float32(log_n, 
		((float *)this->re)+offset, 
		((float *)this->im)+offset);
	    if (divide_stages != NULL)
		rsmult_cfloat32(reclen, factor, 
		    ((float*)this->re)+offset, 
		    ((float*)this->im)+offset);
	}
	} break;
    default:
	badkind("sig_cifft_dif", this->kind);
    }
    free(tmp);
}

/*--------------------------------------------------------------------------
 Run array thru cfft_direct.
--------------------------------------------------------------------------*/
void
sig_cfft_direct(this)
    sig_t *this;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int recbytes = reclen * sig_ELBYTES(this);
    int log_n;
    int offset;

    char *tr = malloc(recbytes);
    char *ti = malloc(recbytes);

    log_n = ilog2(this->reclen);

    switch (this->kind) {
    case hdr_CINT16:
	for (offset=0; nrec > 0; nrec--, offset += reclen) {
	    cfft_direct_int16(log_n, 
		((short *)this->re)+offset, 
		((short *)this->im)+offset,
		(short *)tr, 
		(short *)ti);
	    bcopy(tr, ((short *)this->re)+offset, recbytes);
	    bcopy(ti, ((short *)this->im)+offset, recbytes);
	}
	break;
    case hdr_CFLOAT32:
	for (offset=0; nrec > 0; nrec--, offset += reclen) {
	    cfft_direct_float32(log_n, 
		((float *)this->re)+offset, 
		((float *)this->im)+offset, 
		(float *)tr, 
		(float *)ti);
	    bcopy(tr, ((float *)this->re)+offset, recbytes);
	    bcopy(ti, ((float *)this->im)+offset, recbytes);
	}
	break;
    default:
	badkind("sig_cfft_direct", this->kind);
    }

    free(tr);
    free(ti);
}

/*--------------------------------------------------------------------------
 Run array thru cifft_direct.
--------------------------------------------------------------------------*/
void
sig_cifft_direct(this)
    sig_t *this;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int recbytes = reclen * sig_ELBYTES(this);
    int log_n;
    int offset;

    char *tr = malloc(recbytes);
    char *ti = malloc(recbytes);

    log_n = ilog2(this->reclen);

    switch (this->kind) {
#if 0
    case hdr_CINT16:
	for (offset=0; nrec > 0; nrec--, offset += reclen) {
	    cifft_direct_int16(log_n, 
		((short *)this->re)+offset, 
		((short *)this->im)+offset,
		(short *)tr, 
		(short *)ti);
	    bcopy(tr, ((short *)this->re)+offset, recbytes);
	    bcopy(ti, ((short *)this->im)+offset, recbytes);
	}
	break;
#endif
    case hdr_CFLOAT32:
	for (offset=0; nrec > 0; nrec--, offset += reclen) {
	    cifft_direct_float32(log_n, 
		((float *)this->re)+offset, 
		((float *)this->im)+offset, 
		(float *)tr, 
		(float *)ti);
	    bcopy(tr, ((float *)this->re)+offset, recbytes);
	    bcopy(ti, ((float *)this->im)+offset, recbytes);
	}
	break;
    default:
	badkind("sig_cifft_direct", this->kind);
    }

    free(tr);
    free(ti);
}
