/*--------------------------------------------------------------------------
 RPN calculator that operates on complex vectors of integers.
 Used to develop implementations of digital signal processing algorithms.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>

#include "sig.h"
#include "imath.h"
#include "macro.h"

static char sccsid[] = "@(#)drpn.c	1.9 7/23/91";

/* To add a new operator, follow these three steps:
 * 1. Add include file
 * 2. Add help message
 * 3. Add call to operator
 */

#include "ac_chirp.h"
#include "arith.h"
#include "bfp.h"
#include "bitrev.h"
#include "convolve.h"
#include "edit.h"
#include "file.h"
#include "hdr.h"
#include "plot.h"
#include "reverse.h"
#include "sigfft.h"
#include "stack.h"
#include "trig.h"
#include "typecast.h"


/* The RPN stack */
static stack_t stack_;
static stack_t *stack = &stack_;

/* State variables */
int verbose=1;

#define BLANKS " \t"
#define STK(n) stack_peek(stack,n)

/*--------------------------------------------------------------------------
 Execute a command.  Return -1 if command unrecognized.
--------------------------------------------------------------------------*/
int
do_cmd(cmd, args)
    char *cmd;
    char *args;
{
    char *tok;
    char garbage[128];
    sig_t *sig, *tos;

    /* State variable */
    static int may_append;		/* Tested when executing append */
    static int next_may_append = 0;	/* Set=1 when executing new, append */

    /* The append command may only be executed after 'new' or 'append'.
     * Notice whether these commands were executed last time.
     */
    may_append = next_may_append;
    next_may_append = 0;

    if (strcmp(cmd, "help")==0) {
	puts("integer dsp rpn calculator.\n\
Note: TOS stands for Top Of Stack, NOS stands for Next to top Of Stack.\n\
Commands:\n\
 help			print this message\n\
I/O operations:\n\
 echo text...		echo given text to output \n\
 verbose n		if n=1, print commands as they are executed\n\
 new			push empty signal onto stack\n\
 append r0 r1...	append real record to TOS.  Only after new or append.\n\
 append r0,i0 r1,i1...	append complex record to TOS\n\
 real n0 n1...		set real part of TOS, push\n\
 imag n0 n1...		set imaginary part of TOS; must follow real.\n\
 name NAME		set name part of TOS; must follow real or imag.\n\
 print [nrec nel]	print [1st nel elements of 1st nrec records of] TOS\n\
 read fname		read TOS from file\n\
 write fname		write TOS to file\n\
 verify			Verify that signal contains valid numbers\n\
Plotting:\n\
 pl_open Y margin cmd	Prepare to plot.  Y is sum of dY's for all plot_r cmds.\n\
			To preview, use Y 0 xplot; to print, use Y 1 lpr -g.\n\
 pl_close		Finish plotting.\n\
 plot_r n m dY		plot real part of rec m of stack el. n; height=dY\"\n\
 plot_i n m dY		plot imag part of rec m of stack el. n; height=dY\"\n\
Stack operations:\n\
 push [n]		duplicate stack element n (default: 0 = TOS)\n\
 pop [n]		delete stack element n (default: 0 = TOS)\n\
 pushpop n		pop stack element n and push it onto stack\n\
 xchg n			exchange TOS with stack element n (0 = TOS)\n\
 cat_y n		Concatenate top N stack elements; reclens must be equal\n\
 split_y n		Split TOS into N stack elements\n\
 stack			print summary of stack contents\n\
Shuffling operations:\n\
 bitrev			Reorder each record of TOS, e.g. after fft_dif\n\
 reverse_x		Reverse each record of TOS\n\
 delete_x nleft nright	Delete 1st nleft, last nright elements of each record\n\
 delete_y nstart nend	Delete 1st nstart, last nend records of TOS\n\
 edit_x	nleft ncopy	Delete 1st nleft, keep the next ncopy elements of TOS\n\
 rotate_x n		Rotate TOS n places towards +x (-reclen<n<reclen)\n\
 pad_x nleft nright	Pad each record of TOS with nleft, nright zero elements\n\
 midpad_x nleft npad	Pad each record with npad zeroes starting at n_left\n\
 context_x nleft	Pad start of each record with nleft words from end of previous record\n\
 redimension_x reclen	Join or split records to yield given record length\n\
 redimension_y nrec	Join or split records to yield given number of records\n\
 transpose n f		Transpose a block of f*n records every n records\n\
Math operations (note: for binary operators, NOS may have fewer dimensions):\n\
 add			add NOS to TOS.\n\
 sub			subtract NOS from TOS.\n\
 compare		compare TOS with NOS, print differences\n\
 mult			multiply TOS by NOS.\n\
 cconjugate		Negate imaginary portion of TOS.\n\
 fft_dit		forward fft of TOS, input in bitrev order\n\
 fft_dif		forward fft of TOS, output in bitrev order\n\
 fft_dif_scale		fft_dif, but also divide by 2 at stage i if NOS[i]==1\n\
 ifft_dif		inverse fft of TOS, input in bitrev order\n\
 ifft_dif_scale		ifft_dif, but only divide by 2 at stage i if NOS[i]==1\n\
 dft			direct forward dft of TOS\n\
 idft			direct inverse dft of TOS\n\
 convolve [-b] fftlen	TOS*NOS->TOS -b => block floating point; fftlen=0 -> direct\n\
 phase		 	TOS=phase(TOS)\n\
 magnitude	 	TOS=magnitude(TOS)\n\
 square		 	TOS=magnitude(TOS)^2\n\
 sum_x			sum TOS in X direction; result has reclen=1\n\
 mean_x			take mean of TOS in X direction; result has reclen=1\n\
 max_x			take max of TOS in X direction; result has reclen=1\n\
 argmax_x		take argmax of TOS in X direction; result is int16, reclen=1\n\
Conversion:\n\
 cvt_real		convert to real (truncate imaginary part)\n\
 cvt_complex		convert to complex (add zero imaginary part)\n\
 int8			convert to bytes\n\
 int16			convert to shorts\n\
 bfp16			convert to block floating point\n\
 float32		convert to floating point\n\
 bfp_normalize n	normalize a bfp signal to exp=n, then set exp to zero\n\
Built-in Generators:\n\
 ac_chirp tau bw f0 p0 nu_slow nu_fast		create an airsar chirp\n\
 tone n f p0		create n samples of constant tone exp(i*(ft+p0))\n\
");
    } else if (strcmp(cmd, "//")==0) {
	;
    } else if (strcmp(cmd, "#")==0) {
	;
    } else if (strcmp(cmd, "echo")==0) {
	fprintf(stdout, "%s\n", args);
	fflush(stdout);
    } else if (strcmp(cmd, "verbose")==0) {
	if (1 != sscanf(args, "%d", &verbose))
	    verbose=1;
    } else if (strcmp(cmd, "stack")==0) {
	stack_show(stack);
    } else if (strcmp(cmd, "push")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n))
	    n = 0;
	stack_push(stack, sig_copy(STK(n)));
    } else if (strcmp(cmd, "pop")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n))
	    n = 0;
	sig_destroy(stack_pop(stack, n));
    } else if (strcmp(cmd, "pushpop")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n)) {
	    fprintf(stdout, "usage: pushpop stack_element_num\n");
	    exit(1);
	}
	stack_push(stack, stack_pop(stack, n));
    } else if (strcmp(cmd, "xchg")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n)) {
	    fprintf(stdout, "usage: xchg stack_element_num\n");
	    exit(1);
	}
	stack_xchg(stack, n);
    } else if (strcmp(cmd, "cat_y")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n)) {
	    fprintf(stdout, "usage: cat_y stack_element_count\n");
	    exit(1);
	}
	stack_cat_y(stack, n);
    } else if (strcmp(cmd, "split_y")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n)) {
	    fprintf(stdout, "usage: split_y stack_element_count\n");
	    exit(1);
	}
	stack_split_y(stack, n);
    } else if (strcmp(cmd, "print")==0) {
	int nrec, nel;
	/* Defaults */
	nrec=0;
	nel=0;
	/* Parameters */
	if (strlen(args)>0) {
	    if (!sscanf(args, "%d %d", &nrec, &nel)){
		fprintf(stderr,
		    "Usage: print [nrec nel]; got %s\n", args);
		exit(1);
	    }
	}
	sig_fprint(STK(0), nrec, nel, stdout);
    } else if (strcmp(cmd, "verify")==0) {
	sig_verify(STK(0));
    } else if (strcmp(cmd, "pl_close")==0) {
	plot_close();
    } else if (strcmp(cmd, "pl_open")==0) {
	int height;
	int margin;
	char cmd[1024];

	/* Defaults */
	height=8;
	margin = 0;	/* use 1 for laserwriter */
	strcpy(cmd, "xplot");
	/* Parameters */
	if (strlen(args)>0) {
	    if (!sscanf(args, "%d %d %[^\n]", &height, &margin, cmd)){
		fprintf(stderr, 
		    "Usage: pl_open Y margin cmd; got %s\n", args);
		exit(1);
	    }
	}
	plot_open(cmd, height, margin);
    } else if (strcmp(cmd, "plot_r")==0) {
	int n, m;
	int height;

	/* Defaults */
	n=0;
	m=0;
	height=4;
	/* Parameters */
	if (strlen(args)>0) {
	    if (!sscanf(args, "%d %d %d", &n,&m,&height)){
		fprintf(stderr,
		    "Usage: plot_r n record dY; got %s\n", args);
		exit(1);
	    }
	}
	sig_plot_r(STK(n), m, height);
    } else if (strcmp(cmd, "plot_i")==0) {
	int n, m;
	int height;

	/* Defaults */
	n=0;
	m=0;
	height=4;
	/* Parameters */
	if (strlen(args)>0) {
	    if (!sscanf(args, "%d %d %d", &n,&m,&height)){
		fprintf(stderr,
		    "Usage: plot_i n record dY; got %s\n", args);
		exit(1);
	    }
	}
	sig_plot_i(STK(n), m, height);
    } else if (strcmp(cmd, "read")==0) {
	char fname[128];
	/* Get filename, read complex vector from file */
	if (1 != sscanf(args, "%s", fname)) {
	    fprintf(stdout, "usage: read_r8 filename\n");
	    exit(1);
	}
	stack_push(stack, sig_read(fname));
    } else if (strcmp(cmd, "write")==0) {
	char fname[128];
	/* Get filename, write complex vector to file */
	if (1 != sscanf(args, "%s", fname)) {
	    fprintf(stdout, "usage: write filename\n");
	    exit(1);
	}
	sig_write(STK(0), fname);
    } else if (strcmp(cmd, "new")==0) {
	next_may_append = 1;
	stack_push(stack, sig_new(1, 1, hdr_INT16, ""));
	STK(0)->nrec = 0;
    } else if (strcmp(cmd, "append")==0) {
	/* Append a record to TOS.  If TOS is empty, set reclen & kind. */ 
	int i;
	int offset;

	next_may_append = 1;
	if (!may_append) {
	    fprintf(stdout, "append: you can't append unless the last command was 'new' or 'append'!\nThis keeps you from forgetting the 'new'.\n");
	    exit(1);
	}
	tos = STK(0);

	tos->nrec++;
	if (tos->nrec == 1) {
	    tos->kind = hdr_INT16;
	    if (strchr(args, '.') != NULL)
		tos->kind = hdr_FLOAT32;
	    if (strchr(args, ',') != NULL) {
		tos->kind |= hdr_KIND_COMPLEX;
		tos->im = (short *)malloc(1);
	    }
	    /* make a big initial guess at reclen */
	    tos->reclen = 1000;
	}
	/* Make room for new size */
	tos->re = (short *)realloc((char *)tos->re, 
	    tos->nrec * tos->reclen * sig_ELBYTES(tos));
	if (tos->kind & hdr_KIND_COMPLEX)
	    tos->im = (short *)realloc((char *)tos->im, 
		tos->nrec * tos->reclen * sig_ELBYTES(tos));

	offset = tos->reclen * (tos->nrec-1);
	i = 0;
	for (tok=strtok(args,BLANKS); tok; tok=strtok(NULL, BLANKS)) {
	    if (tok[0] == ',') {
		fprintf(stdout, "append: expected real[,imag], saw %s\n", tok);
		exit(1);
	    }
	    switch (tos->kind) {
	    case hdr_CFLOAT32:
		((float *) tos->im)[i+offset] = 0;	/* default */
		if (sscanf(tok, "%f,%f", 
		((float *) tos->re)+i+offset, 
		((float *) tos->im)+i+offset) < 1){
		    fprintf(stdout,
			"append: expected 'real[,imag]', got %s\n", tok);
		    exit(1);
		}
		break;
	    case hdr_FLOAT32:
		((float *) tos->re)[i+offset] = atof(tok);
		break;
	    case hdr_CINT16:
		tos->im[i+offset] = 0;			/* default */
		if (sscanf(tok, "%hd,%hd", 
		    tos->re+i+offset, tos->im+i+offset) < 1){
		    fprintf(stdout,
			"append: expected 'real[,imag]', got %s\n", tok);
		    exit(1);
		}
		break;
	    case hdr_INT16:
		tos->re[i+offset] = atoi(tok);
		break;
	    default:
		badkind("append", tos->kind);
	    }
	    i++;
	}
	if (tos->nrec == 1)
	    tos->reclen = i;
	else if (tos->reclen != i) {
	    fprintf(stdout, 
		"append: len(TOS)=%d, but only %d numbers were entered!\n",
		tos->reclen, i);
	    exit(1);
	}
    } else if (strcmp(cmd, "real")==0) {
	int i = 0;
	int kind;
	if (strchr(args, '.') != NULL)
	    kind = hdr_FLOAT32;
	else
	    kind = hdr_INT16;
	sig = sig_new(1, 1024, kind, "");/* oversize it, hopefully */
	for (tok=strtok(args,BLANKS); tok; tok=strtok(NULL, BLANKS)) {
	    if (kind == hdr_FLOAT32)
		((float *) sig->re)[i] = atof(tok);
	    else
		sig->re[i] = atoi(tok);
	    i++;
	}
	sig->reclen = i;
	stack_push(stack, sig);
    } else if (strcmp(cmd, "imag")==0) {
	int i = 0;
	int kind;
	sig = STK(0);
	kind = (sig->kind |= hdr_KIND_COMPLEX);
	sig->im = (short *)malloc(sig->nrec*sig->reclen*sig_ELBYTES(sig));
	for (tok=strtok(args,BLANKS); tok; tok=strtok(NULL, BLANKS)) {
	    if (i > sig->reclen) {
		fprintf(stderr, "len(imag) > len(real) = %d\n",sig->reclen);
		exit(1);
	    }
	    if (kind == hdr_CFLOAT32)
		((float *) sig->im)[i] = atof(tok);
	    else
		sig->im[i] = atoi(tok);
	    i++;
	}
	if (sig->reclen != i) {
	    fprintf(stderr, "imag: len(imag) < len(real) ; %d != %d\n",i,sig->reclen);
	    exit(1);
	}
    } else if (strcmp(cmd, "name")==0) {
	sig = STK(0);
	tok=strtok(args,BLANKS);
	if (!tok) {
	    fprintf(stderr, "usage: name NAME\n");
	    exit(1);
	}
	(void) strcpy(sig->fname, tok);
    } else if (strcmp(cmd, "compare")==0) {
	sig_compare(STK(0), STK(1));
	sig_destroy(stack_pop(stack, 0));
	sig_destroy(stack_pop(stack, 0));
    } else if (strcmp(cmd, "convolve")==0) {
	tok=strtok(args, BLANKS);
	if (tok && (tok[0] == '-')) {
	    if (tok[1] == 'b') {
		bfp_do = 1;		/* enable block-floating-point */
	    } else
		printf("convolve: Bad option %s\n", tok);
	    tok=strtok(NULL, BLANKS);
	}
	if (tok == NULL) {
	    fprintf(stdout, "usage: convolve -b fft_size\n");
	    exit(1);
	}
	sig = sig_cconvolve(STK(0), STK(1), atoi(tok));
	/* Replace TOS and NOS with result of convolution */
	sig_destroy(stack_pop(stack, 0));
	sig_destroy(stack_pop(stack, 0));
	stack_push(stack, sig);
	bfp_do = 0;
    } else if (strcmp(cmd, "phase")==0)	sig_phase(STK(0));
    else if (strcmp(cmd, "magnitude")==0)	sig_magnitude(STK(0));
    else if (strcmp(cmd, "square")==0)	sig_square(STK(0));
    else if (strcmp(cmd, "sum_x")==0)	sig_sum_x(STK(0));
    else if (strcmp(cmd, "mean_x")==0)	sig_mean_x(STK(0));
    else if (strcmp(cmd, "max_x")==0)	sig_max_x(STK(0));
    else if (strcmp(cmd, "argmax_x")==0)	sig_argmax_x(STK(0));
    else if (strcmp(cmd, "cconjugate")==0)	sig_cconjugate(STK(0));
    else if (strcmp(cmd, "fft_dit")==0)	sig_cfft_dit(STK(0));
    else if (strcmp(cmd, "fft_dif")==0)	sig_cfft_dif(STK(0), NULL);
    else if (strcmp(cmd, "fft_dif_scale")==0) {
	sig_cfft_dif(STK(0), STK(1));
	sig_destroy(stack_pop(stack, 1));
    } else if (strcmp(cmd, "ifft_dif_scale")==0) {
	sig_cifft_dif(STK(0), STK(1));
	sig_destroy(stack_pop(stack, 1));
    } else if (strcmp(cmd, "ifft_dif")==0)	sig_cifft_dif(STK(0), NULL);
    else if (strcmp(cmd, "dft")==0) 	sig_cfft_direct(STK(0));
    else if (strcmp(cmd, "idft")==0)	sig_cifft_direct(STK(0));
    else if (strcmp(cmd, "add")==0) {
       sig_add(STK(0), STK(1));
       sig_destroy(stack_pop(stack, 1));
    } else if (strcmp(cmd, "sub")==0) {
	sig_sub(STK(0), STK(1));
	sig_destroy(stack_pop(stack, 1));
    } else if (strcmp(cmd, "mult")==0) {
	sig_mult(STK(0), STK(1));
	sig_destroy(stack_pop(stack, 1));
    } else if (strcmp(cmd, "bitrev")==0) {
	sig = sig_bitrev(STK(0));
	sig_destroy(stack_pop(stack, 0));
	stack_push(stack, sig);
    } else if (strcmp(cmd, "reverse_x")==0) {
	sig_reverse(STK(0));
    } else if (strcmp(cmd, "redimension_x")==0) {
	int new_reclen;
	if (1 != sscanf(args, "%d %s", &new_reclen, garbage)) {
	    fprintf(stdout, "usage: redimension_x new_reclen; got %s\n", args);
	    exit(1);
	}
	sig_redimension_x(STK(0), new_reclen);
    } else if (strcmp(cmd, "redimension_y")==0) {
	int new_nrec;
	if (1 != sscanf(args, "%d %s", &new_nrec, garbage)) {
	    fprintf(stdout, "usage: redimension_y new_nrec; got %s\n", args);
	    exit(1);
	}
	sig_redimension_y(STK(0), new_nrec);
    } else if (strcmp(cmd, "transpose")==0) {
	int n, f;
	if (2 != sscanf(args, "%d %d %s", &n, &f, garbage)) {
	    fprintf(stdout, "usage: transpose n f; got %s\n", args);
	    exit(1);
	}
	sig_transpose(STK(0), n, f);
    } else if (strcmp(cmd, "edit_x")==0) {
	int n_skip;
	int n_copy;
	if (2 != sscanf(args, "%d %d", &n_skip, &n_copy)) {
	    fprintf(stdout, "usage: edit_x n_skip n_copy; got %s\n", args);
	    exit(1);
	}
	sig_edit_x(STK(0), n_skip, n_copy);
    } else if (strcmp(cmd, "rotate_x")==0) {
	int n;
	if (1 != sscanf(args, "%d", &n)) {
	    fprintf(stdout, "usage: rotate_x n; got %s\n", args);
	    exit(1);
	}
	sig_rotate_x(STK(0), n);
    } else if (strcmp(cmd, "delete_x")==0) {
	int n_left;
	int n_right;
	if (2 != sscanf(args, "%d %d", &n_left, &n_right)) {
	    fprintf(stdout, "usage: delete_x n_left n_right; got %s\n", args);
	    exit(1);
	}
	sig_delete_x(STK(0), n_left, n_right);
    } else if (strcmp(cmd, "delete_y")==0) {
	int n_begin;
	int n_end;
	if (2 != sscanf(args, "%d %d", &n_begin, &n_end)) {
	    fprintf(stdout, "usage: delete_y n_begin n_end; got %s\n", args);
	    exit(1);
	}
	sig_delete_y(STK(0), n_begin, n_end);
    } else if (strcmp(cmd, "context_x")==0) {
	int n_left;
	if (1 != sscanf(args, "%d %s", &n_left, garbage)) {
	    fprintf(stdout, "usage: context_x n_left; got %s\n", args);
	    exit(1);
	}
	tos = stack_pop(stack, 0);
	sig = sig_context_x(tos, n_left);
	sig_destroy(tos);
	stack_push(stack, sig);
    } else if (strcmp(cmd, "pad_x")==0) {
	int n_right;
	int n_left;
	if (2 != sscanf(args, "%d %d", &n_left, &n_right)) {
	    fprintf(stdout, "usage: pad_x n_left n_right; got %s\n", args);
	    exit(1);
	}
	tos = stack_pop(stack, 0);
	sig = sig_pad_x(tos, n_left, n_right);
	sig_destroy(tos);
	stack_push(stack, sig);
    } else if (strcmp(cmd, "midpad_x")==0) {
	int n_pad;
	int n_left;
	if (2 != sscanf(args, "%d %d", &n_left, &n_pad)) {
	    fprintf(stdout, "usage: midpad_x n_left n_pad; got %s\n", args);
	    exit(1);
	}
	tos = stack_pop(stack, 0);
	sig = sig_midpad_x(tos, n_left, n_pad);
	sig_destroy(tos);
	stack_push(stack, sig);
    } else if (strcmp(cmd, "ac_chirp")==0) {
	double tau, bw, f_0, p_0;
	double nu_slow, nu_fast;
	if (6 != sscanf(args, "%lf %lf %lf %lf %lf %lf", 
	    &tau, &bw, &f_0, &p_0, &nu_slow, &nu_fast)) {
	    fprintf(stdout, "usage: ac_chirp tau bw f0 p0 nu_slow nu_fast\n");
	    exit(1);
	}
	stack_push(stack, sig_ac_chirp(tau, bw, f_0, p_0, nu_slow, nu_fast));
    } else if (strcmp(cmd, "tone")==0) {
	int reclen;
	double f, p0;
	if (3 != sscanf(args, "%d %lf %lf", &reclen, &f, &p0)) {
	    fprintf(stdout, "usage: tone reclen f p0\n");
	    exit(1);
	}
	stack_push(stack, sig_tone(reclen, f, p0));
    } else if (strcmp(cmd, "cvt_real")==0) {
	sig_real(STK(0));
    } else if (strcmp(cmd, "cvt_complex")==0) {
	sig_complex(STK(0));
    } else if (strcmp(cmd, "int8")==0) {
	sig_int8(STK(0));
    } else if (strcmp(cmd, "int16")==0) {
	sig_int16(STK(0));
    } else if (strcmp(cmd, "bfp16")==0) {
	sig_bfp16(STK(0));
    } else if (strcmp(cmd, "float32")==0) {
	sig_float32(STK(0));
    } else if (strcmp(cmd, "bfp_normalize")==0) {
	int bits;
	if (1 != sscanf(args, "%d %s", &bits, garbage)) {
	    fprintf(stdout, "usage: bfp_normalize bits; got %s\n", args);
	    exit(1);
	}
	sig_bfp_normalize(STK(0), bits);
    } else {
	return 1;
    }
    return 0;
}

/*--------------------------------------------------------------------------
 Get the next non-commented line.
 Obey line continuation char \ at EOL.
--------------------------------------------------------------------------*/
char *
gets_skiphash(s)
    char *s;
{
    do {
	if (!gets(s))
	    return NULL;
    } while (s[0] == '#' || strncmp(s, "//", 2)==0);

    while ((s[strlen(s)-1] == '\\') && gets(&s[strlen(s)-1]))
	;

    return s;
}

#define MAC_STACK_DEPTH 100

/*--------------------------------------------------------------------------
 Read commands from stdin.
--------------------------------------------------------------------------*/
main()
{
    /* Macro stuff */
    macro_inst_t *inst;
    macro_inst_t *mac_stack[MAC_STACK_DEPTH];
    int mac_sp = -1;
    int defining_macro = 0;

    /* Temporary variables */
    static char ibuf[2048];

    /* Clear the stack */
    stack->n = 0;

    /* Initialize things for weak modules */
    i_unity = (float) (1 << FRAC_BITS);
    bfp_do = 0;

    /* Parse commands of the form
     * cmdname args\n
     */
    for (;;) {
	char cmd[100];
	char args[1024];

	/* Fetch next line. */

	/* Try to expand macro.  Keep trying until stack empty. */
	while (mac_sp >= 0 && macro_expand_next(mac_stack[mac_sp], ibuf)) {
	    /* Hit end of macro.  Pop stack. */ 
	    macro_expand_end(mac_stack[mac_sp]);
	    mac_sp--;
	}
	/* If mac_sp >= 0, we have fetched a line from a macro. */

	if (mac_sp < 0) {
	    if (!gets_skiphash(ibuf))
		break;
	}

	if (verbose) {
	    puts(ibuf);
	    fflush(stdout);
	}
	args[0] = 0;
	if (sscanf(ibuf, "%s %[^\n]", cmd, args)<1)
	    continue;

	if (strcmp(cmd, "macro")==0) {
	    char mname[128];
	    if (sscanf(args, "%s", mname) != 1) {
		fprintf(stderr, "Usage: macro macroname\n");
		exit(1);
	    }
	    /* Defining macro. */
	    if (defining_macro || (mac_sp >= 0)) {
		fprintf(stderr, "Cannot nest macro definitions, macro %s.\n",
		    args);
		exit(1);
	    }
	    defining_macro = 1;
	    macro_define(mname);
	} else if (strcmp(cmd, "end")==0) {
	    if (defining_macro)
		defining_macro = 0;
	    else
		break;
	} else if (defining_macro) {
	    macro_append(ibuf);
	} else if (do_cmd(cmd, args)==0) {
	    ;
	} else if ((inst=macro_expand_begin(cmd, args))!=NULL) {
	    /* Start expanding macro */
	    mac_sp++;
	    if (mac_sp >= MAC_STACK_DEPTH) {
		fprintf(stderr, "Macros nested too deeply\n");
		exit(1);
	    }
	    mac_stack[mac_sp]=inst;
	} else {
	    fprintf(stderr, "Unrecognized command %s\n", cmd);
	    exit(1);
	}
    }

    exit(0);
    /*NOTREACHED*/
}
