#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <values.h>

#include "imath.h"
#include "sig.h"
static char sccsid[] = "@(#)arith.c	1.5 7/23/91";

/*--------------------------------------------------------------------------
 Routines for checking type and dimension compatibility.
--------------------------------------------------------------------------*/
void
badkind(msg, kind)
    char *msg;
    int kind;
{
    char buf[128];

    hdr_kind2a(kind, buf);
    fprintf(stderr, "%s: unsupported data type %s\n", msg, buf);
    exit(1);
}

void
samekind(msg, this, that)
    char *msg;
    sig_t *this, *that;
{
    if (this->kind != that->kind) {
	char thisbuf[128];
	char thatbuf[128];

	hdr_kind2a(this->kind, thisbuf);
	hdr_kind2a(that->kind, thatbuf);
	fprintf(stderr, "%s: arguments must be same data type (%s != %s)\n",
	    msg, thisbuf, thatbuf);
	exit(1);
    }
}

/******************* Multiplication *************************/

/*--------------------------------------------------------------------------
 Multiply vector b by scalar s.
--------------------------------------------------------------------------*/
static void
smult_cint16(n, sr, si, br, bi)
    int n;
    short sr, si;
    short *br, *bi;
{
    while (n--) {
	*br = (I_HALF + *br * sr - *bi * si) >> FRAC_BITS;
	*bi = (I_HALF + *br * si + *bi * sr) >> FRAC_BITS;
	br++;
	bi++;
    }
}

/*--------------------------------------------------------------------------
 Multiply vector b by scalar s.
 Both b and s are block-floating-point.
--------------------------------------------------------------------------*/
static void
smult_cbfp16(n, sr, si, se, br, bi, be)
    int n;
    short sr, si, se;
    short *br, *bi;			/* pointer to array of n shorts */
    short *be;				/* pointer to single short */
{
    while (n--) {
	*br = (I_HALF + *br * sr - *bi * si) >> FRAC_BITS;
	*bi = (I_HALF + *br * si + *bi * sr) >> FRAC_BITS;
	br++;
	bi++;
    }
    *be += se;
}

static void
smult_float32(n, sr, br)
    int n;
    float sr;
    float *br;
{
    while (n--)
	*br++ *= sr;
}

static void
smult_cfloat32(n, sr, si, br, bi)
    int n;
    float sr, si;
    float *br, *bi;
{
    while (n--) {
	*br = (*br * sr - *bi * si);
	*bi = (*br * si + *bi * sr);
	br++;
	bi++;
    }
}
/*--------------------------------------------------------------------------
 Multiply vector this by scalar that.
 That must have exactly one record, with reclen 1.
--------------------------------------------------------------------------*/
static void
sig_smult(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    if (that->reclen != 1) {
	fprintf(stderr, "sig_smult(%s,%s): 2nd reclen must be 1, is %d\n",
	    this->fname, that->fname, that->reclen);
	exit(1);
    }
    if (that->nrec != 1) {
	fprintf(stderr, "sig_smult(%s,%s): 2nd nrec must be 1, but is %d\n",
	    this->fname, that->fname, that->nrec);
	exit(1);
    }
    samekind("sig_smult", this, that);

    switch (this->kind) {
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    smult_cint16(this->reclen, 
		that->re[0], 
		that->im[0], 
		this->re+offset, 
		this->im+offset);
	}
	break;
    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    smult_cbfp16(this->reclen, 
		that->re[0], 
		that->im[0], 
		that->exp[0], 
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    smult_cfloat32(this->reclen, 
		((float *) that->re)[0], 
		((float *) that->im)[0], 
		((float *) this->re)+offset, 
		((float *) this->im)+offset);
	}
	break;
    case hdr_FLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    smult_float32(this->reclen, 
		((float *) that->re)[0], 
		((float *) this->re)+offset);
	}
	break;
    default:
	badkind("sig_smult", this->kind);
    }
}

/*--------------------------------------------------------------------------
 Multiply vector B by vector A, result to vector B.
--------------------------------------------------------------------------*/
void
mult_cint16(n, ar, ai, br, bi)
    int n;
    short *ar, *ai;
    short *br, *bi;
{
    while (n--) {
	short brc, bic;
	brc = *br;
	bic = *bi;

	*br = (I_HALF + *ar * brc - *ai * bic) >> FRAC_BITS;
	*bi = (I_HALF + *ar * bic + *ai * brc) >> FRAC_BITS;
	ar++; ai++; br++; bi++;
    }
}

void
mult_cbfp16(n, ar, ai, ae, br, bi, be)
    int n;
    short *ar, *ai;		/* pointers to array of n shorts */
    short ae;
    short *br, *bi;		/* pointers to array of n shorts */
    short *be;			/* pointer to single short */
{
    while (n--) {
	short brc, bic;
	brc = *br;
	bic = *bi;

	*br = (I_HALF + *ar * brc - *ai * bic) >> FRAC_BITS;
	*bi = (I_HALF + *ar * bic + *ai * brc) >> FRAC_BITS;
	ar++; ai++; br++; bi++;
    }
    *be += ae;
}

void
mult_cfloat32(n, ar, ai, br, bi)
    int n;
    float *ar, *ai;
    float *br, *bi;
{
    while (n--) {
	float brc, bic;
	brc = *br;
	bic = *bi;

	*br = *ar * brc - *ai * bic;
	*bi = *ar * bic + *ai * brc;
	ar++; ai++; br++; bi++;
    }
}

/*--------------------------------------------------------------------------
 Multiply vector this by that, result to this.
 That must have exactly one record, and must have same record length as this.
--------------------------------------------------------------------------*/
static void
sig_mult_repeatthat(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    if (that->nrec != 1) {
	fprintf(stderr, "sig_mult_repeatthat(%s,%s): 2nd nrec must be 1, but is %d\n",
	    this->fname, that->fname, that->nrec);
	exit(1);
    }

    /* Promote that */
    if (that->reclen == 1) {
	sig_smult(this, that);
	return;
    }

    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_mult_repeatthat(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }
    samekind("sig_mult_repeatthat", this, that);

    switch (this->kind) {
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cint16(this->reclen, 
		that->re, 
		that->im, 
		this->re+offset, 
		this->im+offset);
	}
	break;
    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cbfp16(this->reclen, 
		that->re, 
		that->im, 
		that->exp[0], 
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cfloat32(this->reclen, 
		((float *)that->re), 
		((float *)that->im), 
		((float *)this->re)+offset, 
		((float *)this->im)+offset);
	}
	break;
    default:
	badkind("sig_mult_repeatthat", this->kind);
    }
}

/*--------------------------------------------------------------------------
 Multiply vector this by that, result to this.
 This and that must have same nrec and reclen.
 However, if that is a scalar or a single record, it is promoted
 by replication.
--------------------------------------------------------------------------*/
void
sig_mult(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    /* Promote that */
    if (that->nrec == 1) {
	sig_mult_repeatthat(this, that);
	return;
    }

    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_mult(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }
    if (this->nrec != that->nrec) {
	fprintf(stderr, "sig_mult(%s,%s): nrecs differ: %d != %d\n",
	    this->fname, that->fname, this->nrec, that->nrec);
	exit(1);
    }
    samekind("sig_mult", this, that);

    switch (this->kind) {
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cint16(this->reclen, 
		that->re+offset, 
		that->im+offset, 
		this->re+offset, 
		this->im+offset);
	}
	break;
    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cbfp16(this->reclen, 
		that->re+offset, 
		that->im+offset, 
		that->exp[n],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    mult_cfloat32(this->reclen, 
		((float *)that->re)+offset, 
		((float *)that->im)+offset, 
		((float *)this->re)+offset, 
		((float *)this->im)+offset);
	}
	break;
    default:
	badkind("sig_mult", this->kind);
    }
}

/******************* Addition *************************/

/*--------------------------------------------------------------------------
 Add scalar s to vector b.
--------------------------------------------------------------------------*/
static void
sadd_int8(n, s, b)
    int n;
    char s;
    char *b;
{
    while (n--)
	*b++ += s;
}

static void
sadd_int16(n, s, b)
    int n;
    short s;
    short *b;
{
    while (n--)
	*b++ += s;
}

static void
sadd_cbfp16(n, sr, si, se, br, bi, be)
    int n;
    short sr, si, se;
    short *br, *bi;		/* array of n shorts */
    short *be;			/* a single short */
{
    /* If possible, adjust s to match b's exponant without overflow. 
     * This can be done if be>=se.
     */
    if (*be >= se) {
	sr >>= (*be - se);
	si >>= (*be - se);
	while (n--) {
	    *br++ += sr;
	    *bi++ += si;
	}
    } else {
	/* Vector needs to be shifted right to match scalar's
	 * exponant before adding in the scalar.
	 */
	int shift = se - *be;
	*be += shift;
	while (n--) {
	    *br = (*br >> shift) + sr;
	    br++;
	    *bi = (*bi >> shift) + si;
	    bi++;
	}
    }
}

/*--------------------------------------------------------------------------
 Add scalar that to vector this.
 That must have exactly one record, with reclen 1.
--------------------------------------------------------------------------*/
static void
sig_sadd(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    if (that->reclen != 1) {
	fprintf(stderr, "sig_sadd(%s,%s): 2nd reclen must be 1, is %d\n",
	    this->fname, that->fname, that->reclen);
	exit(1);
    }
    if (that->nrec != 1) {
	fprintf(stderr, "sig_sadd(%s,%s): 2nd nrec must be 1, but is %d\n",
	    this->fname, that->fname, that->nrec);
	exit(1);
    }
    samekind("sig_sadd", this, that);

    switch (this->kind) {
    case hdr_CINT8:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sadd_int8(this->reclen, 
		    ((char *)that->im)[0], 
		    ((char *)this->im)+offset);
	/* FALL THRU */
    case hdr_INT8:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sadd_int8(this->reclen, 
		    ((char *)that->re)[0],
		    ((char *)this->re)+offset);
	break;

    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sadd_int16(this->reclen, that->im[0], this->im+offset);
	/* FALL THRU */
    case hdr_INT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sadd_int16(this->reclen, that->re[0], this->re+offset);
	break;

    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    sadd_cbfp16(this->reclen, 
		that->re[0], 
		that->im[0], 
		that->exp[0],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;
    default:
	badkind("sig_sadd", this->kind);
    }
}

/*--------------------------------------------------------------------------
 Add vector A to vector B.
--------------------------------------------------------------------------*/
void
add_int16(n, a, b)
    int n;
    short *a;
    short *b;
{
    while (n--)
	*b++ += *a++;
}

void
add_float32(n, a, b)
    int n;
    float *a;
    float *b;
{
    while (n--)
	*b++ += *a++;
}

/*--------------------------------------------------------------------------
 Add vector A to vector B.
 Complex.
 Both A and B are block-floating-point.
--------------------------------------------------------------------------*/
void
add_cbfp16(n, ar, ai, ae, br, bi, be)
    int n;
    short *ar, *ai;
    short ae;
    short *br, *bi;
    short *be;
{
    int shift;

    if (*be == ae) {
	while (n--) {
	    *br++ += *ar++;
	    *bi++ += *ai++;
	}
    } else if (*be > ae) {
	shift = *be - ae;
	while (n--) {
	    *br++ += (*ar++ >> shift);
	    *bi++ += (*ai++ >> shift);
	}
    } else {
	shift = ae - *be;
	*be += shift;
	while (n--) {
	    *br = (*br >> shift) + *ar++;
	    br++;
	    *bi = (*bi >> shift) + *ai++;
	    bi++;
	}
    }
}

/*--------------------------------------------------------------------------
 Add that to this.
 That has exactly one record, but has reclen=1 or this->reclen.
--------------------------------------------------------------------------*/
static void
sig_add_repeatthat(this, that)
    sig_t *this, *that;
{
    int nrec = this->nrec;
    int reclen = this->reclen;

    int n;
    int offset;

    /* Promote that */
    if (that->reclen == 1) {
	sig_sadd(this, that);
	return;
    }
    if (that->nrec != 1) {
	fprintf(stderr, "sig_add_repeatthat(%s,%s): that should have nrec 1, but nrec is %d\n",
	    this->fname, that->fname, that->nrec);
	exit(1);
    }
    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_add_repeatthat(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }

    samekind("sig_add_repeatthat", this, that);

    switch (this->kind) {
    case hdr_CINT16: {
	short *thisim = this->im;
	short *thatim = that->im;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_int16(reclen, thatim, thisim+offset);
	}
	/* FALL THRU */
    case hdr_INT16: {
	short *thisre = this->re;
	short *thatre = that->re;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_int16(reclen, thatre, thisre+offset);
	}
	break;

    case hdr_CBFP16:
	for (n=0, offset=0; n<nrec; n++, offset+=reclen){
	    add_cbfp16(this->reclen, 
		that->re, 
		that->im,
		that->exp[0],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;

    case hdr_CFLOAT32: {
	float *thisim = (float *)this->im;
	float *thatim = (float *)that->im;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_float32(reclen, thatim, thisim+offset);
	}
	/* FALL THRU */
    case hdr_FLOAT32: {
	float *thisre = (float *)this->re;
	float *thatre = (float *)that->re;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_float32(reclen, thatre, thisre+offset);
	}
	break;

    default:
	badkind("sig_add_repeatthat", this->kind);
    }
}

/*--------------------------------------------------------------------------
 Add vector that to this.
--------------------------------------------------------------------------*/
void
sig_add(this, that)
    sig_t *this, *that;
{
    int nrec = this->nrec;
    int reclen = this->reclen;

    int n;
    int offset;

    /* Promote that */
    if (that->nrec == 1) {
	sig_add_repeatthat(this, that);
	return;
    }

    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_add(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }
    if (this->nrec != that->nrec) {
	fprintf(stderr, "sig_add(%s,%s): nrecs differ: %d != %d\n",
	    this->fname, that->fname, this->nrec, that->nrec);
	exit(1);
    }
    samekind("sig_add", this, that);

    switch (this->kind) {
    case hdr_CINT16: {
	short *thisim = this->im;
	short *thatim = that->im;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_int16(reclen, thatim+offset, thisim+offset);
	}
	/* FALL THRU */
    case hdr_INT16: {
	short *thisre = this->re;
	short *thatre = that->re;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_int16(reclen, thatre+offset, thisre+offset);
	}
	break;

    case hdr_CBFP16:
	for (n=0, offset=0; n<nrec; n++, offset+=reclen){
	    add_cbfp16(this->reclen, 
		that->re+offset, 
		that->im+offset,
		that->exp[n],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;

    case hdr_CFLOAT32: {
	float *thisim = (float *)this->im;
	float *thatim = (float *)that->im;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_float32(reclen, thatim+offset, thisim+offset);
	}
	/* FALL THRU */
    case hdr_FLOAT32: {
	float *thisre = (float *)this->re;
	float *thatre = (float *)that->re;
	for (n=0, offset=0; n<nrec; n++, offset+=reclen)
	    add_float32(reclen, thatre+offset, thisre+offset);
	}
	break;

    default:
	badkind("sig_add", this->kind);
    }
}

/******************* Subtraction *************************/

/*--------------------------------------------------------------------------
 Subtract scalar s from vector b.
--------------------------------------------------------------------------*/
static void
ssub_int8(n, s, b)
    int n;
    char s;
    char *b;
{
    while (n--)
	*b++ -= s;
}

static void
ssub_int16(n, s, b)
    int n;
    short s;
    short *b;
{
    while (n--)
	*b++ -= s;
}

static void
ssub_cbfp16(n, sr, si, se, br, bi, be)
    int n;
    short sr, si, se;
    short *br, *bi;		/* array of n shorts */
    short *be;			/* a single short */
{
    /* If possible, adjust s to match b's exponant without overflow. 
     * This can be done if be>=se.
     */
    if (*be >= se) {
	sr >>= (*be - se);
	si >>= (*be - se);
	while (n--) {
	    *br++ -= sr;
	    *bi++ -= si;
	}
    } else {
	/* Vector needs to be shifted right to match scalar's
	 * exponant before adding in the scalar.
	 */
	int shift = se - *be;
	*be += shift;
	while (n--) {
	    *br = (*br >> shift) - sr;
	    br++;
	    *bi = (*bi >> shift) - si;
	    bi++;
	}
    }
}

/*--------------------------------------------------------------------------
 Subtract scalar that from vector this.
 That must have exactly one record, with reclen 1.
--------------------------------------------------------------------------*/
static void
sig_ssub(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    if (that->reclen != 1) {
	fprintf(stderr, "sig_ssub(%s,%s): 2nd reclen must be 1, is %d\n",
	    this->fname, that->fname, that->reclen);
	exit(1);
    }
    if (that->nrec != 1) {
	fprintf(stderr, "sig_ssub(%s,%s): 2nd nrec must be 1, but is %d\n",
	    this->fname, that->fname, that->nrec);
	exit(1);
    }
    samekind("sig_ssub", this, that);

    switch (this->kind) {
    case hdr_CINT8:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    ssub_int8(this->reclen, 
		    ((char *)that->im)[0], 
		    ((char *)this->im)+offset);
	/* FALL THRU */
    case hdr_INT8:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    ssub_int8(this->reclen, 
		    ((char *)that->re)[0],
		    ((char *)this->re)+offset);
	break;

    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    ssub_int16(this->reclen, that->im[0], this->im+offset);
	/* FALL THRU */
    case hdr_INT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    ssub_int16(this->reclen, that->re[0], this->re+offset);
	break;

    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    ssub_cbfp16(this->reclen, 
		that->re[0], 
		that->im[0], 
		that->exp[0],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;
    default:
	badkind("sig_ssub", this->kind);
    }
}

/*--------------------------------------------------------------------------
 Subtract vector A from vector B.
--------------------------------------------------------------------------*/
void
sub_int16(n, a, b)
    int n;
    short *a;
    short *b;
{
    while (n--)
	*b++ -= *a++;
}
void
sub_float32(n, a, b)
    int n;
    float *a;
    float *b;
{
    while (n--)
	*b++ -= *a++;
}

/*--------------------------------------------------------------------------
 Subtract vector A from vector B.
 Complex.
 Both A and B are block-floating-point.
--------------------------------------------------------------------------*/
void
sub_cbfp16(n, ar, ai, ae, br, bi, be)
    int n;
    short *ar, *ai;
    short ae;
    short *br, *bi;
    short *be;
{
    int shift;

    if (*be == ae) {
	while (n--) {
	    *br++ -= *ar++;
	    *bi++ -= *ai++;
	}
    } else if (*be > ae) {
	shift = *be - ae;
	while (n--) {
	    *br++ -= (*ar++ >> shift);
	    *bi++ -= (*ai++ >> shift);
	}
    } else {
	shift = ae - *be;
	*be += shift;
	while (n--) {
	    *br = (*br >> shift) - *ar++;
	    br++;
	    *bi = (*bi >> shift) - *ai++;
	    bi++;
	}
    }
}

/*--------------------------------------------------------------------------
 Subtract vector that from this.
 This and that must be same size.
--------------------------------------------------------------------------*/
void
sig_sub(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;

    /* Promote that */
    if (that->nrec == 1 && that->reclen == 1) {
	sig_ssub(this, that);
	return;
    }

    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_sub(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }
    if (this->nrec != that->nrec) {
	fprintf(stderr, "sig_sub(%s,%s): nrecs differ: %d != %d\n",
	    this->fname, that->fname, this->nrec, that->nrec);
	exit(1);
    }
    samekind("sig_sub", this, that);

    switch (this->kind) {
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sub_int16(this->reclen, that->im+offset, this->im+offset);
	/*FALL THRU*/
    case hdr_INT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sub_int16(this->reclen, that->re+offset, this->re+offset);
	break;

    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen){
	    sub_cbfp16(this->reclen, 
		that->re+offset, 
		that->im+offset,
		that->exp[n],
		this->re+offset, 
		this->im+offset,
		this->exp+n);
	}
	break;

    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sub_float32(this->reclen, 
		((float *)that->im)+offset, 
		((float *)this->im)+offset);
	/*FALL THRU*/
    case hdr_FLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    sub_float32(this->reclen, 
		((float *)that->re)+offset, 
		((float *)this->re)+offset);
	break;

    default:
	badkind("sig_sub", this->kind);
    }
}

/******************* Conjugation *************************/

/*--------------------------------------------------------------------------
 Negate the given vector.
--------------------------------------------------------------------------*/
void
neg_int16(n, a)
    int n;
    short *a;
{
    while (n--)
	*a++ *= -1;
}

void
neg_float32(n, a)
    int n;
    float *a;
{
    while (n--)
	*a++ *= -1.0;
}

/*--------------------------------------------------------------------------
 Negate the imaginary portion of this.
--------------------------------------------------------------------------*/
void
sig_cconjugate(this)
    sig_t *this;
{
    int n;
    int offset;

    switch (this->kind) {
    case hdr_CBFP16:
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    neg_int16(this->reclen, this->im+offset);
	/*FALL THRU*/
    case hdr_BFP16:
    case hdr_INT16:
	break;

    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    neg_float32(this->reclen, ((float *)this->im)+offset);
	/*FALL THRU*/
    case hdr_FLOAT32:
	break;

    default:
	badkind("sig_cconjugate", this->kind);
    }
}

/******************* Comparison *************************/

/*--------------------------------------------------------------------------
 Compare vectors A and B.
 Return number of errors.
 Set ai = NULL if vector is not complex, ae=be=0 if vector is not bfp.
--------------------------------------------------------------------------*/
int
compare_cbfp16(n, ar, ai, ae, br, bi, be, rec, errs)
    int n;
    short *ar, *ai, ae;
    short *br, *bi, be;
    int rec;
    int errs;
{
    int i;

    for (i=0; i<n; i++) {
	long ari = (ae>0)? (ar[i] << ae) : (ar[i] >> -ae);
	long bri = (be>0)? (br[i] << be) : (br[i] >> -be);
	if (ari != bri) {
	    if (errs < 10)
		fprintf(stdout,
		"compare %d: point %d real; TOS %ld, NOS %ld.\n",
		rec, i, ari, bri);
	    errs++;
	}
	if (ai != NULL) {
	    long aii = (ae>0)? (ai[i] << ae) : (ai[i] >> -ae);
	    long bii = (be>0)? (bi[i] << be) : (bi[i] >> -be);
	    if (aii != bii) {
		if (errs < 10)
		    fprintf(stdout,
		    "compare %d: point %d imag; TOS %ld, NOS %ld.\n",
		    rec, i, aii, bii);
		errs++;
	    }
	}
    }
    return errs;
}

int
compare_cfloat32(n, ar, ai, br, bi, rec, errs)
    int n;
    float *ar, *ai;
    float *br, *bi;
    int rec;
    int errs;
{
    int i;

    for (i=0; i<n; i++) {
	if (ar[i] != br[i]) {
	    if (errs < 10)
		fprintf(stdout,
		"compare %d: point %d real; TOS %f, NOS %f.\n",
		rec, i, ar[i], br[i]);
	    errs++;
	}
	if (ai != NULL && ai[i] != bi[i]) {
	    if (errs < 10)
		fprintf(stdout,
		"compare %d: point %d imag; TOS %f, NOS %f.\n",
		rec, i, ai[i], bi[i]);
	    errs++;
	}
    }
    return errs;
}

/*--------------------------------------------------------------------------
 Compare vector this with that.
--------------------------------------------------------------------------*/
void
sig_compare(this, that)
    sig_t *this, *that;
{
    int n;
    int offset;
    int nrec;
    int errs = 0;
    int cmplx = this->kind & hdr_KIND_COMPLEX;

    if (this->reclen != that->reclen) {
	fprintf(stderr, "sig_compare(%s,%s): reclens differ: %d != %d\n",
	    this->fname, that->fname, this->reclen, that->reclen);
	exit(1);
    }
    nrec = this->nrec;
    if (that->nrec < nrec) nrec = that->nrec;
    samekind("sig_compare", this, that);

    switch (this->kind) {
    case hdr_INT16:
    case hdr_CINT16:
	for (n=0, offset=0; n<nrec; n++, offset+=this->reclen){
	    errs = compare_cbfp16(this->reclen, 
		that->re+offset, cmplx?(that->im+offset):NULL, 0,
		this->re+offset, cmplx?(this->im+offset):NULL, 0,
		n, errs);
	}
	break;
    case hdr_BFP16:
    case hdr_CBFP16:
	for (n=0, offset=0; n<nrec; n++, offset+=this->reclen){
	    errs = compare_cbfp16(this->reclen, 
		that->re+offset, cmplx?(that->im+offset):NULL, that->exp[n],
		this->re+offset, cmplx?(this->im+offset):NULL, this->exp[n],
		n, errs);
	}
	break;
    case hdr_FLOAT32:
    case hdr_CFLOAT32: {
	float *thisre = (float *)this->re;
	float *thisim = (float *)this->im;
	float *thatre = (float *)that->re;
	float *thatim = (float *)that->im;
	for (n=0, offset=0; n<nrec; n++, offset+=this->reclen){
	    errs = compare_cfloat32(this->reclen, 
		thatre+offset, cmplx?(thatim+offset):NULL, 
		thisre+offset, cmplx?(thisim+offset):NULL, n, errs);
	}
	} break;
    default:
	badkind("sig_compare", this->kind);
    }

    if (errs)
	fprintf(stdout, "sig_compare(%s,%s): %d errors.\n", 
	    this->fname, that->fname, errs);
    if (this->nrec < nrec)
	fprintf(stdout, "sig_compare(%s,%s): EOF on 1st arg.\n",
	    this->fname, that->fname);
    if (that->nrec < nrec)
	fprintf(stdout, "sig_compare(%s,%s): EOF on 2nd arg.\n",
	    this->fname, that->fname);
}

/*--------------------------------------------------------------------------
 Compute square of each element.  For complex signals, compute
 a * (a*)
--------------------------------------------------------------------------*/
void
square_float32(n, ar)
    int n;
    float *ar;
{
    int i;

    for (i=0; i<n; i++) {
	float f;
	f = *ar;
	*ar++ = f * f;
    }
}

void
square_cfloat32(n, ar, ai)
    int n;
    float *ar, *ai;
{
    int i;

    for (i=0; i<n; i++) {
	float rv, iv;
	rv = *ar;
	iv = *ai++;
	*ar++ = rv * rv + iv * iv;
    }
}

/*--------------------------------------------------------------------------
 Replace signal with its square.  Complex signals become real signals.
--------------------------------------------------------------------------*/
void
sig_square(this)
    sig_t *this;
{
    int reclen = this->reclen;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    square_cfloat32(reclen, 
		((float *)this->re) + offset,
		((float *)this->im) + offset);
	break;
    case hdr_FLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    square_float32(reclen, ((float *)this->re) + offset);
	break;
    default:
	badkind("sig_square", this->kind);
    }

    if (this->kind & hdr_KIND_COMPLEX) {
	free(this->im);
	this->im = NULL;
	this->kind &= ~hdr_KIND_COMPLEX;
    }
}

/*--------------------------------------------------------------------------
 Compute sum in the x direction.
--------------------------------------------------------------------------*/
float
sum_x_float32(n, ar)
    int n;
    float *ar;
{
    float sum = 0;
    int i;

    for (i=0; i<n; i++)
	sum += *ar++;
    return sum;
}

/*--------------------------------------------------------------------------
 Replace signal with its sum in the x direction; each record is replaced
 by its sum.
--------------------------------------------------------------------------*/
void
sig_sum_x(this)
    sig_t *this;
{
    int reclen = this->reclen;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_FLOAT32: {
	float *re = (float *)this->re;
	for (n=0, offset=0; n<this->nrec; n++, offset += reclen)
	    re[n] = sum_x_float32(reclen, re + offset);
	} break;
    default:
	badkind("sig_sum_x", this->kind);
    }
    this->reclen = 1;
    this->re = (short *)realloc((char *)this->re,
	this->nrec*this->reclen*sig_ELBYTES(this));
    if (this->kind & hdr_KIND_COMPLEX)
	this->im = (short *)realloc((char *)this->im,
	    this->nrec*this->reclen*sig_ELBYTES(this));
}

/*--------------------------------------------------------------------------
 Replace signal with its mean in the x direction; each record is replaced
 by its mean.
--------------------------------------------------------------------------*/
void
sig_mean_x(this)
    sig_t *this;
{
    int reclen = this->reclen;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_FLOAT32: {
	float *re = (float *)this->re;
	for (n=0, offset=0; n<this->nrec; n++, offset+=reclen)
	    re[n] = sum_x_float32(reclen, re + offset) / reclen;
	} break;
    default:
	badkind("sig_mean_x", this->kind);
    }
    this->reclen = 1;
    this->re = (short *)realloc((char *)this->re,
	this->nrec*this->reclen*sig_ELBYTES(this));
    if (this->kind & hdr_KIND_COMPLEX)
	this->im = (short *)realloc((char *)this->im,
	    this->nrec*this->reclen*sig_ELBYTES(this));
}

/*--------------------------------------------------------------------------
 Compute max in the x direction.
 Set *argmax to the index of the 
--------------------------------------------------------------------------*/
float
max_x_float32(n, ar, argmax)
    int n;
    float *ar;
    int *argmax;
{
    float peak = -MAXFLOAT;
    int i;
    int apeak = -1;

    for (i=0; i<n; i++, ar++) {
	if (peak < *ar) {
	    peak = *ar;
	    apeak = i;
	}
    }
    if (argmax != NULL)
	*argmax = apeak;

    return peak;
}

/*--------------------------------------------------------------------------
 Replace signal with its max in the x direction; each record is replaced
 by its max.
--------------------------------------------------------------------------*/
void
sig_max_x(this)
    sig_t *this;
{
    int reclen = this->reclen;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_FLOAT32: {
	float *re = (float *)this->re;
	for (n=0, offset=0; n<this->nrec; n++, offset+=reclen)
	    re[n] = max_x_float32(reclen, re + offset, NULL);
	} break;
    default:
	badkind("sig_max_x", this->kind);
    }
    this->reclen = 1;
    this->re = (short *)realloc((char *)this->re,
	this->nrec*this->reclen*sig_ELBYTES(this));
    if (this->kind & hdr_KIND_COMPLEX)
	this->im = (short *)realloc((char *)this->im,
	    this->nrec*this->reclen*sig_ELBYTES(this));
}


/*--------------------------------------------------------------------------
 Replace signal with the index of its max in the x direction; each record is 
 replaced by the index of its max.
--------------------------------------------------------------------------*/
void
sig_argmax_x(this)
    sig_t *this;
{
    int reclen = this->reclen;

    int n;
    int offset;

    switch (this->kind) {
    case hdr_FLOAT32: {
	float *re = (float *)this->re;
	for (n=0, offset=0; n<this->nrec; n++, offset+=reclen) {
	    int amax;
	    (void) max_x_float32(reclen, re + offset, &amax);
	    ((short *)re)[n] = amax;
	}
	} break;
    default:
	badkind("sig_argmax_x", this->kind);
    }
    this->reclen = 1;
    this->re = (short *)realloc((char *)this->re,
	this->nrec*this->reclen*sizeof(short));
    if (this->kind & hdr_KIND_COMPLEX) {
	free(this->im);
	this->im = NULL;
    }
    if (this->kind & hdr_KIND_BFP) {
	free(this->exp);
	this->exp=NULL;
    }
    this->kind = hdr_INT16;
}
