;Desugar and syntax-check an expression.
;
; Internal defines are converted to letrecs.
;
; (define (<variable> <formals>) <body>)
;  is converted to
;  (define <variable> (lambda <formals> <body>))
; 
; A <body> of the form
;  <define-binding>* <exp>+
; is desugared to
;  (letrec <binding>* (begin <exp>+))
; (where the letrec and begin don't appear if they are unnecessary)
;
; REVISE?: are nested begins handled properly?

(define call/cc call-with-current-continuation)

(define (sm-error0 message)  ;;may break driver-loop tail-recursion,
             ;; wasting control space stacking up driver-loops which never return 
  (call/cc
   (lambda(c)
     (newline)
     (display message)
     (newline)
     (driver-loop))))

(define (sm-error1 message irritant)
  (call/cc
   (lambda(c)
     (newline)
     (display (string-append message ": "))
     (display irritant)
     (newline)
     (driver-loop))))

(define (sm-error2 message irritant)
  (call/cc
   (lambda(c)
     (newline)
     (display (string-append message ":"))
     (pp irritant)
     (newline)
     (driver-loop))))

(define sm-error
  (lambda mes-irr-list
    (let ((message (car mes-irr-list)))
      (if (null? (cdr mes-irr-list))
          (sm-error0 message)
          (let ((irritant (cadr mes-irr-list)))
            ((if (long? irritant) sm-error2 sm-error1)
             message
             irritant))))))

(define (long? exp)                     ;REVISE: make long? smarter
  (and (pair? exp)
       (pair? (cdr exp))
       (pair? (cddr exp))
       (list? (caddr exp))
       (> (length (caddr exp)) 2)))

(define (top-level-desugar exp)
  (if (define? exp)
      (let* ((exp0 (do-define exp))
             (def-exp (caddr exp0))
             (def-variable (cadr exp0)))
        (make-define def-variable (desugar def-exp)))
      (desugar exp)))

(define (do-define exp)
  (if (not (= (length exp) 3))
      (sm-error "Syntax: define must have 2 subforms" exp)
      (let* ((body (cddr exp))
             (part2 (cadr exp)))
        (cond ((pair? part2)     ; handle definitions of the form
                                 ;    (define (<variable> <formals>) <body>)
                                 ; These are desugared to
                                 ;    (define <variable> (lambda (<formals>) <body>))
               (let ((def-variable (car part2))
                     (def-formals (cdr part2)))
                 (record-step "DSGR" def-variable)
                 (cond ((and (variable? def-variable)
                             (for-all? def-formals variable?))
                        (make-define def-variable
                                     (cons 'lambda (cons def-formals body))))
                       (else
                        (sm-error "Syntax: only variables can be defined or be formals" part2)))))
              ((variable? part2) ; otherwise definition is of the essential form
                                 ;    (define <variable> <expression>)
                                 ; and no desugaring is needed.
                exp)
              (else
               (sm-error "Syntax: only variables can be defined:" part2))))))

(define (desugar exp)
  (cond ((nil-builtin? exp) exp)
        ((known-constant? exp) exp)
        ((variable? exp) exp)
        ((string? exp) exp)
        ((and (quote? exp) (desugar-quote exp))) ;REVISED
        ((lambda? exp) (desugar-lambda exp))
        ((let? exp) (desugar-let exp))
        ((let*? exp) (desugar-let* exp))
        ((named-let? exp) (desugar-named-let exp))
        ((letrec? exp) (desugar-letrec exp))
        ((if? exp) (desugar-if exp))
        ((cond? exp) (desugar-cond exp))
        ((and? exp) (desugar-and exp))
        ((or? exp) (desugar-or exp))
        ((sequence? exp) (desugar-sequence exp))
        ((combination? exp) (desugar-combination exp))
        ((define? exp)
         (sm-error "Syntax: unexpected define"))
        ((bracketed-identifier? exp)
         (sm-error "Syntax: unknown system constant" exp))
        (else (sm-error "Syntax not found in grammar" exp))))

(define (desugar-quote exp)
  (if (= 2 (length exp))
      (let ((exp (desugar-quoted-body (cadr exp))))
        (if (listexp? exp)
            (record-step "QUOT"))
        exp)
      (else (sm-error "QUOTE takes one argument" exp))))

(define (desugar-quoted-body obj)
  (cond                                
        ((quotestable-value? obj) obj)
        ((symbol? obj) (list 'quote obj))
        ((list? obj)
         (cons '<<list>> (map desugar-quoted-body obj)))
        (else (error "Can't desugar quoted object: " obj))))

(define (desugar-combination exp)
  (map desugar exp))

(define (desugar-lambda exp)  ;REVISED: repeated check added
  (if (< (length exp) 3)
      (sm-error "Syntax: lambda: missing formals or body" (cdr exp))
      (let ((frmls (lam-formals exp)))
         (if (not (for-all? frmls variable?))             ;;REVISED
             (sm-error "Syntax: lambda: formals must be variables" frmls)
             (let ((rep (repeated frmls)))
                (if rep
                    (sm-error
                     "Syntax: lambda: repeated formal" rep) ;REVISED
                    (make-lambda frmls (desugar-body (cddr exp)))))))))

(define (get-defines body)              ;REVISE? to ignore nested BEGIN's
  (cond ((null? body) (sm-error "Syntax: body does not end with expression"))
	((define? (car body)) (cons (car body) (get-defines (cdr body))))
	(else '())))

(define (get-exprs body)
  (cond ((null? body) (sm-error "Syntax: body does not end with expression"))
	((define? (car body)) (get-exprs (cdr body)))
	(else body)))

(define (maybe-make-sequence seq)
  (cond ((null? seq) (sm-error "Syntax: empty sequence"))
        ((and (list? seq) (null? (cdr seq))) (car seq))
        (else
         ; add explicit begin around a sequence
         (record-step "DSGR: BEG")
         (make-sequence seq))))

(define (desugar-body body)
  (let ((defines (get-defines body))
        (exprs (get-exprs body)))
    (cond ((= (length defines) 0)
           (desugar (maybe-make-sequence exprs)))
          (else (let* ((defines1 (map do-define defines))
                       (defines2 (map cdr defines1))
                       (def-vars (map car defines2))
                       (string-of-vars
                        (string-append
                         "("
                         (apply string-append
                                (map
                                 (lambda (var) (string-append (symbol->string var) " "))
                                 def-vars))
                         ")")))
                       (record-step "DEF->LTR" (string->symbol string-of-vars))
                                    ; internal defines changed to letrecs
                       (desugar (make-letrec
                                 defines2
                                 (maybe-make-sequence exprs))))))))

(define (desugar-let exp)
  (cond ((not (>= (length exp) 3))
	 (sm-error "Syntax: illformed let expression" exp))
	(else (do-let (bindings-letc exp) (cddr exp) make-let))))

(define (desugar-let* exp)
  (cond ((not (>= (length exp) 3))
	 (sm-error "Syntax: illformed let* expression" exp))
	(else (do-let (bindings-letc exp) (cddr exp) make-let*))))

(define (desugar-named-let exp)
  (cond ((not (>= (length exp) 4))
	 (sm-error "Syntax: illformed named-let expression" exp))
	(else (do-let (bindings-named-letc exp)
		      (cdddr exp)
		      (lambda (bindings body)
			(make-named-let (cadr exp)
					bindings
					body))))))

(define (desugar-letrec exp)
  (cond ((not (>= (length exp) 3))
	 (sm-error "Syntax: illformed letrec expression" exp))
	(else (do-let (bindings-letc exp) (cddr exp) make-letrec))))

(define (do-let bindings body make-let)
  (cond ((not (list? bindings))
         (sm-error "Syntax: illformed bindings" bindings))
        ((not (for-all?
               bindings
               (lambda (x) (and (list? x)
                                (= (length x) 2)
                                (variable? (binding-variable-letc x))))))
         (sm-error "Syntax: illformed binding" bindings))
        (else
         (let ((rep (repeated (binding-variable-list bindings))))
           (if rep
               (sm-error "Syntax: repeated binding" rep)
               (make-let
                (map (lambda (binding) (list (car binding)
                                             (desugar (cadr binding))))
                     bindings)
                (desugar-body body)))))))

(define (repeated lst)
  (cond ((null? lst) #f)                ;REVISED
        ((memq (car lst) (cdr lst))
         (car lst))
        (else (repeated (cdr lst)))))

(define (desugar-cond exp)              ;;REVISED a lot
  (let ((clauses (clauses-cond exp)))
    (if (null? clauses)
        '(cond)
        (if (list? clauses)
            (let* ((clauses2
                    (map
                     (lambda (seq)
                       (if (clause? seq)
                           (cons
                            (if (eq? 'else (car seq))
                                'else
                                (desugar (car seq)))
                            (if (null? (cdr seq))
                                '()
                                (list (maybe-make-sequence
                                       (map desugar (cdr seq))))))
                           (sm-error "Syntax: illformed cond clause" seq)))
                     clauses))
                   (from-else (memq 'else (map car clauses2))))
              (if (and from-else (cdr from-else))
                  (sm-error "Syntax: premature ELSE clause" clauses2)
                  (make-cond clauses2)))
            (sm-error "Syntax: illformed cond clauses" clauses)))))

(define (desugar-sequence exp)
  (if (> (length exp) 1)
      (make-sequence (map desugar (expression-sequence exp))) ;REVISE?: desugar-body
      (sm-error "Syntax: begin: no subforms in sequence" exp)))

(define (desugar-if exp) ; does one-branch-ifs
  (if (or (= (length exp) 4)
          (= (length exp) 3))
      (cons 'if (map desugar (cdr exp)))
      (sm-error "Syntax: if: incorrect number of subforms" exp)))

(define (desugar-and exp)
 (cons 'and (map desugar (cdr exp))))

(define (desugar-or exp)
 (cons 'or (map desugar (cdr exp))))

(define (bracketed-identifier? exp)
  (and (symbol? exp)
           (let* ((str (symbol->string exp))
                  (len (string-length str)))
             (and (>= len 4)
                  (string=? (string-tail str (- len 2)) ">>")
                  (string=? (string-head str 2) "<<")))))

