
/* front end to Max Ribles marching cubes:  Brian Tierney, LBL 12/90 */

/* usage: march [-n][-s NN] < 3d_data.hips > polygon_file
   -n           create vertex normals
   -a           ascii file output
   -s NN        surface threshold value (default = 50)
   -t NN        transparency (range = 0 to 10, 10 = off, default = 10)
*/

/*  This front end to the marching cubes algorithm (see march.c for more info)
 * handles HIPS data format for input, and SCRY data format for output */

/* NOTE: the grid array is used for segmenting the data set. In this
 *  version, the grid is created by just thresholding the input data.
 *  By using a grid in this way, the segmentation can be kept separate
 *  from the tesselation. A number of methods could be used to
 *  create the grid, such as region growing, edge detection, boolean
 *  operations with geometric shapes, etc.
 */

#include "march.h"
#include <hipl_format.h>

 /* set defaults */
int       DO_NORMALS = 0;
int       TVAL = 50;
int       TRANS = 10;		/* transparency: 10 = off */
char      MY_NAME[80];
int       VERBOSE = 1;
int       ASCII_OUTPUT = 0;	/* not ascii is the default */

char     *Progname = "march";	/* used by hips error routines */

/*****************************************************/
main(argc, argv)
    int       argc;
    char     *argv[];
{
    float     threshold;	/* the threshold to look for */
    int       xdim, ydim, zdim;	/* the dimensions of the data */
    float  ***darray = NULL;
    u_char ***grid = NULL;
    int       num_tri, num_voxels;
    float    *edgelist[3], *normlist[3];
    long     *triangles;
    int       edges = 0, sides = 0;

    int       parse_options();	/* parse the command line options */
    float  ***get_data();	/* reads in the data */
    u_char ***alloc_3d_byte_array();

    strcpy(MY_NAME, argv[0]);

    if (parse_options(argc, argv) == -1) {
	fprintf(stderr, "%s: error from parse_options\n", MY_NAME);
	exit(1);
    }
    threshold = (float) TVAL;

    if (VERBOSE)
	fprintf(stderr, "%s: looking for threshold %f \n", MY_NAME, threshold);

    darray = get_data(stdin, &xdim, &ydim, &zdim);

    grid = alloc_3d_byte_array(xdim, ydim, zdim);

    /* use simple thresholding to segment data */
    num_voxels = identify_objects(threshold, darray, grid,
				  (long) (xdim * ydim * zdim));
    fprintf(stderr, " %d voxels selected in the mask \n", num_voxels);

    if (DO_NORMALS)
	num_tri = (int) march(xdim, ydim, zdim, (float) threshold,
			      darray, grid,
			      edgelist, normlist, &triangles,
			      &edges, &sides);	/* return values */
    else
	num_tri = (int) march(xdim, ydim, zdim, (float) threshold,
			      darray, grid,
			      edgelist, NULL, &triangles,
			      &edges, &sides);	/* return values */


    fprintf(stderr, " located %d triangles \n", num_tri);

    if (VERBOSE)
	fprintf(stderr, "%s: writing output\n", MY_NAME);

    if (ASCII_OUTPUT)
	write_ascii_polys(stdout, TRANS, DO_NORMALS, edges, sides,
			  edgelist, normlist, triangles);
    else
	write_polys(stdout, TRANS, DO_NORMALS, edges, sides,
		    edgelist, normlist, triangles);

    exit(0);
}

/***************************************************************/
int
identify_objects(thresh_value, data, grid, size)	/* labels objects with a
							 * '1' */
    float     thresh_value;
    float  ***data;
    u_char ***grid;
    long      size;
{
    int       num_on = 0;

    register int i;
    register u_char *gptr;
    register float *dptr;

    gptr = **grid;
    dptr = **data;

    for (i = 0; i < size; i++) {
	if (dptr[i] < thresh_value)
	    gptr[i] = 0;
	else {
	    gptr[i] = 1;
	    num_on++;
	}
    }

    return (num_on);
}

/**************************** parse_options ****************************/
int
parse_options(argc, argv)
    int       argc;
    char     *argv[];
{

    void      usage();

    /* Interpret options  */
    while (--argc > 0 && (*++argv)[0] == '-') {
	char     *s;
	for (s = argv[0] + 1; *s; s++)
	    switch (*s) {
	    case 's':
		if (argc < 2)
		    usage();
		sscanf(*++argv, "%d", &TVAL);
		argc--;
		break;
	    case 't':
		if (argc < 2)
		    usage();
		sscanf(*++argv, "%d", &TRANS);
		argc--;
		break;
	    case 'n':
		DO_NORMALS = 1;
		break;
	    case 'a':
		ASCII_OUTPUT = 1;
		break;
	    case 'h':
		usage();
		break;
	    default:
		usage();
		break;
	    }
    }				/* while */

    if (DO_NORMALS)
	fprintf(stderr, "Computing iso-values with normals ... \n");

    if (TRANS != 10)
	fprintf(stderr, "Transparency set to %d. \n", TRANS);

    return 0;
}

/**************************** usage ****************************/

void
usage()
{
    fprintf(stderr, "usage: %s [-n][-a][-s NN][-t NN] < 3d_data.hips > polygon_file\n",
	    MY_NAME);
    fprintf(stderr, "   -n           create vertex normals \n");
    fprintf(stderr, "   -a           ascii file output \n");
    fprintf(stderr, "   -s NN        surface threshold value (default = %d) \n", TVAL);
    fprintf(stderr, "   -t NN        transparency (range = 0 to 10, 10 = off, default = %d)\n\n", TRANS);
    exit(-1);
}
