/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*     file:     model.c
*
*     Purpose:  Switch between LINEAR and QUADRATIC models.
*
*/

#include "include.h"

/*******************************************************************
*
*   Function:  change_model()
*
*   Purpose:  Ask user what model he wants.
*/

void change_model()
{
  char ans[100];

    if ( web.modeltype == LINEAR )
      { outstring("Current model type is LINEAR.\n");
        if ( web.symmetry_flag )
          {
            outstring("Quotient spaces support only linear patches.\n");
            return;
          }
        if ( web.symmetric_content )
          {
            outstring("SYMMETRIC CONTENT supported only in linear model.\n");
            return;
          }
        if ( web.surfen_count )
          {
            outstring("Surface energy integrands supported only in linear model.\n");
            return;
          }
        if ( web.sdim > 3 )
          {
            outstring("Space dimension > 3 supported only in linear model.\n");
            return;
          }
        prompt("Pick new model type, 1 LINEAR  2 QUADRATIC: ",ans);
        if ( logfd ) fprintf(logfd,"%s\n",ans);
        switch ( ans[0] )
          {
             case '2': linear_to_quad(); break;
             default : break;
          }
      }
    else
      { outstring("Current model type is QUADRATIC.\n");
        prompt("Pick new model type, 1 LINEAR  2 QUADRATIC: ",ans);
        if ( logfd ) fprintf(logfd,"%s\n",ans);
        switch ( ans[0] )
          {
             case '1': quad_to_linear(); break;
             default : break;
          }
      }

    gauss_setup();   /* set up gaussian integration arrays */

    /* recalculate stuff */
    calc_content();
    calc_pressure();
    calc_energy();
}

/***************************************************************
*
*  Function: linear_to_quad()
*
*  Purpose:  Changes linear patch model to quadratic patch
*            model by inserting midpoints in all edges
*            and resetting function pointers.
*/

void linear_to_quad()
{
  edge_id e_id;

  FOR_ALL_EDGES(e_id)
    {
      REAL *t,x[MAXCOORD];
      int i;
      vertex_id headv,tailv;
      vertex_id new_v;
      REAL side[MAXCOORD];
      
      headv = get_edge_headv(e_id);
      tailv = get_edge_tailv(e_id);
      t = get_coord(tailv);
      get_edge_side(e_id,side);
      for ( i = 0 ; i < web.sdim ; i++ )
        x[i] = t[i] + side[i]/2;
      new_v = new_vertex(x);
      set_edge_midv(e_id,new_v);
      set_vertex_fe(new_v,get_edge_fe(e_id));

      if ( get_eattr(e_id) & FIXED ) 
          set_attr(new_v,FIXED);

      /* for boundary edges, cannot just interpolate parameters
         due to wrap-around of angular parameters. So tangent extrapolate
         from one endpoint.
       */
      if ( get_eattr(e_id) & BOUNDARY )
        { 
          struct boundary *bdry;
          REAL *paramb,*parammid,*mu,*mv;
          vertex_id base_v;
    
          bdry = get_edge_boundary(e_id);
          if ( get_boundary(headv) == bdry )
             base_v = headv;
          else if ( get_boundary(tailv) == bdry )
             base_v = tailv;
          else
            { sprintf(errmsg,"Vertices %d and %d of edge %d are on different boundaries.\n",
                ordinal(headv)+1,ordinal(tailv)+1,ordinal(e_id)+1);
              error(errmsg,UNRECOVERABLE);
            }

          set_attr(new_v,BOUNDARY);
          set_boundary(new_v,bdry);
    
          /* projecting on tangent */
          mv = get_coord(new_v);
          mu = get_coord(base_v);
          paramb = get_param(base_v);
          parammid = get_param(new_v);
          b_extrapolate(bdry,mu,mv,mv,paramb,parammid);

        }
      else if ( get_eattr(e_id) & CONSTRAINT )
        { 
          ATTR attr = get_eattr(e_id) & (BDRY_ENERGY|BDRY_CONTENT|CONSTRAINT );
          MAP conmap = get_e_constraint_map(e_id);
    
          set_attr(new_v,attr);
          set_v_conmap(new_v,conmap);
          project_v_constr(new_v);
        }
    }

  web.modeltype = QUADRATIC;

  /* redirect functions */
  calc_facet_volume = facet_volume_q;
  film_grad = film_grad_q;
  calc_edge_area = edge_area_q;
  string_grad = string_grad_q;
  calc_facet_energy = facet_energy_q;
  calc_facet_forces = facet_force_q;
  if ( web.metric_flag )
   {
     calc_edge_energy = edge_energy_q_metric;
     calc_edge_forces  = edge_force_q_metric;
   }
  else
   {
     calc_edge_energy = edge_energy_q;
     calc_edge_forces  = edge_force_q;
   }

}
  

/***************************************************************
*
*  Function: quad_to_linear()
*
*  Purpose:  Changes quadratic patch model to linear patch
*            model by deleting midpoints from all edges
*            and resetting function pointers.
*/

void quad_to_linear()
{
  edge_id e_id;

  FOR_ALL_EDGES(e_id)
    {
      free_element(get_edge_midv(e_id));
    }

  web.modeltype = LINEAR;
  calc_facet_energy = (web.sdim > 3) ? facet_energy_l_hi_d : facet_energy_l;
  calc_facet_forces = (web.sdim > 3) ? facet_force_l_hi_d :  facet_force_l;
  calc_facet_volume = facet_volume_l;
  film_grad = film_grad_l;
  if ( web.metric_flag )
   {
     calc_edge_energy = edge_energy_l_metric;
     calc_edge_forces  = edge_force_l_metric;
   }
  else
   {
     calc_edge_energy = edge_energy_l;
     calc_edge_forces  = edge_force_l;
   }
  calc_edge_area = edge_area_l;
  string_grad = string_grad_l;

}
 
/**********************************************************************
*
*  function: gauss_setup()
*
*  purpose:  Initialize arrays used in gaussian integration.
*
*  Does arbitrary order in 1D.
*  Currently does only 7-pt integration for 2D
*  and n+1 pt for over 2D.
*
***********************************************************************/

void gauss_setup()
{  int i,j,k,m;

   /* allocate tangent vector matrix */
   tang = dmatrix(0,web.dimension-1,0,web.sdim-1);

   /* set number of control points */
   ctrl_num = web.dimension + 1;
   if ( web.modeltype == QUADRATIC )
     ctrl_num += web.dimension*(web.dimension+ 1)/2;
   if ( ctrl_num > MAXCTRL )
     error("Number of control vertices per facet exceeds MAXCTRTL.",
       RECOVERABLE);

   /* set number of integration points and weights */
   if ( web.dimension == 1 )
     gauss_num = abs(web.gauss_order);
   else if ( web.dimension == 2 )
     { if ( web.gauss_order > 3 ) 
	 { gauss_num = 7;
	   gauss2Dpt = gauss2Dpt7;
	   gauss2Dwt = gauss2Dwt7;
	 }
       else if ( web.gauss_order > 1 )
	 { gauss_num = 3;
	   gauss2Dpt = gauss2Dpt3;
	   gauss2Dwt = gauss2Dwt3;
	 }
       else
	 { gauss_num = 1;
	   gauss2Dpt = gauss2Dpt1;
	   gauss2Dwt = gauss2Dwt1;
	 }
     }
   else gauss_num = web.dimension + 1;

   /* always have 1D integration for edges on constraints */
   web.gauss1D_order = abs(web.gauss_order);
   gauss1Dpt = (double *)mycalloc(abs(web.gauss_order),sizeof(double));
   gauss1Dwt = (double *)mycalloc(abs(web.gauss_order),sizeof(double));
   grule(web.gauss_order,gauss1Dpt,gauss1Dwt);

   if ( web.dimension == 1 )
      gausswt = gauss1Dwt;
   else if ( web.dimension == 2 )
     gausswt = gauss2Dwt;
   else 
     { gausswt = (double *)mycalloc(gauss_num,sizeof(double));
       for ( i = 0 ; i < gauss_num ; i++ )
         gausswt[i] = 1.0/gauss_num;  /* trivial points */
      }

   /* set up interpolation polynomial values */
   if ( gpoly ) free_matrix(gpoly);
   gpoly = dmatrix(0,gauss_num-1,0,ctrl_num-1);
   if ( gpolypartial ) free_matrix3(gpolypartial);
   gpolypartial = dmatrix3(gauss_num,web.dimension,ctrl_num);
   if ( web.dimension == 1 )
    for ( j = 0 ; j < gauss_num ; j++ )
     for ( i = 0 ; i < ctrl_num ;  i++ )
       {
	 double p=1.0,sum=0.0;
	 int scale = ctrl_num - 1;
	 for ( m = 0 ; m < ctrl_num ; m++ )
	   { if ( m == i ) continue;
	     p *= (gauss1Dpt[j]*scale - m)/(i - m);
	     if ( fabs(p) < 1e-100 ) break;
	     sum += scale/(gauss1Dpt[j]*scale - m);
	   }
         gpoly[j][i] = p;
         gpolypartial[j][0][i] = sum*p;
       }
   else if ( web.dimension == 2 )
     { if ( web.modeltype == LINEAR )
	 { for ( k = 0 ; k < gauss_num ; k++ )
	    {
	      gpoly[k][0] = gauss2Dpt[k][0];
	      gpoly[k][1] = gauss2Dpt[k][1];
	      gpoly[k][2] = gauss2Dpt[k][2];
	      gpolypartial[k][0][0] = -1.0;
	      gpolypartial[k][1][0] = -1.0;
	      gpolypartial[k][0][1] =  1.0;
	      gpolypartial[k][1][2] =  1.0;   /* others 0 */
	    }
	  }
	else /* QUADRATIC */
	  { for ( k = 0 ; k < gauss_num ; k++ )
	     for ( j = 0 ; j < ctrl_num ; j++ )
	      {
                gpoly[k][j] = intpoly6(j,2*gauss2Dpt[k][1],2*gauss2Dpt[k][2]);
                for ( i = 0 ; i < 2 ; i++ )
                   gpolypartial[k][i][j] = 2*intpoly6part(j,i,
                	2*gauss2Dpt[k][1], 2*gauss2Dpt[k][2]);
			  /* since intpoly was on side 2 triangle */
			  /* and gauss2Dpt barycentric  */
	      }
	  }
      }
    else /* higher dimension */
      { /* crude: gauss pts same as control points */
	for ( k = 0 ; k < gauss_num ; k++ )
	  { gpoly[k][k] = 1.0;  /* rest 0 */
	    for ( j = 0 ; j < web.dimension ; j++ )
	     {
	       gpolypartial[k][j][0] = -1.0;
	       gpolypartial[k][j][j+1] = 1.0;
	     }	
	  }
      }
}

   
