

/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*     file:     quotient.c
*
*     Purpose:  Illustration of how to do quotient spaces.
*               This example does a flat torus in R^3.
*               The user needs to define the integer representation
*               used for group elements and three functions to
*               do group operations.  WRAPTYPE is the type of
*               group element representation, currently long int.
*               You must write your own versions of group_wrap(),
*               group_compose(), and group_inverse().
*
*               You don't have the privilege
*               of defining your own datafile syntax, but you can access
*               values of parameters defined in the datafile as
*               web.params[n].value.  Order is order of appearance.
*/

#include "include.h"



/* change torus stuff in following lines to your own names */

char *symmetry_name = "genus2";  /* name appearing in datafile. lowercase! */

/*
       Group elements are encoded in four-bit fields, with 3 bits
       representing a signed integer and one overflow bit.
       There is one field for each dimension, with low dimension
       in the low bits.  You are free to encode
       any way you want, but use wrap = 0 for the identity.
*/

/* specific case of torus symmetry representation */
#define GENUS2BITS 4
#define GENUS2MASK 7
#define ALLWRAPMASK 0x77

#define DIMENSION 3
#define PDIM 4

#define CHOP(g) ((g)&GENUS2MASK)
#define END(g) CHOP((g)>>GENUS2BITS)
#define DFF(g,h,d) (CHOP((h)-(g))==(d))
#define ENC(g1,gk) (CHOP(g1) | CHOP(gk)<<GENUS2BITS)
#define WRAP(g1,gk) (DFF(g1,gk,0)? 0:(DFF(g1,gk,7)? ENC(g1+3,g1+4):ENC(g1,gk)))


void conf2proj(z,w)
REAL z[DIMENSION],w[PDIM];
{
    int j;
    REAL r2 = 0.;
    REAL factor;

    for (j=0;j<DIMENSION;j++)
	r2 += z[j]*z[j];
    factor = 2/(1+r2);
    for (j=0;j<DIMENSION;j++)
	w[j] = z[j]*factor;
    w[DIMENSION] = 1.;
}

void proj2conf(z,w)
REAL z[PDIM],w[DIMENSION];
{
    int j;
    REAL r2 = 0.;
    REAL factor;

    for (j=0;j<DIMENSION;j++)
	z[j] /= z[DIMENSION];
    for (j=0;j<DIMENSION;j++)
	r2 += z[j]*z[j];
    factor = 1/r2 - sqrt((1/r2 - 1)/r2);
    if (r2 == 0.) factor = 1.;
    for (j=0;j<DIMENSION;j++)
	w[j] = z[j]*factor;
}

#define COSH 2.414213562373 /* 1+sqrt(2) */
#define SINH 2.19736822693562

#define SIN 0.70710678118655 /* sqrt(2)/2 */
#define COS -0.70710678118655 /* sqrt(2)/2 */
REAL TrMat[PDIM][PDIM] =
    {{COSH,0.,0.,SINH},{0.,1.,0.,0.},{0.,0.,1.,0.},{SINH,0.,0.,COSH}};
REAL RotMat[PDIM][PDIM] = 
    {{COS,SIN,0.,0.},{-SIN,COS,0.,0.},{0.,0.,1.,0.},{0.,0.,0.,1.}};

void copy(z,w)
REAL z[PDIM],w[PDIM];
{
    int j;

    for (j=0; j<PDIM; j++)
	w[j]=z[j];
}

void matmult(m,z,w)
REAL m[PDIM][PDIM],z[PDIM],w[PDIM];
{
    int i,j;
    REAL t[PDIM];

    for (j=0; j<PDIM; j++)
	for (i=0,t[j]=0.; i<PDIM; i++)
	    t[j] += m[j][i] * z[i];
    copy(t,w);
}

void applygen(g,z,w)
int g;
REAL z[PDIM],w[PDIM];
{
    REAL t[PDIM];
    int i;

    g = CHOP(g);
    copy(z,t);
    for (i=0;i<g;i++)
	matmult(RotMat,t,t);
    matmult(TrMat,t,t);
    matmult(TrMat,t,t);
    for (i=g;i<8;i++)
	matmult(RotMat,t,t);
    copy(t,w);
}

/*******************************************************************
*
*  function: group_wrap
*
*  purpose:  Provide adjusted coordinates for vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*/

void group_wrap(x,y,wrap)
REAL *x;   /* original coordinates */
REAL *y;   /* wrapped coordinates  */
WRAPTYPE wrap;  /* encoded symmetry group element */
{
    REAL z[2][PDIM];
    int j=0;
    int g1,gk, g;

    if (wrap==0)
    {
	for (j=0; j<DIMENSION; j++)
	    y[j]=x[j];
	return;
    }

    g1 = CHOP(wrap);
    gk = CHOP(wrap>>GENUS2BITS);
    if (gk<g1) gk += 8;

    conf2proj(x,z[j]);
    for (g=g1; g<gk; g++,j=1-j)
    {
/* fprintf(stderr,"%f %f %f %f\n",z[j][0],z[j][1],z[j][2],z[j][3]); */
	applygen(g,z[j],z[1-j]);
    }
    proj2conf(z[j],y);
/* fprintf(stderr,"%f %f %f %f\n",z[j][0],z[j][1],z[j][2],z[j][3]); */
/* fprintf(stderr,"%f %f %f --%d,%d-> %f %f %f\n",
	x[0],x[1],x[2], g1,gk, y[0],y[1],y[2]); */
}


/********************************************************************
*
*  function: group_compose()
*
*  purpose:  do composition of two group elements
group_wrap(group_wrap(pt,w),v) = group_wrap(pt,group_compose(v,w))
If the current wrap is v, and we get to an edge with wrap w,
and its head is at pt, then either of these should give the unwrapped
coordinates of its head (relative to group_wrap(tail,v) being its tail).
*
*/

/* generators 1=A,2=b,3=C,4=d,5=a,6=B,7=c,0=8=D */
/* inverses differ by 4 */

#define TRY(g1,gk,h1,hk) if (DFF(gk,h1,0)) return WRAP(g1,hk)

WRAPTYPE group_compose(hw,gw)
WRAPTYPE gw,hw;  /* the elements to compose */
{
    int g1,gk, h1,hk,c;

    if (gw==0) return hw;
    if (hw==0) return gw;
    g1 = CHOP(gw); gk = END(gw);
    h1 = CHOP(hw); hk = END(hw);

    TRY(g1,gk,h1,hk);
    if (DFF(h1,hk,1)) TRY(g1,gk,h1-3,h1-4);
    if (DFF(g1,gk,1)) TRY(g1-3,g1-4,h1,hk);
    if (DFF(h1,hk,1) && DFF(g1,gk,1)) TRY(g1-3,g1-4,h1-3,h1-4);

    fprintf(stderr,"Trying to compose %d-%d and %d-%d\n",g1,gk,h1,hk);
    error("Wrap outside known range\n",RECOVERABLE);
}


/********************************************************************
*
*  function: group_inverse()
*
*  purpose:  return inverse of group element.
*
*/

WRAPTYPE group_inverse(wrap)
WRAPTYPE wrap;  /* the element to invert */
{
   return WRAP(END(wrap),CHOP(wrap));
}
