// filecommand.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

// FileCommand is the base class for classes which either create, read, or
// write to files after querying the user for appropriate information.
 
#ifndef FILECOMMAND_H
#ifdef __GNUG__
#pragma interface
#endif
#define FILECOMMAND_H

#include "localdefs.h"
#include "modifier.h"
#include <String.h>

class FileCommand : public Modifier {
	typedef Modifier Super;
public:
	FileCommand() : theController(nil) {}
	redefined boolean configure(Controller *);
	redefined int apply();
protected:
	virtual int doApply() = 0;
	void setFileName(const char* name) { theFileName = name; }
	const char* fileName() { return theFileName; }
	Controller* controller() { return theController; }
private:
	String theFileName;
	Controller* theController;
};

class Data;

class FileCreator : public FileCommand {
	typedef FileCommand Super;
public:
	FileCreator() : newController(nil), theData(nil) {}
	Controller* getNewController() { return newController; }
protected:
	redefined int doApply();
	virtual Data* newData();
	boolean setData(Data *);
private:
	Controller* newController;
	Data* theData;
};

class SoundCreator : public FileCreator {
	typedef FileCreator Super;
public:
	SoundCreator() {}
	redefined const char* message() { return "Creating new sound..."; }
protected:
	redefined void configureRequest(Request&);
	redefined boolean setValues(Request&);
private:
	redefined const QueryInfo* requestInfo();
};

class LPCFileCreator : public FileCreator {
	typedef FileCreator Super;
public:
	LPCFileCreator() {}
	redefined const char* message() { return "Creating new LPC file..."; }
protected:
	redefined void configureRequest(Request&);
	redefined boolean setValues(Request&);
private:
	redefined const QueryInfo* requestInfo();
};

class PvocFileCreator : public FileCreator {
	typedef FileCreator Super;
public:
	PvocFileCreator() {}
	redefined const char* message() { return "Creating new Pvoc file..."; }
protected:
	redefined void configureRequest(Request&);
	redefined boolean setValues(Request&);
private:
	redefined const QueryInfo* requestInfo();
};

class EnvelopeFileCreator : public FileCreator {
	typedef FileCreator Super;
public:
	EnvelopeFileCreator() {}
	redefined const char* message() { return "Creating new envelope..."; }
protected:
	redefined void configureRequest(Request&);
	redefined boolean setValues(Request&);
private:
	redefined const QueryInfo* requestInfo();
};

class FileAccessCommand : public FileCommand {
	typedef FileCommand Super;
public:
	FileAccessCommand(const char* dir) : defaultDir(dir) {}
protected:
	redefined Request* createRequest();
	virtual const char* requestTitle() = 0;
	virtual const char* defaultDirectory() { return defaultDir; }
	virtual const char* defaultSuffixes() { return nil; }
private:
	redefined const QueryInfo* requestInfo() { return nil; } // not used
private:
	String defaultDir;
};

class FileSaver : public FileAccessCommand {
	typedef FileAccessCommand Super;
public:
	FileSaver(const char* dir, Data* data)
		: FileAccessCommand(dir), dataToSave(data),
		  isNewFile(true), forceHeader(false) {}
	FileSaver(const char* name, const char* dir, Data* data);
	redefined const char* message();
protected:
	redefined const char* requestTitle();
	redefined void configureRequest(Request &);
	redefined boolean setValues(Request&);
	redefined boolean confirmValues(Controller *);
	redefined int doApply();
protected:
	enum Format { Raw = 0x1, WithHeader = 0x2 };
	Data* dataToSave;
private:
	boolean isNewFile;
	boolean forceHeader;
};

class SoundFileSaver : public FileSaver {
	typedef FileSaver Super;
public:
	SoundFileSaver(const char* dir, Data* data) : FileSaver(dir, data) {}
	SoundFileSaver(const char* name, const char* dir, Data* data)
		: FileSaver(name, dir, data) {}
protected:
	redefined void configureRequest(Request &);
	redefined boolean setValues(Request &);
};

class LPCFileSaver : public FileSaver {
	typedef FileSaver Super;
public:
	LPCFileSaver(const char* dir, Data* data) : FileSaver(dir, data) {}
	LPCFileSaver(const char* name, const char* dir, Data* data)
		: FileSaver(name, dir, data) {}
protected:
	redefined boolean setValues(Request &);
};

class DataDumper : public FileSaver {
	typedef FileSaver Super;
public:
	DataDumper(const char* dir, Data* data) : FileSaver(dir, data) {}
	redefined const char* message();
protected:
	redefined const char* requestTitle();
	redefined void configureRequest(Request &) {}
	redefined int doApply();
};

class FileOpener : public FileAccessCommand {
public:
	FileOpener(const char* dir, const char* title);
	redefined const char* message();
	Controller* getNewController() { return newController; }
protected:
	redefined const char* requestTitle() { return theTitle; }
	redefined void configureRequest(Request&);
	redefined boolean setValues(Request&);
	redefined int doApply();
private:
	String theTitle;
	double skipTime;
	double duration;
	Controller* newController;
};

#endif
