// filecommand.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "data.h"
#include "editor.h"
#include "datafile.h"
#include "envelope.h"
#include "filecommand.h"
#include "filename.h"
#include "lpcdata.h"
#include "pvocdata.h"
#include "query.h"
#include "request.h"
#include "sndconfig.h"
#include "sound.h"
#include "soundheader.h"

boolean
FileCommand::configure(Controller *c) {
	theController = c;		// cache a ref to this for use
	return Super::configure(c);
}

int
FileCommand::apply() {
    int status = false;
	if(ok()) {
		Application::inform(message());
		status = doApply();
	}
	return status;
}

//********

boolean
FileCreator::setData(Data *d) {
	theData = d;
	return (theData->length() != 0);
}

Data *
FileCreator::newData() {
	return theData;
}

int
FileCreator::doApply() {
	newController = new Controller(newData(), fileName());
	newController->display(controller()->world());
	return true;
}

//********

const QueryInfo *
SoundCreator::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Create New Soundfile: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Duration (seconds): ", "1.00", CharCheck::posNumsOnly },
		{ nil }
	};
	static QueryChoiceInfo choices[] = {
		{ "Channels:", "|1|2|4|", 0x1, true },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values, choices },
		{ nil }
	};
	return info;
}

void
SoundCreator::configureRequest(Request& request) {
	request.appendValue("Sample rate: ", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendValue("File name: ", 
		FileName::untitledName(Sound::defaultFileSuffix()),
		CharCheck::anyChar);
	request.appendChoice("Sample format:",
		"|8-bit linear|8-bit mu law|16-bit linear|floating point|",
		linearEnumToPowerOfTwo(Sound::defaultDataType()),
		true
	);
}

boolean
SoundCreator::setValues(Request& request) {
	const int nVals = 3;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	double dur = v[0];
	int srate = v[1];
	setFileName(v[2]);
	const int nChoices = 2;
	QueryChoice ch[nChoices];
	request.retrieveChoices(ch, nChoices);
	int state = ch[0];
	int chans = (state == 0x1) ? 1 : (state == 0x2) ? 2 : 4;
	DataType format = DataType(powerOfTwoToLinearEnum(ch[1]));
	dur = max(0.001, dur);	// duration never zero
	return setData(new Sound(dur, srate, chans, format));
}

//********

const QueryInfo *
LPCFileCreator::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Create New LPC Datafile: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Length (in frames): ", "32", CharCheck::posIntsOnly },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values },
		{ nil }
	};
	return info;
}

void
LPCFileCreator::configureRequest(Request& request) {
	request.appendValue("Number of filter poles: ",
		LPCData::defaultNumberOfPoles(), Range(12, 64));
	request.appendValue("Frame rate: (frames per second) ",
		LPCData::defaultFrameRate(), CharCheck::posNumsOnly);
	request.appendValue("Source sample rate: ", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendValue("File name: ", 
		FileName::untitledName(".lpc"), CharCheck::anyChar);
}

boolean
LPCFileCreator::setValues(Request& request) {
	const int nVals = 5;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int len = v[0];
	int poles = v[1];
	double frate = v[2];
	int srate = v[3];
	setFileName(v[4]);
	len = max(4, len);
	return setData(new LPCData(len, poles, srate, frate));
}

//********

const QueryInfo *
PvocFileCreator::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Create New PVoc Datafile: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Length (in frames): ", "32", CharCheck::posIntsOnly },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values },
		{ nil }
	};
	return info;
}

void
PvocFileCreator::configureRequest(Request& request) {
	request.appendValue("Frame rate (frames per second):",
		PvocData::defaultFrameRate(), CharCheck::posNumsOnly);
	request.appendValue("Source sample rate:", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendValue("File name:", 
		FileName::untitledName(".pv"), CharCheck::anyChar);
	request.appendChoice("Number of frequency bands (N/2):",
		"|64|128|256|512|1024|2048|4096|8192|16384|",
		PvocData::defaultNumberOfBands() >> 6,
		true);
}

boolean
PvocFileCreator::setValues(Request& request) {
	const int nVals = 4;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int len = v[0];
	double frate = v[1];
	int srate = v[2];
	setFileName(v[3]);
	QueryChoice choice;
	request.retrieveChoices(choice);
	int npoints = int(choice) << 6;	// maps {1, 2, 4, ..} into {64, 128, ...}
	len = max(4, len);
	return setData(new PvocData(len, (npoints + 1) * 2, srate, frate));
}

//********

const QueryInfo *
EnvelopeFileCreator::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Create New Envelope: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Length (in frames): ", "512", CharCheck::posIntsOnly },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values },
		{ nil }
	};
	return info;
}

void
EnvelopeFileCreator::configureRequest(Request& request) {
	request.appendValue("File name: ", 
		FileName::untitledName(".evp"), CharCheck::anyChar);
}

boolean
EnvelopeFileCreator::setValues(Request& request) {
	const int nVals = 2;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int newlen = v[0];
	setFileName(v[1]);
	newlen = max(4, newlen);
	return setData(new Envelope(newlen));
}

//********

Request *
FileAccessCommand::createRequest() {
	return new FileRequest(
		requestTitle(), defaultDirectory(), defaultSuffixes(), createDelegate()
	);
}

//********

extern const char *scat(const char *, const char *);
extern const char *scat(const char *, const char *, const char *);

FileSaver::FileSaver(const char* name, const char* defaultDir, Data* data)
		: FileAccessCommand(defaultDir), dataToSave(data) {
	// dont allow saving to "untitled_*" or "tmp_*" temp names
	if(!FileName::isTempName(name)) {
		// prepend default directory to name if none
		const char* filename = (FileName::isFullPathName(name)) ?
			name : scat(defaultDir, "/", name);
		isNewFile = (DiskFile::exists(filename) == false);
		setFileName(filename);
		initialize();
	}
	else isNewFile = true;
}

const char *
FileSaver::requestTitle() {
	return "Save to File:";
}

void
FileSaver::configureRequest(Request& request) {
	request.appendChoice("File format:", "|Raw (no header)|Use header|",
		WithHeader, true);
}

boolean
FileSaver::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	setFileName(v);
	isNewFile = (DiskFile::exists(fileName()) == false);
	return true;
}

boolean
FileSaver::confirmValues(Controller *) {
	boolean confirmed = true;
	if(!isNewFile) {
		Response r = Application::choice(
			scat("The file ", fileName(), " already exists."),
			"Do you wish to overwrite it?", nil, Cancel,
			"force new header", "keep old header", "cancel");
		switch (r) {
		case Yes:
			forceHeader = true; break;
		case No:
			forceHeader = false; break;
		case Cancel:
		default:
			forceHeader = false;
			confirmed = false;
			break;
		};
	}
	return confirmed;
}

const char *
FileSaver::message() {
	return isNewFile ?
		"Writing new file to disk..." : "Overwriting file on disk...";
}

int
FileSaver::doApply() {
	DataFile file;
	// File truncated if new header type is to be forced
	file.open(fileName(), (isNewFile || forceHeader) ? "w" : "r+");
	int status = (file.writable() && dataToSave->write(&file));
	if(status)
		controller()->setFileName(fileName());	// rename file to new name
	else if(isNewFile)
		file.remove();		// if new file was unwritable, delete it
	return status;
}

//********

void
SoundFileSaver::configureRequest(Request& request) {
	request.appendChoice("Sound file header type:",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIF-C|WAV|",
		SoundHeader::defaultHeaderType(),
		true
	);
}

boolean
SoundFileSaver::setValues(Request& request) {
	boolean status = false;
	if(Super::setValues(request)) {
		QueryChoice headerType;
		request.retrieveChoices(headerType);
		SoundHeader::setDefaultHeaderType(SoundHeader::Type(int(headerType)));
		status = true;
	}
	return status;
}

//********

boolean
LPCFileSaver::setValues(Request& request) {
	boolean status = false;
	if(Super::setValues(request)) {
		QueryChoice headerType;
		request.retrieveChoices(headerType);
		LPCHeader::setDefaultHeaderType(
			(headerType == WithHeader) ? LPCHeader::With : LPCHeader::None
		);
		status = true;
	}
	return status;
}

//********

const char *
DataDumper::requestTitle() {
	return "Dump selection's frame/sample data as ASCII file:";
}

const char *
DataDumper::message() {
	return "Dumping data to file...";
}

int
DataDumper::doApply() {
	int status = true;
	DataFile file;
	file.open(fileName(), "w+");
	if ((status = file.writable()) == true)
		dataToSave->print(file);
	return status;
}

//********

FileOpener::FileOpener(const char* dir, const char* title)
	: FileAccessCommand(dir), theTitle(title),
	  skipTime(0), duration(0), newController(nil) {}

const char *
FileOpener::message() {
	return nil;		// until other routines redone
}

void
FileOpener::configureRequest(Request& request) {
	request.appendValue("Time to skip (secs.): ", 0.0, CharCheck::posNumsOnly);
	request.appendValue("Duration to read: (secs.):", "",
		CharCheck::posNumsOnly);
}

boolean
FileOpener::setValues(Request& request) {
	const int nVals = 3;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	setFileName(v[0]);
	skipTime = v[1];
	duration = v[2];
	return true;
}

int
FileOpener::doApply() {
	DataFile* file = new DataFile(fileName(), "r", skipTime, duration);
	file->ref();
	if(file->is_open()) {
		if(file->readable())
			newController = Controller::create(file);
		else
			Application::error(scat("Unable to read file ", file->name()));
	}
	if(newController)
		newController->display(controller()->world());
	Resource::unref(file);
	return (newController != nil);
}
